<?php
namespace boru\cli2;

use boru\cli2\Models\RouteNode;
use boru\cli2\Models\CommandRouter;
use boru\cli2\CLIContext;

use boru\cli2\Models\Args;
use boru\cli2\Models\Params;
use boru\cli2\Output\OutputInterface;
use boru\cli2\Output\StdOutput;

use boru\cli2\CommandInterface;
use boru\cli2\CommandGroupInterface;
use boru\cli2\ParentPathInterface;
use boru\cli2\CommandDescriptionInterface;
use boru\cli2\CommandParamsInterface;
use boru\cli2\GroupParamsInterface;
use boru\cli2\RootParamsInterface;

use boru\cli2\Traits\ClassCommandTrait;

/**
 * Router-based CLI v2.
 */
class CLI
{
    /** @var string */
    protected $name = 'cli2';

    /** @var string */
    protected $description = '';

    /** @var Params|null */
    protected $rootParams;

    /** @var RouteNode */
    protected $rootNode;

    /** @var CommandRouter */
    protected $router;

    /** @var Args */
    protected $args;

    /** @var OutputInterface */
    protected $output;


    use ClassCommandTrait;
    public function __construct($syntaxOrArray = '', $params = null)
    {
        if (is_array($syntaxOrArray)) {
            // optional: support array-based init similar to CLI1
            if (isset($syntaxOrArray['name'])) {
                $this->name = $syntaxOrArray['name'];
            }
            if (isset($syntaxOrArray['description'])) {
                $this->description = $syntaxOrArray['description'];
            }
        } elseif (is_string($syntaxOrArray) && $syntaxOrArray !== '') {
            $this->parseSyntaxString($syntaxOrArray);
        }

        $this->rootNode = new RouteNode(array($this->name), $this->description, null, null, true);
        $this->router   = new CommandRouter($this->rootNode);

        // default output
        $this->output = new StdOutput();
        
        if ($params !== null && $params !== false) {
            $this->params($params);
        }
    }

    public static function create($syntaxOrArray = '', $params = null)
    {
        return new static($syntaxOrArray, $params);
    }

    protected function parseSyntaxString($syntaxString)
    {
        $parts = explode('|', $syntaxString);
        $this->name        = isset($parts[0]) ? $parts[0] : $this->name;
        $this->description = isset($parts[1]) ? $parts[1] : '';
    }

    /**
     * Get or set the output implementation.
     *
     * @param OutputInterface|null $output
     * @return OutputInterface|$this
     */
    public function output(OutputInterface $output = null)
    {
        if ($output === null) {
            return $this->output;
        }
        $this->output = $output;
        return $this;
    }

    public function name($name = null)
    {
        if ($name === null) {
            return $this->name;
        }
        $this->name = $name;
        return $this;
    }

    public function description($description = null)
    {
        if ($description === null) {
            return $this->description;
        }
        $this->description = $description;
        return $this;
    }

    /**
     * Get the underlying CommandRouter.
     *
     * @return CommandRouter
     */
    public function getRouter()
    {
        return $this->router;
    }

    /**
     * Get the root RouteNode for this CLI2 instance.
     *
     * @return RouteNode
     */
    public function getRootNode()
    {
        return $this->rootNode;
    }

    /**
     * Root/global params for CLI2.
     *
     * @param Params|array|null $params
     * @return Params
     */
    public function params($params = null)
    {
        if ($params === null) {
            if ($this->rootParams === null) {
                // No auto-help for now; callers can add their own flag.
                $this->rootParams = new Params();
                // Global params should not complain about route tokens.
                $this->rootParams->ignoreUnexpected(true);
            }
            return $this->rootParams;
        }

        if ($params === false) {
            $this->rootParams = new Params();
            return $this->rootParams;
        }

        if ($params instanceof Params) {
            $this->rootParams = $params;
        } else {
            $this->rootParams = new Params($params);
        }

        // Ensure global params are tolerant of unexpected tokens
        $this->rootParams->ignoreUnexpected(true);

        return $this->rootParams;
    }

    /**
     * Define a route (leaf command).
     *
     * @param string|string[] $pathOrSyntax e.g. 'deploy' or 'user add' or 'cmd|A test command'
     * @param array|Params|null $params
     * @param callable|null $handler
     * @return $this
     */
    public function route($pathOrSyntax, $params = null, $handler = null)
    {
        // If a syntax string with description is provided, split it
        $description = '';
        $path = $pathOrSyntax;
        if (is_string($pathOrSyntax) && strpos($pathOrSyntax, '|') !== false) {
            list($pathPart, $description) = explode('|', $pathOrSyntax, 2);
            $path = trim($pathPart);
            $description = trim($description);
        }

        $node = $this->router->addRoute($path, $params, $handler, $description);
        return $this;
    }

    /**
     * Define a group of routes under a common prefix.
     *
     * Usage:
     *   $cli->group('user|User commands', function ($group) {
     *       $group->route('add', [...], $cb);
     *   });
     *
     * @param string|string[] $pathOrSyntax
     * @param callable|null $builder
     * @return $this
     */
    public function group($pathOrSyntax, $builder = null)
    {
        $description = '';
        $path = $pathOrSyntax;
        if (is_string($pathOrSyntax) && strpos($pathOrSyntax, '|') !== false) {
            list($pathPart, $description) = explode('|', $pathOrSyntax, 2);
            $path = trim($pathPart);
            $description = trim($description);
        }

        // ensure a node exists for the group path
        $node = $this->router->addRoute($path, null, null, $description);

        if ($builder !== null) {
            // Build a "scoped" router wrapper for this group
            $group = new CLIGroup($this, $node);
            $builder($group);
        }

        return $this;
    }

    /**
     * Parse and dispatch.
     *
     * @param array|string|null $args
     * @return mixed
     * @throws \Exception
     */
    public function parse($args = null)
    {
        if (php_sapi_name() !== 'cli') {
            throw new \Exception("CLI2::parse() must be run from the command line");
        }

        // Build Args from input or $argv
        $this->args = new Args($args);

        // Resolve route + parse params at each scope
        list($node, $globalResults, $scopeResults, $routeResults) = $this->resolveRouteAndParams($this->args);

        // At this point, errors (if any) are already printed and we can bail out.
        // If there was a fatal error, node will still be root and no handler will run.
        $handler = $node->handler();
        $hasChildren = !empty($node->children());
        $isRoot = ($node === $this->rootNode);

        if ($isRoot) {
            $this->printHelp();
            return false;
        }

        if ($handler === null && $hasChildren) {
            $this->printNamespaceHelp(implode(' ', $node->path()));
            return false;
        }

        if ($handler === null) {
            $this->printHelp();
            return false;
        }

        $nodeParams = $node->params();
        if ($handler !== null && $nodeParams instanceof Params) {
            // Quick check: did parse() set any errors?
            if (!empty($nodeParams->errors())) {
                // Errors already printed in resolveRouteAndParams();
                // Print help for this command.
                $this->output->line("");
                $this->help($node->path());
                // Skip handler and bail out
                return false;
            }
        }

        // Merge global+scope params for context
        $mergedGlobal = array_merge($globalResults, $scopeResults);

        $context = new CLIContext(
            $this,
            $node,
            $this->args,
            $mergedGlobal,
            $routeResults
        );

        return call_user_func($handler, $context);
    }


    /**
     * Resolve route and parse params at each scope (root, groups, leaf).
     *
     * Walks the route tree starting at rootNode:
     *  - At each node with Params, runs parse() (with ignoreUnexpected set by caller),
     *  - Then uses remaining unused tokens to decide which child to descend into.
     *
     * Returns array: [RouteNode $node, array $globalResults, array $scopeResults, array $routeResults]
     *
     * For now:
     *  - "globalResults" = root params.
     *  - "scopeResults"  = merged group/subgroup params.
     *  - "routeResults"  = leaf command params.
     *
     * @param Args $args
     * @return array
     */
    protected function resolveRouteAndParams(Args $args)
    {
        $node          = $this->rootNode;
        $globalResults = array();
        $scopeResults  = array(); // group/subgroup level
        $routeResults  = array();

        // 1) Root/global params
        if ($this->rootParams instanceof Params) {
            $result = $this->rootParams->parse($args);
            if (!$result['ok']) {
                $this->printErrors($result['errors']);
                // Treat global errors as fatal
                return array($node, $globalResults, $scopeResults, $routeResults);
            }
            $globalResults = $result['results'];
        }

        // 2) Walk down the tree:
        //    At each step:
        //      - If node has its own Params (and it's not the root params instance), parse them.
        //      - Then look at remaining unused tokens for a child name.
        while (true) {
            // Parse params attached to this node (excluding the root Params instance we already handled)
            $nodeParams = $node->params();
            if ($nodeParams instanceof Params && $nodeParams !== $this->rootParams) {
                $result = $nodeParams->parse($args);
                if (!$result['ok']) {
                    $this->printErrors($result['errors']);
                    return array($node, $globalResults, $scopeResults, $routeResults);
                }

                // If this node has children, treat its params as "scope" (group-level).
                // If it has no children, treat its params as route-specific (leaf).
                if (!empty($node->children())) {
                    $scopeResults = array_merge($scopeResults, $result['results']);
                } else {
                    // Leaf node: treat as route params
                    $routeResults = $result['results'];
                }
            }


            // Decide next child based on remaining unused tokens
            $tokens = $args->unused();
            if (empty($tokens)) {
                break;
            }

            // Next non-option token is the candidate child name
            $nextToken = null;
            foreach ($tokens as $t) {
                if (substr($t->string, 0, 1) === '-') {
                    // Skip options here; they belong to params at this or deeper scopes
                    continue;
                }
                $nextToken = $t;
                break;
            }

            if ($nextToken === null) {
                // No more route segments; only options/positionals left for this node's params
                break;
            }

            $child = $node->getChild($nextToken->string);
            if (!$child) {
                // Unrecognized subcommand; stop at current node
                break;
            }

            // Consume exactly one token for this segment from Args
            $args->markTokenUsed($nextToken);
            $node = $child;
        }

        return array($node, $globalResults, $scopeResults, $routeResults);
    }

    protected function printErrors($errors)
    {
        // small, simple error output; can be made nicer
        $this->output->line("Errors:");
        foreach ((array)$errors as $err) {
            $this->output->line(" - " . $err);
        }
    }

    protected function printHelp()
    {
        $this->output->line($this->name . " - " . $this->description);
        $this->output->line("");

        $root = $this->rootNode;
        $children = $root->children();

        if (empty($children)) {
            $this->output->line("No commands defined.");
            return;
        }

        $this->output->line("Available commands:");
        $this->printRouteTree($root, array());
    }

    /**
     * Recursively print the route tree (excluding the root itself).
     *
     * @param RouteNode $node
     * @param string[] $prefixPath
     * @return void
     */
    protected function printRouteTree(RouteNode $node, array $prefixPath)
    {
        foreach ($node->children() as $child) {
            $path = array_merge($prefixPath, array($child->name()));
            $fullName = implode(' ', $path);
            $desc = $child->description();

            // Print this route/group line
            if ($desc !== '') {
                $this->output->line("  " . $fullName . "  -  " . $desc);
            } else {
                $this->output->line("  " . $fullName);
            }

            // Recurse into children (nested subcommands)
            $this->printRouteTree($child, $path);
        }
    }

        /**
     * Collect Params objects for a given node:
     * - root/global Params
     * - Params from each group node along the path
     * - Params for the leaf node itself
     *
     * @param RouteNode $node
     * @return array [Params|null $rootParams, Params[] $groupParamsList, Params|null $leafParams]
     */
    protected function collectParamsForNode(RouteNode $node)
    {
        $rootParams = $this->rootParams instanceof Params ? $this->rootParams : null;

        // Walk from root down to this node, collecting Params at intermediate group nodes.
        $groupParamsList = array();

        $path = $node->path(); // e.g. ['user','add']
        $current = $this->rootNode;

        // For each segment except the last (leaf), treat node as group-level.
        $segmentCount = count($path);
        for ($i = 0; $i < $segmentCount - 1; $i++) {
            $seg = $path[$i];
            $child = $current->getChild($seg);
            if (!$child instanceof RouteNode) {
                break;
            }
            $current = $child;

            $p = $current->params();
            if ($p instanceof Params && $p !== $this->rootParams) {
                $groupParamsList[] = $p;
            }
        }

        // Leaf params: whatever is attached to the final node
        $leafParams = $node->params();
        if (!($leafParams instanceof Params) || $leafParams === $this->rootParams) {
            $leafParams = null;
        }

        return array($rootParams, $groupParamsList, $leafParams);
    }

    /**
     * Print help for a specific RouteNode.
     *
     * - If it's root: same as printHelp()
     * - If it has children and no handler: namespace help
     * - If it has a handler: detailed command help (path, description, params)
     *
     * @param RouteNode $node
     * @return void
     */
    protected function printNodeHelp(RouteNode $node)
    {
        if ($node === $this->rootNode) {
            $this->printHelp();
            return;
        }

        $hasChildren = !empty($node->children());
        $handler     = $node->handler();

        if ($hasChildren && $handler === null) {
            // Namespace help (unchanged)
            $this->printNamespaceHelp(implode(' ', $node->path()));
            return;
        }

        // Leaf or node with handler: detailed command help

        $pathStr = implode(' ', $node->path());
        $desc    = $node->description();

        $this->output->line($pathStr . ($desc !== '' ? ' - ' . $desc : ''));
        $this->output->line('');

        // Collect params along the path: root + all group nodes + this leaf
        list($rootParams, $groupParamsList, $leafParams) = $this->collectParamsForNode($node);

        // Build usage: command path + positionals from leaf params
        $usage = $pathStr;

        $leafPositionals = array();
        if ($leafParams instanceof Params) {
            // Use reflection to access positionals, which are protected
            $ref = new \ReflectionClass($leafParams);
            if ($ref->hasProperty('positionals')) {
                $prop = $ref->getProperty('positionals');
                $prop->setAccessible(true);
                $leafPositionals = (array)$prop->getValue($leafParams);
            }
        }

        foreach ($leafPositionals as $pos) {
            /** @var \boru\cli2\Params\Positional $pos */
            $name = $pos->name();
            if ($name === null || $name === '') {
                continue;
            }
            // Required vs optional: if required() then <name>, else [name]
            if ($pos->required()) {
                $usage .= ' <' . $name . '>';
            } else {
                $usage .= ' [' . $name . ']';
            }
        }

        $this->output->line('Usage:');
        $this->output->line('  ' . $usage);
        $this->output->line('');

        // Arguments (leaf positionals)
        if (!empty($leafPositionals)) {
            $this->output->line('Arguments:');
            foreach ($leafPositionals as $pos) {
                $name = $pos->name();
                if ($name === null || $name === '') {
                    continue;
                }
                $desc = $pos->description();
                $label = $name;
                if ($pos->required()) {
                    $label .= ' (required)';
                }
                $this->output->line(sprintf('  %-12s %s', $label, $desc));
            }
            $this->output->line('');
        }

        // Options (root + group + leaf options/flags)
        // We'll walk each Params set and list non-positional params.
        $optionLines = array();

        $appendOptionsFrom = function (Params $params) use (&$optionLines) {
            $ref = new \ReflectionClass($params);
            if (!$ref->hasProperty('definitions')) {
                return;
            }
            $prop = $ref->getProperty('definitions');
            $prop->setAccessible(true);
            $defs = (array)$prop->getValue($params);

            foreach ($defs as $def) {
                // Skip positionals here; we already listed them
                if ($def instanceof \boru\cli2\Params\Positional) {
                    continue;
                }

                /** @var \boru\cli2\Params\Param $def */
                $long  = $def->name();
                $short = $def->short();
                $desc  = $def->description();

                $parts = array();
                if ($short !== null && $short !== '') {
                    $parts[] = '-' . $short;
                }
                if ($long !== null && $long !== '') {
                    $parts[] = '--' . $long;
                }

                if (empty($parts)) {
                    continue;
                }

                $optLabel = implode(', ', $parts);
                $optionLines[] = array($optLabel, $desc);
            }
        };

        if ($rootParams instanceof Params) {
            $appendOptionsFrom($rootParams);
        }
        foreach ($groupParamsList as $gp) {
            if ($gp instanceof Params) {
                $appendOptionsFrom($gp);
            }
        }
        if ($leafParams instanceof Params) {
            $appendOptionsFrom($leafParams);
        }

        if (!empty($optionLines)) {
            $this->output->line('Options:');
            foreach ($optionLines as $line) {
                list($optLabel, $optDesc) = $line;
                $this->output->line(sprintf('  %-18s %s', $optLabel, $optDesc));
            }
        }
    }


    public function printNamespaceHelp($namespace)
    {
        $node = $this->router->root();
        $parts = preg_split('/\s+/', trim($namespace));
        foreach ($parts as $part) {
            $child = $node->getChild($part);
            if (!$child) {
                $this->output->line("Unknown namespace: " . $namespace);
                return;
            }
            $node = $child;
        }

        $this->output->line(implode(' ', $node->path()) . " - " . $node->description());
        $this->output->line("");
        $this->output->line("Subcommands:");
        $this->printRouteTree($node, $node->path());
    }

    /**
     * Public API to display help for an optional path.
     *
     * @param string|string[]|null $path
     * @return void
     */
    public function help($path = null)
    {
        if ($path === null || $path === '' || $path === false) {
            $this->printHelp();
            return;
        }

        if (!is_array($path)) {
            $path = preg_split('/\s+/', trim((string)$path));
        }

        $node = $this->findNode($path);
        if (!$node instanceof RouteNode) {
            $this->output->line('Unknown command or group: ' . (is_array($path) ? implode(' ', $path) : $path));
            return;
        }

        $this->printNodeHelp($node);
    }

}
