<?php
namespace boru\cli2;

use boru\cli2\Models\Args;
use boru\cli2\Output\OutputInterface;
use boru\cli2\Models\RouteNode;

/**
 * Immutable(ish) context passed to CLI2 handlers.
 */
class CLIContext
{
    /** @var CLI */
    protected $cli;

    /** @var RouteNode */
    protected $routeNode;

    /** @var Args */
    protected $args;

    /** @var array */
    protected $globalParams;

    /** @var array */
    protected $routeParams;

    /** @var OutputInterface */
    protected $output;

    public function __construct(
        CLI $cli,
        RouteNode $routeNode,
        Args $args,
        array $globalParams = array(),
        array $routeParams = array()
    ) {
        $this->cli          = $cli;
        $this->routeNode    = $routeNode;
        $this->args         = $args;
        $this->globalParams = $globalParams;
        $this->routeParams  = $routeParams;
        $this->output       = $cli->output();
    }

    /** @return CLI */
    public function cli()
    {
        return $this->cli;
    }

    /**
     * Returns the output interface, optionally writing a line.
     * @return OutputInterface
     */
    public function output($output = null)
    {
        if($output !== null) {
            $this->output->line($output);
        }
        return $this->output;
    }

    /** @return RouteNode */
    public function routeNode()
    {
        return $this->routeNode;
    }

    /** @return string[] */
    public function routePath()
    {
        return $this->routeNode->path();
    }

    /** @return string */
    public function routeName()
    {
        return $this->routeNode->name();
    }

    /** @return string */
    public function routeDescription()
    {
        return $this->routeNode->description();
    }

    /** @return Args */
    public function args()
    {
        return $this->args;
    }

    /**
     * All global/root parameter results.
     *
     * @return array
     */
    public function globalParams()
    {
        return $this->globalParams;
    }

    /**
     * All route-specific parameter results.
     *
     * @return array
     */
    public function params()
    {
        return $this->routeParams;
    }

    /**
     * Convenience accessor that merges global + route params.
     * Route params override global on conflicts.
     *
     * @return array
     */
    public function allParams()
    {
        return array_merge($this->globalParams, $this->routeParams);
    }


    /**
     * Find a route node by path.
     *
     * @param string|string[] $path
     * @return RouteNode|null
     */
    public function findNode($path)
    {
        return $this->cli()->findNode($path);
    }

    /**
     * Get a single param from route params, falling back to global.
     *
     * @param string $name
     * @param mixed $default
     * @return mixed
     */
    public function get($name, $default = null)
    {
        if (array_key_exists($name, $this->routeParams)) {
            return $this->routeParams[$name];
        }
        if (array_key_exists($name, $this->globalParams)) {
            return $this->globalParams[$name];
        }
        return $default;
    }
}
