<?php
namespace boru\cli2;

use boru\cli2\Models\Params;
use boru\cli2\Models\RouteNode;

class CLIGroup
{
    /** @var CLI */
    protected $cli2;

    /** @var RouteNode */
    protected $node;

    public function __construct(CLI $cli2, RouteNode $node)
    {
        $this->cli2 = $cli2;
        $this->node = $node;
    }

    /**
     * Define parameters that apply to this group (namespace) and all its child commands.
     *
     * By default, group-level params ignore unexpected arguments so that
     * deeper subcommands/positionals are not treated as errors during this pass.
     *
     * @param Params|array|null $params
     * @return Params
     */
    public function params($params = null)
    {
        if ($params === null) {
            $existing = $this->node->params();
            if ($existing instanceof Params) {
                return $existing;
            }

            $groupParams = new Params();
            // Group-level params should not complain about subcommand tokens.
            $groupParams->ignoreUnexpected(true);
            $this->node->params($groupParams);

            return $groupParams;
        }

        if ($params instanceof Params) {
            $groupParams = $params;
        } else {
            $groupParams = new Params($params);
        }

        // Ensure group-level params are tolerant of unexpected route tokens
        $groupParams->ignoreUnexpected(true);
        $this->node->params($groupParams);

        return $groupParams;
    }

    /**
     * Define a route under this group.
     *
     * @param string $syntaxOrName e.g. 'add|Add user' or 'add'
     * @param array|Params|null $params
     * @param callable|null $handler
     * @return $this
     */
    public function route($syntaxOrName, $params = null, $handler = null)
    {
        $description = '';
        $name = $syntaxOrName;
        if (is_string($syntaxOrName) && strpos($syntaxOrName, '|') !== false) {
            list($namePart, $description) = explode('|', $syntaxOrName, 2);
            $name = trim($namePart);
            $description = trim($description);
        }

        // build full path: parent path + this name
        $parentPath = $this->node->path();
        $path = array_merge($parentPath, array($name));

        // Use the existing router from CLI2 instead of creating a new one
        $router = $this->cli2->getRouter();
        $router->addRoute($path, $params, $handler, $description);

        return $this;
    }

    /**
     * Register a CommandInterface under this group's namespace.
     *
     * Commands with ParentPathInterface keep full control over their parent
     * path; otherwise, the parent defaults to this group's path.
     *
     * @param CommandInterface $command
     * @return $this
     */
    public function register(CommandInterface $command)
    {
        if (!method_exists($this->cli2, 'registerCommandWithDefaultParent')) {
            throw new \LogicException('CLIGroup::register() requires CLI to support registerCommandWithDefaultParent().');
        }

        $path = $this->node->path();
        $this->cli2->registerCommandWithDefaultParent($command, $path);

        return $this;
    }

    /**
     * Get this group's full path segments.
     *
     * @return string[]
     */
    public function path()
    {
        return $this->node->path();
    }
}
