<?php
namespace boru\cli2\Models;

class Args
{
    /** @var ArgsToken[] */
    protected $tokens = array();

    public function __construct($args = null)
    {
        if ($args === null) {
            global $argv;
            if (is_array($argv)) {
                $args = $argv;
                array_shift($args); // drop script name
            } else {
                $args = array();
            }
        }

        if (!is_array($args)) {
            $args = preg_split('/\s+/', trim((string)$args));
        }

        $normalized = array();
        foreach ($args as $raw) {
            if ($raw === '' || $raw === null) {
                continue;
            }
            foreach ($this->normalizeToken((string)$raw) as $t) {
                $normalized[] = new ArgsToken($t);
            }
        }

        $this->tokens = array_values($normalized);
    }

    protected function normalizeToken($token)
    {
        // --value=123 => ['--value', '123']
        if (strlen($token) > 3 && substr($token, 0, 2) === '--') {
            $pos = strpos($token, '=');
            if ($pos !== false) {
                $name  = substr($token, 0, $pos);
                $value = substr($token, $pos + 1);
                return array($name, $value);
            }
            return array($token);
        }

        // -v123 => ['-v', '123']
        if (strlen($token) > 2 && substr($token, 0, 1) === '-' && substr($token, 1, 1) !== '-') {
            $name  = substr($token, 0, 2);
            $value = substr($token, 2);
            if ($value !== '') {
                return array($name, $value);
            }
            return array($token);
        }

        return array($token);
    }

    /** @return ArgsToken[] */
    public function all()
    {
        return $this->tokens;
    }

    /** @return ArgsToken[] unused tokens (in order) */
    public function unused()
    {
        $out = array();
        foreach ($this->tokens as $token) {
            if (!$token->used) {
                $out[] = $token;
            }
        }
        return $out;
    }

    /**
     * Consume and return the next unused token, or null if none.
     *
     * @return ArgsToken|null
     */
    public function shift()
    {
        foreach ($this->tokens as $token) {
            if (!$token->used) {
                $token->used = true;
                return $token;
            }
        }
        return null;
    }

    /**
     * Peek at the next unused token without consuming it.
     *
     * @return ArgsToken|null
     */
    public function peek()
    {
        foreach ($this->tokens as $token) {
            if (!$token->used) {
                return $token;
            }
        }
        return null;
    }

    public function markTokenUsed(ArgsToken $tok)
    {
        $tok->used = true;
    }


    /**
     * Mark the first N unused tokens as used.
     *
     * @param int $n
     * @return void
     */
    public function markFirstNUnusedUsed($n)
    {
        if ($n <= 0) {
            return;
        }
        foreach ($this->tokens as $token) {
            if (!$token->used) {
                $token->used = true;
                $n--;
                if ($n === 0) {
                    break;
                }
            }
        }
    }

    /**
     * Mark a specific index as used.
     *
     * @param int $index
     * @return void
     */
    public function markUsed($index)
    {
        if (isset($this->tokens[$index])) {
            $this->tokens[$index]->used = true;
        }
    }
}
