<?php
namespace boru\cli2\Params;

/**
 * Parses syntax strings like "*b|book+|Book or books".
 *
 * Legend:
 *   * at start      => required
 *   + after name    => multiple
 *
 * We support two styles:
 *   - Options/flags: "*b|book+|Book or books"
 *   - Positionals:   "*file|File to process" or "files+|Files to process"
 */
class ParamSyntax
{
    /**
     * Generic parse: returns ['required','multiple','short','name','description'].
     *
     * @param string $syntax
     * @return array
     */
    public static function parse($syntax)
    {
        $syntax = trim($syntax);
        $required = false;

        if ($syntax === '') {
            return array(
                'required'    => false,
                'multiple'    => false,
                'short'       => null,
                'name'        => null,
                'description' => '',
            );
        }

        // 1) Leading '*'
        if (substr($syntax, 0, 1) === '*') {
            $required = true;
            $syntax   = substr($syntax, 1);
        }

        // 2) Split into up to 3 parts
        $parts = explode('|', $syntax, 3);
        $short = isset($parts[0]) ? trim($parts[0]) : '';
        $name  = isset($parts[1]) ? trim($parts[1]) : '';
        $desc  = isset($parts[2]) ? trim($parts[2]) : '';

        $multiple = false;

        // 3) '+' on short or long
        if ($short !== '' && substr($short, -1) === '+') {
            $multiple = true;
            $short    = substr($short, 0, -1);
        }
        if ($name !== '' && substr($name, -1) === '+') {
            $multiple = true;
            $name     = substr($name, 0, -1);
        }

        return array(
            'required'    => $required,
            'multiple'    => $multiple,
            'short'       => ($short !== '') ? $short : null,
            'name'        => ($name  !== '') ? $name  : null,
            'description' => $desc,
        );
    }

    /**
     * Special-case parsing for positionals.
     * Accepts "name|Description" or "*name|Description" or "name+|Description".
     *
     * @param string $syntax
     * @return array
     */
    public static function parsePositional($syntax)
    {
        $syntax = trim($syntax);
        $required = false;

        if ($syntax === '') {
            return array(
                'required'    => false,
                'multiple'    => false,
                'name'        => null,
                'description' => '',
            );
        }

        // Leading '*'
        if (substr($syntax, 0, 1) === '*') {
            $required = true;
            $syntax   = substr($syntax, 1);
        }

        $parts = explode('|', $syntax, 2);
        $name  = isset($parts[0]) ? trim($parts[0]) : '';
        $desc  = isset($parts[1]) ? trim($parts[1]) : '';

        $multiple = false;

        if ($name !== '' && substr($name, -1) === '+') {
            $multiple = true;
            $name     = substr($name, 0, -1);
        }

        return array(
            'required'    => $required,
            'multiple'    => $multiple,
            'name'        => ($name !== '') ? $name : null,
            'description' => $desc,
        );
    }
}
