<?php
namespace boru\cli2\Traits;

use boru\cli2\Models\RouteNode;
use boru\cli2\CLIGroup;
use boru\cli2\Models\Params;

use boru\cli2\Contracts\CommandInterface;
use boru\cli2\Contracts\CommandGroupInterface;
use boru\cli2\Contracts\ParentPathInterface;
use boru\cli2\Contracts\CommandDescriptionInterface;
use boru\cli2\Contracts\CommandGroupCommandsInterface;
use boru\cli2\Contracts\CommandParamsInterface;
use boru\cli2\Contracts\GroupParamsInterface;
use boru\cli2\Contracts\RootParamsInterface;

trait ClassCommandTrait
{
    /**
     * Register a command or command group object.
     *
     * @param CommandInterface|CommandGroupInterface $thing
     * @return $this
     */
    public function register($thing)
    {
        if ($thing instanceof CommandGroupInterface) {
            return $this->registerGroup($thing);
        }

        if ($thing instanceof CommandInterface) {
            return $this->registerCommand($thing);
        }

        throw new \InvalidArgumentException(
            'CLI::register() expects CommandInterface or CommandGroupInterface'
        );
    }

    /**
     * Public entrypoint for CLIGroup: register a command with a default parent path.
     *
     * If the command implements ParentPathInterface, its own parentPath() wins.
     * Otherwise, $defaultParentPath is used as the parent segments.
     *
     * @param CommandInterface $command
     * @param string[] $defaultParentPath
     * @return $this
     */
    public function registerCommandWithDefaultParent(CommandInterface $command, array $defaultParentPath)
    {
        $this->registerCommand($command, $defaultParentPath);
        return $this;
    }

    /**
     * Register a CommandInterface, using capability interfaces for placement/params.
     *
     * @param CommandInterface $command
     * @param string[] $defaultParentPath Optional default parent path segments
     * @return $this
     */
    protected function registerCommand(CommandInterface $command, array $defaultParentPath = array())
    {
        // 1) Root params capability
        if ($command instanceof RootParamsInterface) {
            $this->applyRootParamsFrom($command->rootParams());
        }

        // 2) Determine parent path (if any)
        //    - Command's explicit ParentPathInterface wins
        //    - Otherwise, use $defaultParentPath (e.g. from CLIGroup)
        $parentSegments = array();
        if ($command instanceof ParentPathInterface) {
            $parentSegments = $this->normalizeParentPath($command->parentPath());
        } else {
            $parentSegments = $defaultParentPath;
        }

        // 3) Ensure group node exists
        $groupNode = $this->ensureGroupNode($parentSegments);

        // 4) Group params capability (attach to group node)
        if ($command instanceof GroupParamsInterface) {
            $this->attachParamsToNode($groupNode, $command->groupParams());
        }

        // 5) Route params capability (leaf)
        $routeParams = null;
        if ($command instanceof CommandParamsInterface) {
            $routeParams = $command->commandParams();
        }

        // 6) Description capability
        $desc = '';
        if ($command instanceof CommandDescriptionInterface) {
            $desc = (string)$command->description();
        }

        // 7) Build full path
        $leafName = $command->name();
        $fullSegments = $parentSegments;
        $fullSegments[] = $leafName;
        $path = implode(' ', $fullSegments);

        $syntax = $desc !== '' ? $path . '|' . $desc : $path;

        // 8) Register leaf route
        $this->route($syntax, $routeParams, array($command, 'handle'));

        return $this;
    }

    /**
     * Register a CommandGroupInterface as a named namespace.
     *
     * @param CommandGroupInterface $group
     * @return $this
     */
    protected function registerGroup(CommandGroupInterface $group)
    {
        $name = $group->name();
        $desc = $group->description();
        $syntax = $desc !== '' ? $name . '|' . $desc : $name;

        $params = null;
        if ($group instanceof GroupParamsInterface) {
            $params = $group->groupParams();
        }

        // Capture the CLIGroup wrapper created for this namespace
        $capturedGroup = null;

        $this->group($syntax, function (CLIGroup $cliGroup) use ($params, &$capturedGroup) {
            $capturedGroup = $cliGroup;
            if ($params !== null) {
                $cliGroup->params($params);
            }
        });

        // If the group can self-register commands, give it the CLIGroup wrapper.
        if ($group instanceof CommandGroupCommandsInterface && $capturedGroup instanceof CLIGroup) {
            $group->registerGroupCommands($capturedGroup);
        }

        return $this;
    }

    /**
     * Normalize a parent path value into an array of segments.
     *
     * @param string|string[]|null|false $parentPath
     * @return string[]
     */
    protected function normalizeParentPath($parentPath)
    {
        if ($parentPath === null || $parentPath === false) {
            return array();
        }

        if (is_array($parentPath)) {
            $segments = array();
            foreach ($parentPath as $part) {
                $part = trim((string)$part);
                if ($part !== '') {
                    $segments[] = $part;
                }
            }
            return $segments;
        }

        $parentPath = trim((string)$parentPath);
        if ($parentPath === '') {
            return array();
        }

        $parts = preg_split('/\s+/', $parentPath);
        if ($parts === false) {
            return array();
        }

        $segments = array();
        foreach ($parts as $part) {
            $part = trim($part);
            if ($part !== '') {
                $segments[] = $part;
            }
        }

        return $segments;
    }

    /**
     * Ensure a RouteNode exists for the given path segments and return it.
     *
     * @param string[] $segments
     * @return RouteNode
     */
    protected function ensureGroupNode(array $segments)
    {
        if (empty($segments)) {
            return $this->rootNode;
        }

        $path = implode(' ', $segments);

        // addRoute() will create nodes as needed; no params/handler here.
        return $this->router->addRoute($path, null, null, '');
    }

    /**
     * Attach or merge Params onto a RouteNode.
     *
     * @param RouteNode $node
     * @param Params|array|null $params
     * @return void
     */
    protected function attachParamsToNode(RouteNode $node, $params)
    {
        if ($params === null) {
            return;
        }

        // Normalize to Params instance
        if ($params instanceof Params) {
            $incoming = $params;
        } else {
            $incoming = new Params((array)$params);
        }

        $existing = $node->params();

        if ($existing instanceof Params) {
            $existing->merge($incoming);
        } else {
            // New params for this node: make them tolerant of unexpected tokens,
            // since this is used for group/subgroup params in the routing tree.
            $incoming->ignoreUnexpected(true);
            $node->params($incoming);
        }
    }

    /**
     * Merge additional params into the root/global Params.
     *
     * @param Params|array|null $extra
     * @return void
     */
    protected function applyRootParamsFrom($extra)
    {
        if ($extra === null) {
            return;
        }

        // Ensure we have a root Params instance
        $rootParams = $this->params();

        if ($extra instanceof Params) {
            $rootParams->merge($extra);
        } else {
            $rootParams->merge(new Params((array)$extra));
        }
    }

    /**
     * Find a RouteNode by a path string or array.
     *
     * Examples:
     *   findNode('user');        // user group
     *   findNode('user add');    // user add command
     *   findNode(['user','add']); // same
     *
     * Returns null if no such node exists.
     *
     * @param string|string[] $path
     * @return RouteNode|null
     */
    public function findNode($path)
    {
        if ($path === '' || $path === null || $path === false) {
            return $this->rootNode;
        }

        if (is_array($path)) {
            $segments = $path;
        } else {
            $segments = preg_split('/\s+/', trim((string)$path));
        }

        $node = $this->rootNode;
        foreach ($segments as $segment) {
            if ($segment === '') {
                continue;
            }
            $child = $node->getChild($segment);
            if (!$child instanceof RouteNode) {
                return null;
            }
            $node = $child;
        }

        return $node;
    }
}
