# boru/cli2

`boru/cli2` is a router-based command-line library for PHP (v2 of the boru CLI tooling). It is designed to:

- Model commands and subcommands as a **route tree** (like HTTP routing, but for CLI).
- Provide a clean, composable **parameter system** (options, flags, positionals).
- Support **root/global options**, **group/namespaced options**, and **route-specific options**.
- Be usable both with **inline closures** and **class-based commands**.
- Keep the core small (`CoreCLI`) and hang richer “sugar” APIs off a higher-level facade (`CLI`).

> `cli2` is intentionally independent from `boru/cli` v1. The old library can coexist, but cli2 does not reuse v1 internals.

---

## Core vs. Facade

The library is split into two layers:

- **`CoreCLI`** (`boru\cli2\CoreCLI`)
  - Minimal routing + parsing core.
  - You define routes, groups, params, and call `parse()`.
- **`CLI`** (`boru\cli2\CLI`)
  - Extends `CoreCLI`.
  - Uses `Traits\ClassCommandTrait` to add **class-based command registration** and helper utilities (like `help()`).

If you only want the lower-level router/params API, use `CoreCLI`. For most applications, use `CLI`.

---

## Quick example (closure-based, CoreCLI-style)

This is the “core” way to use cli2: define routes and groups with closures.

```php
use boru\cli2\CLI;
use boru\cli2\CLIContext;
use boru\cli2\Params\Flag;
use boru\cli2\Params\Option;
use boru\cli2\Params\Positional;

$cli = CLI::create('app|CLI2 example');

// Root/global params
$cli->params(array(
    Flag::create('d|debug|Enable debug output'),
));

// Group: user
$cli->group('user|User commands', function ($user) {
    // Group-level params: available for all `user` commands
    $user->params(array(
        Option::create('u|username|Default username'),
    ));

    // user add <name>
    $user->route('add|Add a user', array(
        Positional::create('name|Name of user'),
    ), function (CLIContext $ctx) {
        $params = $ctx->allParams();
        $name   = $ctx->get('name');
        echo "Adding user: {$name}\n";
        if (!empty($params['debug'])) {
            echo "(debug mode)\n";
        }
    });

    // user update <name> -f field -v value
    $user->route('update|Update a user', array(
        Positional::create('name|Name of user'),
        Option::create('f|field|Field to update'),
        Option::create('v|value|New value'),
    ), function (CLIContext $ctx) {
        echo "Updating user: {$ctx->get('name')}\n";
        echo "Field: {$ctx->get('field')}, value: {$ctx->get('value')}\n";
    });
});

// Leaf: deploy
$cli->route('deploy|Deploy the app', array(
    Flag::create('f|force|Force deployment'),
    Option::create('e|env|Environment name'),
), function (CLIContext $ctx) {
    echo "Deploying to env: " . $ctx->get('env', 'default') . "\n";
    if ($ctx->get('force')) {
        echo "Force: yes\n";
    }
});

$cli->parse(); // or $cli->parse('deploy -f --env=prod');
```

Example invocations:

- `php app.php` → root help.
- `php app.php deploy -f --env=prod`
- `php app.php -d user add alice`
- `php app.php user --username bob add alice`
- `php app.php user update alice -f email -v new@example.com`

---

## Class-based commands (CLI + ClassCommandTrait)

On top of the core API, `CLI` adds support for **class-based commands** via `Traits\ClassCommandTrait` and a small set of interfaces.

### Command/group interfaces

Core interfaces:

- `CommandInterface`
  - Minimal: `name()` + `handle(CLIContext $ctx)`.
- `CommandGroupInterface`
  - Named group / namespace: `name()` + `description()`.

Capability interfaces (all optional, mix-and-match):

- `CommandDescriptionInterface`
  - Adds `description(): string` to a command.
- `ParentPathInterface`
  - Adds `parentPath(): string|string[]|null|false` to say where the command lives (`user`, `user admin`, etc.).
- `RouteParamsInterface`
  - Adds `routeParams(): Params|array|null` for leaf-level parameters.
- `GroupParamsInterface`
  - Adds `groupParams(): Params|array|null` to attach params at the parent group level.
- `RootParamsInterface`
  - Adds `rootParams(): Params|array|null` to attach params at the root/global level.

The `CLI::register()` method looks at the interfaces implemented by a class and wires everything into the route tree and params model.

### Example: user group + user add command

```php
use boru\cli2\CLI;
use boru\cli2\CLIContext;
use boru\cli2\CommandInterface;
use boru\cli2\CommandGroupInterface;
use boru\cli2\CommandDescriptionInterface;
use boru\cli2\ParentPathInterface;
use boru\cli2\RouteParamsInterface;
use boru\cli2\GroupParamsInterface;
use boru\cli2\RootParamsInterface;
use boru\cli2\Params\Flag;
use boru\cli2\Params\Option;
use boru\cli2\Params\Positional;

// Group: "user"
class UserGroup implements CommandGroupInterface
{
    public function name()
    {
        return 'user';
    }

    public function description()
    {
        return 'User commands';
    }
}

// Command: user add <name>
class UserAddCommand implements
    CommandInterface,
    CommandDescriptionInterface,
    ParentPathInterface,
    RouteParamsInterface,
    GroupParamsInterface,
    RootParamsInterface
{
    public function name()
    {
        return 'add';
    }

    public function description()
    {
        return 'Add a user (class-based)';
    }

    // Lives under "user"
    public function parentPath()
    {
        // Could also be ['user']; both are supported.
        return 'user';
    }

    // Leaf-level params
    public function routeParams()
    {
        return array(
            Positional::create('name|Name of user'),
        );
    }

    // Params for the "user" namespace
    public function groupParams()
    {
        return array(
            Option::create('u|username|Default username (from class)'),
        );
    }

    // Params at the root/global level
    public function rootParams()
    {
        return array(
            Flag::create('d|debug|Enable debug (from class)'),
        );
    }

    public function handle(CLIContext $ctx)
    {
        $params = $ctx->params();
        $name   = isset($params['name']) ? $params['name'] : null;

        $ctx->output("UserAddCommand: adding {$name}");
        $ctx->output('Global params: ' . var_export($ctx->globalParams(), true));
        $ctx->output('Route params: ' . var_export($ctx->params(), true));
        $ctx->output('Route path: ' . implode(' ', $ctx->routePath()));
    }
}

// Bootstrapping
$cli = CLI::create('app|CLI2 class-based example');

// You can still define baseline root params directly:
$cli->params(array(
    Flag::create('v|verbose|Verbose output'),
));

// Register group + commands via the sugar:
$cli
    ->register(new UserGroup())
    ->register(new UserAddCommand());

$cli->parse(); // php app.php user add alice -u bob -d
```

`CLI::register()` will:

- Place `UserAddCommand` under the `user` group (via `ParentPathInterface`).
- Merge `rootParams()` into the root/global params.
- Merge `groupParams()` into the `user` group node.
- Attach `routeParams()` to the `user add` leaf.
- Use `description()` for help text if `CommandDescriptionInterface` is implemented.

This makes it easy to have commands that “bring along” their own routing and parameter definitions.

---

## Help and inspection

The `CLI` facade adds a small, context-aware help layer on top of `CoreCLI`:

- `CLI::help($path = null)`:
  - `help()` – print root help (same as running with no valid route).
  - `help('user')` – help for the `user` namespace.
  - `help('user add')` – help for the `user add` command.

Internally this uses:

- `CLI::findNode($path)` (from `ClassCommandTrait`) to resolve a `RouteNode` from a path string/array.
- `CLI::printNodeHelp(RouteNode $node)` to decide whether to show:
  - full root help,
  - namespace help (group with children),
  - or a simple one-line description for a leaf command.

You can also wire this behind a class-based `help` command that uses the same capability interfaces, e.g.:

```php
class HelpCommand implements
    CommandInterface,
    CommandDescriptionInterface,
    RouteParamsInterface
{
    public function name()
    {
        return 'help';
    }

    public function description()
    {
        return 'Show help for a command or namespace';
    }

    public function routeParams()
    {
        return array(
            Positional::create('topic|Command or group to show help for'),
        );
    }

    public function handle(CLIContext $ctx)
    {
        $cli    = $ctx->cli();
        $params = $ctx->params();
        $topic  = isset($params['topic']) ? trim($params['topic']) : '';

        if ($topic === '') {
            $cli->help();
        } else {
            $cli->help($topic);
        }
    }
}
```

Register it like any other command:

```php
$cli->register(new HelpCommand());
```

Then:

- `php app.php help`
- `php app.php help user`
- `php app.php help "user add"`

will all render context-aware help.

---

## Core concepts

### 1. CLI, CoreCLI, and routing

The main entry points are:

- `CoreCLI`
  - Pure routing/params/parsing.
  - Methods: `__construct()`, `output()`, `name()`, `description()`, `params()`, `route()`, `group()`, `parse()`, and internal helpers.
- `CLI`
  - Extends `CoreCLI`.
  - Adds:
    - `Traits\ClassCommandTrait` (class-based `register()`, `findNode()`, etc.).
    - A small public `help()` wrapper and `printNodeHelp()`.

Under the hood, both use a `RouteNode` tree managed by `CommandRouter`:

- Root node is the CLI name.
- Each group / command adds nodes beneath that.
- Resolution walks the tree based on non-option tokens.

### 2. Args model (`Models\Args`)

`boru\cli2\Models\Args` is responsible for tokenizing and normalizing CLI input:

- Normalizes forms like:
  - `--env=prod` → `['--env', 'prod']`
  - `-eprod` → `['-e', 'prod']`
- Tracks **used** vs **unused** tokens:
  - `all()` → all tokens.
  - `unused()` → tokens not yet consumed by any parser.
  - `shift()` / `peek()` → iterate over unused tokens.
- `Params::parse()` marks tokens as used so they disappear from `unused()`.

### 3. Params model (`Params` + `Models\Params`)

Parameter types live under `boru\cli2\Params`, the parser under `boru\cli2\Models\Params`.

Param types:

- **Option** (`Params\Option`)
  - Has a value.
  - Built via syntax strings:

    ```php
    Option::create('*e|env|Environment name');   // required
    Option::create('f+|file|Input file(s)');     // multiple
    ```

- **Flag** (`Params\Flag`)
  - Boolean-like; no value.
  - E.g. `Flag::create('d|debug|Enable debug')`.

- **Positional** (`Params\Positional`)
  - For positional arguments.
  - E.g. `Positional::create('name|Username')`, `Positional::create('files+|Files to process')`.

Syntax rules:

- Leading `*` → required.
- Trailing `+` on a name → multiple.
- `short|long|Description` pattern:
  - `d|debug|Enable debug`:
    - short: `d` → `-d`
    - long: `debug` → `--debug`
  - `files+|Files to process` (positional):
    - name: `files` (multiple)

`Models\Params`:

```php
use boru\cli2\Models\Params;
use boru\cli2\Params\Flag;
use boru\cli2\Params\Positional;

$params = new Params(array(
    'v|verbose|Verbose output',          // inferred Option
    Flag::create('d|debug|Enable debug'),
    Positional::create('name|Name'),
));
```

- `parse(Args $args)` returns:

  ```php
  [
      'ok'      => bool,
      'results' => ['name' => $value, ...],
      'errors'  => ['Error message', ...],
  ]
  ```

- During parse, it:
  - Consumes matching option/flag tokens and their values.
  - Assigns positional tokens in order.
  - Applies defaults and required flags.
  - Can ignore unknown options/extra args via `ignoreUnexpected(true)` — used for global and group-level params so route names and deeper options aren’t treated as errors.

`Params` also supports merging definitions (used by `ClassCommandTrait`) via an internal `merge()` helper.

### 4. Resolution and scopes (`CoreCLI::parse()`)

`CoreCLI::parse()` orchestrates:

1. Build `Args` from `$argv` (or explicit `$args`).
2. Call `resolveRouteAndParams(Args $args)`:

   - Start at root `RouteNode`.
   - **Root/global params**:
     - If `params(...)` was called, parse that `Params` once with `ignoreUnexpected(true)`.
   - Walk down the route tree:
     - At each node:
       - If the node has `Params` (and it’s not the root params), parse them.
       - For group nodes (with children), results go into “scope” results.
       - For leaf nodes, results go into “route” results.
     - Look at `Args::unused()` to find the next **non-option** token:
       - If it matches a child name, consume it and descend.
       - If not, stop at the current node.

3. Dispatch:

   - If still at root with children and no handler → root help.
   - If at a group node with children and no handler → namespace help for that group.
   - If at a leaf with a handler → call it with a `CLIContext`.

`CLIContext` gives the handler access to:

- Global/root params.
- Group/scope params.
- Route/leaf params.
- Route path.
- Output.

---

## Installation

Install via Composer:

```bash
composer require boru/cli2
```

Then:

```php
use boru\cli2\CLI;

$cli = CLI::create('app|My CLI');
// define routes/groups/commands...
$cli->parse();
```

---

## When to use what

- Use **closure-based routes** when:
  - You’re prototyping or building small scripts.
  - You’re comfortable defining everything inline in one place.

- Use **class-based commands** when:
  - You want modular features that can register themselves.
  - You want to conditionally register commands based on permissions or feature flags.
  - You like having commands, groups, and params declared as explicit PHP classes.

Both approaches share the same router and parameter parsing under the hood, so you can mix them freely in the same CLI if you want.