<?php
namespace boru\cli2\Models;

use boru\cli2\Models\Args;
use boru\cli2\Params\Flag;
use boru\cli2\Params\Option;
use boru\cli2\Params\Param;
use boru\cli2\Params\ParamSyntax;
use boru\cli2\Params\Positional;

/**
 * Params: v2 parameter parser built on Args (v2).
 *
 * Independent from commands; just parses tokens into named values.
 */
class Params
{
    /** @var array<string,Param> */
    protected $definitions = array();

    /** @var Positional[] list of positional Param definitions, in order */
    protected $positionals = array();

    /** @var array<string,mixed> */
    protected $results = array();

    /** @var array<string> */
    protected $errors  = array();

    /** @var bool whether to suppress 'Unexpected argument: ...' errors */
    protected $ignoreUnexpected = false;

    /**
     * @param array $params An array of Param instances or syntax strings.
     */
    public function __construct(array $params = array())
    {
        foreach ($params as $param) {
            $this->addParam($param);
        }
    }

    /**
     * Control whether 'Unexpected argument: ...' should be treated as an error.
     *
     * @param bool|null $flag
     * @return bool|$this
     */
    public function ignoreUnexpected($flag = null)
    {
        if ($flag === null) {
            return $this->ignoreUnexpected;
        }
        $this->ignoreUnexpected = (bool)$flag;
        return $this;
    }

    /**
     * Accept either a concrete Param object or a syntax string to convert.
     *
     * @param mixed $param
     * @return void
     */
    public function addParam($param)
    {
        // 1) Syntax string case: parse and create underlying Param object
        if (is_string($param)) {
            $param = $this->createParamFromSyntax($param);
            if ($param === null) {
                return;
            }
        }

        // 2) Concrete Param objects
        if ($param instanceof Param) {
            $name = $param->name();
            if ($name === null) {
                $name = $param->short();
            }
            if ($name === null) {
                return;
            }

            $this->definitions[$name] = $param;

            if ($param instanceof Positional) {
                $this->positionals[] = $param;
            }
        }
    }

    /**
     * Parse a syntax string like "*b|book+|Book or books"
     * and return a concrete Flag/Option/Positional Param instance
     * with required/multiple set appropriately.
     *
     * Legend:
     *   * at start   => required
     *   + after name => multiple
     *
     * Heuristics:
     *   - If first segment is empty and second is non-empty: positional.
     *   - For non-positional:
     *       by default treat as Option; callers can explicitly pass Flag::create()
     *       if they want a flag (no value).
     *
     * @param string $syntax
     * @return Param|null
     */
    protected function createParamFromSyntax($syntax)
    {
        $syntax = trim($syntax);
        if ($syntax === '') {
            return null;
        }

        // Quick positional detection: if there's no short part,
        // or caller uses "name|Description" without a short.
        $parts = explode('|', ltrim($syntax, '*'), 3);
        $first = isset($parts[0]) ? trim($parts[0]) : '';
        $second = isset($parts[1]) ? trim($parts[1]) : '';

        // Heuristic: positional if there is no explicit short part (first empty)
        // and a non-empty name, or if syntax visually looks like "name|Desc".
        $looksPositional = ($first === '' && $second !== '');

        if ($looksPositional) {
            return Positional::create($syntax);
        }

        // Otherwise treat as an Option by default.
        // (If user wants a Flag, they can provide Flag::create() explicitly.)
        return Option::create($syntax);
    }

    /**
     * Parse Args into results/errors.
     *
     * @param Args $args
     * @return array ['ok'=>bool, 'results'=>array, 'errors'=>array]
     */
    public function parse(Args $args)
    {
        $this->results = array();
        $this->errors  = array();
        $posIndex      = 0;

        while (true) {
            $tok = $args->peek();
            if ($tok === null) {
                break;
            }

            $token = $tok->string;

            // Option or flag
            if (substr($token, 0, 1) === '-') {
                // Long option: --name
                if (substr($token, 0, 2) === '--') {
                    $tok = $args->shift();          // consume option token
                    $name = substr($tok->string, 2);
                    $param = $this->findByLongName($name);

                    if (!$param instanceof Param || !$param->takesValue()) {
                        if ($param instanceof Flag) {
                            $param->setFlag();
                            continue;
                        }
                        if (!$this->ignoreUnexpected) {
                            $this->errors[] = "Unknown option --" . $name;
                        }
                        continue;
                    }

                    // Value is the next token if present and not another option
                    $nextTok = $args->peek();
                    $value = null;
                    if ($nextTok !== null && substr($nextTok->string, 0, 1) !== '-') {
                        $value = $args->shift()->string;   // consume value
                    }

                    if ($value === null) {
                        if (!$this->ignoreUnexpected) {
                            $this->errors[] = "Missing value for option --" . $name;
                        }
                        continue;
                    }

                    $param->value($value);
                    continue;
                }

                // Short option or flag: -v
                $tok   = $args->shift();               // consume option token
                $token = $tok->string;
                $short = substr($token, 1, 1);
                $param = $this->findByShortName($short);

                if ($param === null) {
                    if (!$this->ignoreUnexpected) {
                        $this->errors[] = "Unknown option -" . $short;
                    }
                    continue;
                }

                if ($param instanceof Flag) {
                    $param->setFlag();
                    continue;
                }

                if ($param instanceof Option) {
                    $nextTok = $args->peek();
                    $value   = null;
                    if ($nextTok !== null && substr($nextTok->string, 0, 1) !== '-') {
                        $value = $args->shift()->string;   // consume value
                    }

                    if ($value === null) {
                        if (!$this->ignoreUnexpected) {
                            $this->errors[] = "Missing value for option -" . $short;
                        }
                        continue;
                    }

                    $param->value($value);
                    continue;
                }

                continue;
            }

            // Positional
            $tok = $args->peek();
            if ($tok === null) {
                break;
            }

            $token = $tok->string;

            // If we have a positional definition at this index, consume it.
            if (isset($this->positionals[$posIndex])) {
                $tok   = $args->shift();   // now actually consume
                $token = $tok->string;

                $posParam = $this->positionals[$posIndex];
                $name     = $posParam->name();
                if ($name === null) {
                    $posIndex++;
                    continue;
                }

                $posParam->value($token);

                if (!$posParam->multiple()) {
                    $posIndex++;
                }

                continue;
            }

            if (!$this->ignoreUnexpected) {
                $this->errors[] = "Unexpected argument: " . $token;
            }
            break;
        }

        // After scanning tokens, apply defaults and check required
        foreach ($this->definitions as $name => $param) {
            $value   = $param->value();
            $default = $param->defaultValue();
            $isReq   = $param->required();

            if ($value === null && $default !== null) {
                $value = $default;
                $param->value($default);
            }

            if ($isReq && $value === null) {
                $this->errors[] = "Missing required parameter: " . $name;
            }

            if ($value !== null) {
                $this->results[$name] = $value;
            }
        }

        return array(
            'ok'      => empty($this->errors),
            'results' => $this->results,
            'errors'  => $this->errors,
        );
    }

    /**
     * Merge definitions from another Params or an array of Param instances/syntax.
     *
     * Later definitions override earlier ones with the same name.
     *
     * @param Params|array $other
     * @return $this
     */
    public function merge($other)
    {
        if ($other instanceof self) {
            // Re-add all definitions from the other instance
            foreach ($other->definitions as $param) {
                $this->addParam($param);
            }
            return $this;
        }

        if (is_array($other)) {
            foreach ($other as $param) {
                $this->addParam($param);
            }
        }

        return $this;
    }


    /**
     * Find a param definition by short name (single char).
     *
     * @param string $short
     * @return Param|null
     */
    protected function findByShortName($short)
    {
        foreach ($this->definitions as $def) {
            if ($def->short() === $short) {
                return $def;
            }
        }
        return null;
    }

    /**
     * Find a param definition by long name.
     *
     * @param string $name
     * @return Param|null
     */
    protected function findByLongName($name)
    {
        if (isset($this->definitions[$name])) {
            return $this->definitions[$name];
        }

        // Fallback: some params might be keyed by short name
        foreach ($this->definitions as $defName => $def) {
            if ($def->name() === $name) {
                return $def;
            }
        }

        return null;
    }

    /**
     * Get parsed results after parse().
     *
     * @return array
     */
    public function results()
    {
        return $this->results;
    }

    /**
     * Get accumulated errors after parse().
     *
     * @return array
     */
    public function errors()
    {
        return $this->errors;
    }
}
