<?php
namespace boru\cli2\Models;

use boru\cli2\Models\Params;

/**
 * Represents a node in the CLI route tree (either a group or a leaf command).
 */
class RouteNode
{
    /** @var string[] path parts from root, e.g. ['user', 'add'] */
    protected $path = array();

    /** @var string short name of this segment, e.g. 'add' */
    protected $name;

    /** @var string human description */
    protected $description = '';

    /** @var Params|null parameters specific to this node */
    protected $params;

    /** @var RouteNode[] keyed by child name */
    protected $children = array();

    /** @var callable|null handler for leaf commands */
    protected $handler;

    /** @var bool */
    protected $isGroup = false;

    public function __construct(array $path, $description = '', Params $params = null, $handler = null, $isGroup = false)
    {
        $this->path        = $path;
        $this->name        = end($path);
        $this->description = $description;
        $this->params      = $params;
        $this->handler     = $handler;
        $this->isGroup     = $isGroup;
    }

    public function name()
    {
        return $this->name;
    }

    public function path()
    {
        return $this->path;
    }

    public function description()
    {
        return $this->description;
    }

    public function params(Params $params = null)
    {
        if ($params === null) {
            return $this->params;
        }
        $this->params = $params;
        return $this;
    }

    public function handler($handler = null)
    {
        if ($handler === null) {
            return $this->handler;
        }
        $this->handler = $handler;
        return $this;
    }

    public function isGroup()
    {
        return $this->isGroup;
    }

    public function children()
    {
        return $this->children;
    }

    public function addChild(RouteNode $child)
    {
        $this->children[$child->name()] = $child;
        return $this;
    }

    public function getChild($name)
    {
        return isset($this->children[$name]) ? $this->children[$name] : null;
    }

    public function setDescription($description)
    {
        $this->description = $description;
        return $this;
    }
}
