<?php
namespace boru\dhapi;

use boru\dhapi\AuthenticatorInterface;
use boru\dhapi\authenticators\BasicKey;
use boru\dhapi\AutoloaderInterface;
use boru\dhapi\autoloaders\ClassPathLoader;
use boru\dhapi\RouterInterface;
use boru\dhapi\routers\ClassPathRouter;
use \boru\dhutils\dhGlobal;
use \boru\dhutils\dhRequest;

class dhAPI {
    protected $startTime;
    protected $request;
    protected $response;

    protected $autoloader;
    protected $router;
    protected $authenticator;

    protected $baseDir = "api/";
    protected $requireAuth = false;
    protected $prettyResponse = false;
    protected $outputResponse = true;
    protected $endOnResponse = true;

    private $start_time;

    protected $user = [
        "authed"=>false,
        "permissions"=>[],
        "data"=>[]
    ];

    public $status = array(  
        200 => 'OK',
        403 => 'Not Authorized',
        404 => 'Not Found',   
        405 => 'Method Not Allowed',
        420 => 'Missing required Inputs',
        500 => 'Internal Server Error',
    );

    public $validMethods = ["DELETE","POST","GET","PUT"];

    /**
     * Valid options:
     * * router - callable|RouterInterface|false
     * * autoloader -callable|AutoloaderInterface|false
     * * authenticator - callable|AuthenticatorInterface|false
     * * requireAuth - (false) boolean
     * * prettyResponse - (false) boolean
     * * outputResponse - (true) boolean
     * * endOnResponse - (true) boolean
     * @param array $options 
     * @return void 
     */
    public function __construct($options=[]) {
        $this->setRouter(dhGlobal::getval($options,"router",null));
        $this->setAutoloader(dhGlobal::getval($options,"autoloader",null));
        $this->setAuthenticator(dhGlobal::getval($options,"authenticator",null));
        $this->setRequireAuth(dhGlobal::getval($options,"requireAuth",false));
        if(($baseDir = dhGlobal::getval($options,"baseDir",false)) !== false) {
            $this->setBaseDir($baseDir);
        }
        $this->setPrettyResponse(dhGlobal::getval($options,"prettyResponse",false));
        $this->setOutputResponse(dhGlobal::getval($options,"outputResponse",true));
        $this->setEndOnResponse(dhGlobal::getval($options,"endOnResponse",true));
    }
    public function init() {
        $this->start_time = microtime(true);
        header("Access-Control-Allow-Orgin: *");
		header("Access-Control-Allow-Methods: *");
		header("Content-Type: application/json");
        
        
        $this->request = dhRequest::fromGlobals(true);
        

        $this->initAutoloader();
        $this->runAuthenticator();
        if(($result = $this->runRouter()) !== false) {
            $className = $result["class"];
            if(class_exists($className,true)) { 
               $obj = new $className($this);
               $obj->process($result["args"]);
            } else {
                $this->response(["message"=>"endpoint unknown","status"=>false],404);
            }
        } else {
            //we shouldn't get here unless router is set to false.
        }
    }

    public function runRouter() {
        if(is_callable($this->router)) {
            //if router is a callback
            $fn = $this->router;
            echo $fn($this);
            return;
        } elseif($this->router instanceof RouterInterface) {
            //if router is a defined router
            return $this->router->route($this);
        } elseif($this->router !== false) {
            //default router
            $this->router = new ClassPathRouter();
            return $this->router->route($this);
        }
        return false;
    }

    public function runAuthenticator() {
        $authed = null;
        if(is_callable($this->authenticator)) {
            //if authenticator is a callback
            $fn = $this->authenticator;
            $authed = $fn($this);
        } elseif($this->authenticator instanceof AuthenticatorInterface) {
            $authed = $this->authenticator->authenticate($this);
        } elseif($this->authenticator !== false) {
            //default authenticator
            $this->authenticator = new BasicKey();
            $authed = $this->authenticator->authenticate($this);
        }
        if($this->requireAuth && !$authed) {
            throw new \Exception("Not authenticated");
        }
    }
    
    public function initAutoloader() {
        if(is_callable($this->autoloader)) {
            $fn = $this->autoloader;
            $fn($this);
        } elseif($this->autoloader instanceof AutoloaderInterface) {
            $this->autoloader->init($this);
        } elseif($this->autoloader !== false) {
            $this->autoloader = new ClassPathLoader($this);
            $this->autoloader->init($this);
        }
    }

    public function response($data=[],$code=200) {
        $header = "HTTP/1.1 " . $code . " " . $this->_requestStatus($code);
        header($header);
		
		if($this->prettyResponse) {
			$output = json_encode($data,JSON_PRETTY_PRINT | JSON_PARTIAL_OUTPUT_ON_ERROR);
		} else {
			$output = json_encode($data, JSON_PARTIAL_OUTPUT_ON_ERROR);
		}
        if($this->outputResponse) {
            echo $output;
        }
		$response = [];
		$response["headers"][] = $header;
		$response["output"] = $output;
        $this->response = $response;
        if($this->endOnResponse) {
            exit();
        }
		return $this->response;
    }

    //request aliases
    public function getMethod() {
        return $this->request->getMethod();
    }
    public function getURI($what=null,$default=null) {
        if(is_null($what)) {
            return $this->request->get("uri");
        }
        return $this->request->get("uri.".$what,$default);
    }
    public function getRequest($what=null,$default=null) {
        $query = $this->getQuery($what,null);
        $post = $this->getPost($what,null);
        $body = $this->getBody($what,null);
        $value = $default;
        $value = !is_null($query) ? $query : $value;
        $value = !is_null($post) ? $post : $value;
        $value = !is_null($body) ? $body : $value;
        return $value;
    }
    public function getQuery($what=null,$default=null) {
        if(is_null($what)) {
            return $this->request->get("get");
        }
        return $this->request->get("get.".$what,$default);
    }
    public function getPost($what=null,$default=null) {
        if(is_null($what)) {
            return $this->request->get("post");
        }
        return $this->request->get("post.".$what,$default);
    }
    public function getBody($what=null,$default=null) {
        if(is_null($what)) {
            return $this->request->get("body");
        }
        return $this->request->get("body.".$what,$default);
    }
    public function getHeader($what=null,$default=null) {
        if(is_null($what)) {
            return $this->request->get("headers");
        }
        return $this->request->get("headers.".$what,$default);
    }
    public function getServer($what=null,$default=null) {
        if(is_null($what)) {
            return $this->request->get("server");
        }
        return $this->request->get("server.".$what,$default);
    }

    /**
     * Set the value of autoloader
     *
     * @param   mixed  $autoloader  
     * @return  self
     */
    public function setAutoloader($autoloader) {
        $this->autoloader = $autoloader;
        return $this;
    }

    /**
     * Set the value of router
     *
     * @param   mixed  $router  
     * @return  self
     */
    public function setRouter($router) {
        $this->router = $router;
        return $this;
    }

    /**
     * Set the value of authenticator
     *
     * @param   mixed  $authenticator  
     * @return  self
     */
    public function setAuthenticator($authenticator) {
        $this->authenticator = $authenticator;
        return $this;
    }

    /**
     * Get the value of baseDir
     *
     * @return  mixed
     */
    public function getBaseDir() {
        return $this->baseDir;
    }

    /**
     * Set the value of baseDir
     *
     * @param   mixed  $baseDir  
     * @return  self
     */
    public function setBaseDir($baseDir) {
        $this->baseDir = $baseDir;
        return $this;
    }

    /**
     * Set the value of pretty
     *
     * @param   mixed  $pretty  
     * @return  self
     */
    public function setPrettyResponse($prettyResponse) {
        $this->prettyResponse = $prettyResponse;
        return $this;
    }
    /**
     * Set the value of output
     *
     * @param   mixed  $output  
     * @return  self
     */
    public function setOutputResponse($outputResponse) {
        $this->outputResponse = $outputResponse;
        return $this;
    }

    public function setRequireAuth($reqAuth) {
        $this->requireAuth = $reqAuth;
        return $this;
    }
    public function getRequireAuth() {
        return $this->requireAuth;
    }

    public function isAuthed() {
        return dhGlobal::getVal($this->user,"authed",false);
    }
    public function getPerms($permissionKey=null,$default=false) {
        if(is_null($permissionKey)) {
            return dhGlobal::getVal($this->user,"permissions",$default);
        }
        return dhGlobal::getVal($this->user["permissions"],$permissionKey,$default);
    }

    public function setUser($authed=false,$permissions=[],$data=[]) {
        $this->user["authed"] = $authed;
        $this->user["permissions"] = $permissions;
        $this->user["data"] = $data;
    }

    private function _requestStatus($code) {
		 
		return ($this->status[$code])?$this->status[$code]:$this->status[500]; 
	}


    /**
     * Set the value of endOnResponse
     *
     * @param   mixed  $endOnResponse  
     * @return  self
     */
    public function setEndOnResponse($endOnResponse) {
        $this->endOnResponse = $endOnResponse;
        return $this;
    }
}