<?php
namespace boru\dhapi\v2;

use boru\dhapi\v2\core\Container;
use boru\dhutils\dhGlobal;
use boru\dhapi\v2\core\Response;
use boru\dhapi\v2\core\Request;
use boru\dhapi\v2\routing\Router;

class API {
    public static $allowedMethods = ["DELETE","POST","GET","PUT"];
    public static $requestVar = "APIREQ";

    /** @var Request */
    private $request;

    /** @var Router */
    private $router;

    /** @var Container */
    private $container;

    /** @var Authenticator */
    private $authenticator;

    /**
     * 
     * @param Router $router 
     * @param Request $request 
     * @return void 
     */
    public function __construct($request=null,$router=null) {
        $this->request = $request;
        $this->router = $router;
        $this->container = new Container();
    }
    
    private $baseClasss = "";

    /** @return Request */
    public static function requestFromGlobals() {
        $method = $_SERVER["REQUEST_METHOD"];
        if($method == "POST" && array_key_exists('HTTP_X_HTTP_METHOD', $_SERVER)) {
            if(in_array($_SERVER['HTTP_X_HTTP_METHOD'],['DELETE','PUT','PATCH'])) {
                $method = $_SERVER['HTTP_X_HTTP_METHOD'];
            } else {
                return Response::fromError(405,"Unusual Request Method: ".$_SERVER['HTTP_X_HTTP_METHOD'],true,true);
            }
        }
        if(!in_array($method,static::$allowedMethods)) {
            Response::fromError(405,"Method Not Allowed",true,true);
        }
        $rawBody = file_get_contents("php://input");
        $body = [];
        if(!empty($rawBody)) {
            $body = json_decode($rawBody,true);
            if($body === null) {
                return Response::fromError(400,"Invalid JSON",true,true);
            }

        }
        $path = "/";
        if(isset($_GET[static::$requestVar])) {
            $path.= trim($_GET[static::$requestVar],"/");
            unset($_GET[static::$requestVar]);
        }
        return new Request($_GET,$body,getallheaders(),$method,$path);
    }
    /**
     * @param Request $request
     * @return API
     */
    public function setRequest($request) {
        $this->request = $request;
        return $this;
    }
    /**
     * @param Router $router
     * @return API
     */
    public function setRouter($router) {
        $this->router = $router;
        return $this;
    }
    /**
     * @param Container $container
     * @return API
     */
    public function setContainer($container) {
        $this->container = $container;
        return $this;
    }
    /**
     * @param Authenticator $authenticator
     * @return API
     */
    public function setAuthenticator($authenticator) {
        $this->authenticator = $authenticator;
        return $this;
    }
    public function setClassBaseNamespace($namespace) {
        $this->baseClasss = $namespace;
        return $this;
    }
    public function getClassBaseNamespace() {
        return $this->baseClasss;
    }
    /**
     * @return Authenticator
     */
    public function getAuthenticator() {
        if(is_null($this->authenticator)) {
            $this->authenticator = new Authenticator();
        }
        return $this->authenticator;
    }
    public function getAuthenticated() {
        return $this->getAuthenticator()->getAuthed();
    }
    public function isAuthed() {
        return $this->getAuthenticator()->isAuthed();
    }
    /** @return Request */
    public function getRequest() {
        if(is_null($this->request)) {
            $this->request = static::requestFromGlobals();
        }
        if(!($this->request instanceof Request)) {
            throw new \Exception("Request is not a Request object");
        }
        return $this->request;
    }
    /** @return Router */
    public function getRouter() {
        return $this->router;
    }
    /** @return Container */
    public function getContainer() {
        return $this->container;
    }
    


    public function getMethod() {
        return $this->request->getMethod();
    }
    public function getPath() {
        return $this->request->getPath();
    }
    public function body($key=null,$defaultValue=null) {
        return $this->request->body($key,$defaultValue);
    }
    public function get($key=null,$defaultValue=null) {
        return $this->request->body($key,$defaultValue);
    }
    public function header($key=null) {
        if(is_null($key)) {
            return $this->request->getHeaders();
        }
        return $this->request->getHeader($key);
    }
    public function headers() {
        return $this->request->getHeaders();
    }

    /**
     * @return Response 
     */
    public function process() {
        if(!is_null($this->getAuthenticator())) {
            $auth = $this->getAuthenticator()->authenticate($this);
            if($auth instanceof Response) {
                return $auth;
            } elseif($auth === false) {
                return Response::fromError("Authentication Failed",401,true,true);
            }
        }
        if(($route = $this->router->matchFromPath($this->getPath(), $this->getMethod(),$this)) !== false) {
            if($route instanceof Response) {
                return $route;
            }
            $parameters = $route->getParameters();
            // $arguments = ['id' => 2]
            $arguments = $route->getVars();
            if(is_array($parameters)) {
                $controllerName = $parameters[0];
                if(substr($controllerName,0,1) !== "\\" && !class_exists($controllerName)) {
                    $controllerName = $this->getClassBaseNamespace().$controllerName;
                }
                $methodName = $parameters[1] ? $parameters[1] : null;
                if(!class_exists($controllerName)) {
                    return Response::fromError("Endpoint controller Not Found",404,true,true);
                }
                $controller = new $controllerName();
                if(!is_callable([$controller,"setAPI"]) || !is_callable([$controller,"isPermissed"])) {
                    return Response::fromError("Endpoint controller not valid",404,true,true);
                }
                $controller->setAPI($this);
                $permissed = $controller->isPermissed();
                if($permissed instanceof Response) {
                    return $permissed;
                } elseif($permissed === false) {
                    return Response::fromError("Not permissed",401,true,true);
                }
                if (!is_callable($controller)) {
                    $controller =  [$controller, $methodName];
                }
                if (!is_callable($controller)) {
                    return Response::fromError("Endpoint controller method not found",404,true,true);
                }
                $response = $controller(...array_values($arguments));
            } else {
                $controller = $parameters;
                if (!is_callable($controller)) {
                    return Response::fromError("Endpoint controller method not found",404,true,true);
                }
                $response = $controller($this,...array_values($arguments));
            }
            if(!($response instanceof Response)) {
                //If the response is not a response object, we didn't return valid data from our endpoint.. so not implemented
                ob_clean();
                return Response::fromError("Endpoint method not implemented",404,true,true);
            }
            return $response;
        } else {
            return Response::fromError("Endpoint Not Found",404,true,true);
        }
    }

    /**
     * @param \mrr\api\v3\router\Route $route
     */
    public function addRoute($route) {
        if(is_null($this->router)) {
            $this->router = new Router();
        }
        $this->router->add($route);
    }
}