<?php
namespace boru\dhapi\v2\auth;

use boru\dhapi\v2\API;
use boru\dhapi\v2\auth\CallbackAuthenticator;
use boru\dhutils\dhGlobal;

abstract class HmacAuthenticator implements CallbackAuthenticator {
    /** @var API */
    private $API;
    private $authed = false;
    private $authMessage = "";

    private $apiKey = false;
    private $apiSign = false;
    private $apiNonce = 0;

    public function getApiKey() {
        return $this->apiKey;
    }
    public function getApiSign() {
        return $this->apiSign;
    }
    public function getApiNonce() {
        return $this->apiNonce;
    }
    public function setApiKey($apiKey) {
        $this->apiKey = $apiKey;
    }
    public function setApiSign($apiSign) {
        $this->apiSign = $apiSign;
    }
    public function setApiNonce($apiNonce) {
        $this->apiNonce = $apiNonce;
    }

    public function authenticate($API) {
        /** @var API */
        $this->API = $API;
        $this->apiKey = $API->header("x-api-key",false);
        $this->apiSign = $API->header("x-api-sign",false);
        $this->apiNonce = $API->header("x-api-nonce",false);
        if($this->validate()) {
            $this->updateKeyNonce();
            $this->log(true);
            return true;
        } else {
            $this->log(false);
            return false;
        }
        return true;
    }
    public function getAuthed() {
        return $this->authed;
    }
    public function isAuthed() {
        return $this->authed;
    }
    public function getAuthMesage() {
        return $this->authMessage;
    }

    private function validate() {
        $keyInfo = $this->getKeyInfo();
        if(is_null($keyInfo) || !is_array($keyInfo) || !isset($keyInfo["apikey"])) {
            $this->authed=false;
            $this->authMessage = "Invalid or missing API Key";
            return false;
        }
        $sign_check = $this->signRequest($keyInfo["secret"]);
        if($sign_check != $this->apiSign) {
            $this->authed=false;
            $this->authMessage = [
                "Signature Failure",
                "String to sign:",
                $this->getSignString(),
                "Expected signature:",
                $sign_check
            ];
            return false;
        }
        if($keyInfo["nonce"] >= $this->apiNonce) {
            $this->authed=false;
            $this->authMessage = "Nonce failure";
            return false;
        }
        return true;
    }

    protected function getSignString() {
        $path = "/".dhGlobal::trimString($this->API->getPath(),"/");
        return $this->apiKey.$this->apiNonce.$path;
    }
    protected function signRequest($keySecret) {
        return hash_hmac("sha1", $this->getSignString(), $keySecret);
    }
    abstract public function updateKeyNonce();
    abstract public function getKeyInfo($force=false);
    abstract public function log($success);
    abstract function permissed($API, $route);
}