<?php
namespace boru\dhapi\v2\core;

use boru\dhutils\dhGlobal;
use boru\dhapi\v2\core\Utils;
use mrr\BaseObject;

class Response {
    protected static $autoEmit = false;
    protected static $autoTerminate = true;

    public static $requestCodeStatus = [
        200=>"OK",
        400=>"Bad Request",
        401=>"Unauthorized",
        403=>"Forbidden",
        404=>"Not Found",
        405=>"Method Not Allowed",
        406=>"Not Acceptable",
        409=>"Conflict",
        410=>"Gone",
        411=>"Length Required",
        412=>"Precondition Failed",
        413=>"Payload Too Large",
        414=>"URI Too Long",
        415=>"Unsupported Media Type",
        416=>"Range Not Satisfiable",
        417=>"Expectation Failed",
        418=>"I'm a teapot",
        420=>"Missing Required Inputs",
        421=>"Invalid Inputs",
        422=>"Unprocessable Entity",
        423=>"Locked",
        424=>"Failed Dependency",
        425=>"Too Early",
        426=>"Upgrade Required",
        428=>"Precondition Required",
        429=>"Too Many Requests",
        431=>"Request Header Fields Too Large",
        451=>"Unavailable For Legal Reasons",
        500=>"Internal Server Error",
        501=>"Not Implemented",
        502=>"Bad Gateway",
        503=>"Service Unavailable",
        504=>"Gateway Timeout",
        505=>"HTTP Version Not Supported",
        506=>"Variant Also Negotiates",
        507=>"Insufficient Storage",
        508=>"Loop Detected",
        510=>"Not Extended",
        511=>"Network Authentication Required",
    ];
    protected $data = [];
    protected $statusCode = 200;
    protected $message = null;

    protected static $pretty = true;
    protected static $fieldList = [];

    public function __construct($data=[], $status=200, $message=null) {
        $this->setData($data);
        $this->setStatus($status);
        $this->setMessage($message);
    }
    
    public function emit($terminate=null) {
        $this->emitStatusCode();
        header("Content-Type: application/json");
        echo $this->jsonEncode(static::parseFieldList($this->data));
        if(static::shouldTerminate($terminate)) {
            exit;
        }
    }
    private function jsonEncode($data) {
        return json_encode($data, $this->getJsonFlags());
    }
    private function emitStatusCode() {
        if(!isset(self::$requestCodeStatus[$this->statusCode])) {
            $this->statusCode = 500;
        }
        header("HTTP/1.1 " .$this->statusCode ." " .self::$requestCodeStatus[$this->statusCode]);
    }
    private function getJsonFlags() {
        if(static::$pretty) {
            return JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_PARTIAL_OUTPUT_ON_ERROR;
        } else {
            return JSON_UNESCAPED_SLASHES | JSON_PARTIAL_OUTPUT_ON_ERROR;
        }
    }

    public function setStatus($status) {
        $this->statusCode = $status;
    }
    public function setMessage($message) {
        $this->message = $message;
    }
    public static function setPretty($pretty) {
        static::$pretty = $pretty;
    }
    /**
     * Set the field list to use when returning data
     * @param array $fieldList the fields to return
     */
    public static function setFieldList($fieldList) {
        if(!is_array($fieldList)) {
            $fieldList = Utils::splitToArray($fieldList);
        }
        static::$fieldList = $fieldList;
    }
    public function setData($data) {
        $this->data = $data;
    }

    private static function parseFieldList($data) {
        if(!empty(static::$fieldList) && is_array($data) && !empty($data)) {
            $newData = static::filterArray(json_decode(json_encode($data,JSON_PARTIAL_OUTPUT_ON_ERROR),true) , static::$fieldList);
            if(empty($newData)) {
                return null;
            } else {
                return $newData;
            }
        } else {
            return $data;
        }
    }
    private static function filterArray($array,$fieldFilter,$currentKey='') {
        $result = [];
        
        foreach ($array as $key => $value) {
            $newKey = $currentKey === '' ? $key : $currentKey . '.' . $key;

            $matchedFilter = '';
            foreach ($fieldFilter as $filter) {
                $pattern = "(\.|\.[0-9]+\.)";
                $prePattern = "([0-9]+\.)?";
                $newFilter = str_replace('.', $pattern, $filter);
                $regex = '/^'.$prePattern. $newFilter . '(\.|$)/';
                if (preg_match($regex, $newKey)) {
                    $matchedFilter = $filter;
                    break;
                }
            }

            if ($matchedFilter !== '') {
                $result[$key] = $value;
            } elseif (is_array($value)) {
                $filteredValue = static::filterArray($value, $fieldFilter, $newKey);
                
                if (!empty($filteredValue)) {
                    $result[$key] = $filteredValue;
                }
            }
        }
        return $result;
    }

    // Static Methods
    /**
     * Create a new Response object from an error, emits and terminates (exit) by default.
     * @param string|array $error the error message or data array
     * @param int $code (default:400) the HTTP status code
     * @param bool $emit (default:true) emit the response (false) return the response object (useful for chaining)
     * @param bool $terminate (default:true) terminate the script after emitting (false) return the response object (useful for chaining)
     * @return Response 
     */
    public static function fromError($error, $code=400, $emit=null, $terminate=null) {
        $response = new ErrorResponse([
            "error"=>$error,
            "code"=>$code,
        ], $code);
        if(static::shouldEmit($emit)) {
            $response->emit($terminate);
        }
        return $response;
    }

    /**
     * Create a new Response object from an exception, emits and terminates (exit) by default.
     * @param \Exception $exception the exception object
     * @param int $code (default:500) the HTTP status code
     * @param bool $emit (default:true) emit the response (false) return the response object (useful for chaining)
     * @param bool $terminate (default:true) terminate the script after emitting (false) return the response object (useful for chaining)
     * @return Response 
     */
    public static function fromException($exception, $code=500, $emit=null, $terminate=null) {
        $response = new ErrorResponse([
            "error"=>$exception->getMessage(),
            "code"=>$code,
        ], $code);
        if(static::shouldEmit($emit)) {
            $response->emit($terminate);
        }
        return $response;
    }

    /**
     * Create a new Response object from a success, emits and terminates (exit) by default.
     * @param string|array $data the data to return
     * @param int $code (default:200) the HTTP status code
     * @param bool $emit (default:true) emit the response (false) return the response object (useful for chaining)
     * @param bool $terminate (default:true) terminate the script after emitting (false) return the response object (useful for chaining)
     * @return Response 
     */
    public static function fromSuccess($data, $code=200, $emit=null, $terminate=null) {
        $response = new Response($data, $code);
        if(static::shouldEmit($emit)) {
            $response->emit($terminate);
        }
        return $response;
    }

    public static function shouldEmit($emit=null) {
        if($emit===null) {
            return self::$autoEmit;
        } else {
            return $emit;
        }
    }
    public static function shouldTerminate($terminate=null) {
        if($terminate===null) {
            return self::$autoTerminate;
        } else {
            return $terminate;
        }
    }
}