<?php
namespace boru\dhapi\v2\routing;

use boru\dhapi\v2\interfaces\ClassControllerInterface;
use boru\dhapi\v2\API;
use boru\dhapi\v2\routing\Router;

class Route {
    private $name;
    private $path;
    private $methods = [];
    private $parameters = [];
    private $vars = [];
    private $varCallbacks = [];
    private $router;
    private $requireAuth = false;
    public function __construct($name,$path,$methods,$parameters, $requireAuth=false) {
        if(!is_array($methods)) $methods = [$methods];
        if(empty($methods)) {
            throw new \Exception("Route must have at least one method");
        }
        $this->name = $name;
        $this->path = $path;
        $this->methods = $methods;
        $this->parameters = $parameters;
        $this->requireAuth = $requireAuth;
    }
    public function setRouter($router) {
        $this->router = $router;
    }
    public function setRequireAuth($requireAuth) {
        $this->requireAuth = $requireAuth;
    }
    public function getRequireAuth() {
        return $this->requireAuth;
    }
    /**
     * 
     * @param API $api 
     * @param Router $router 
     * @return bool|Response|mixed 
     */
    public function isPermissed($api) {
        if(!$this->requireAuth) {
            return true;
        }
        if($this->requireAuth && $api->isAuthed() === false) {
            return false;
        }
        return $api->getAuthenticator()->permissed($api,$this);
    }
    public function getAuth() {
        return $this->router->getAuthenticated();
    }

    public function match($path, $method) {
        $regex = $this->getPath();
        $this->varCallbacks = [];
        $regexPattern = "[^/]+";
        foreach ($this->getVarsNames() as $variable) {
            $varName = trim($variable, '{\}');
            $parts = explode("|",$varName);
            $paramName = array_shift($parts);
            $nameParts = explode(":",$paramName);
            if(count($nameParts) > 1) {
                $paramName = array_shift($nameParts);
                $shortcut = $this->router->getRegex($nameParts[0]);
                $regexPattern = $shortcut["regex"];
                if($shortcut["callback"] !== null && (!isset($this->varCallbacks[$paramName]) || !in_array($shortcut["callback"],$this->varCallbacks[$paramName]))) {
                    $this->varCallbacks[$paramName][] = $shortcut["callback"];
                }
            }
            $regex = str_replace($variable, '(?P<' . $paramName . '>'.$regexPattern.'+)', $regex);
            if(!empty($parts)) {
                foreach($parts as $part) {
                    if(!isset($this->varCallbacks[$paramName]) || !in_array($part,$this->varCallbacks[$paramName])) {
                        $this->varCallbacks[$paramName][] = $part;
                    }
                }
            }
        }
        if (in_array($method, $this->getMethods()) && preg_match('#^' . $regex . '$#sD', self::trimPath($path), $matches)) {
            $values = array_filter($matches, static function ($key) {
                return is_string($key);
            }, ARRAY_FILTER_USE_KEY);
            $this->vars = $values;
            $this->parseVarCallbacks();
            return true;
        }
        return false;
    }

    public function getName() {
        return $this->name;
    }

    public function getPath() {
        return $this->path;
    }

    public function getParameters() {
        return $this->parameters;
    }

    public function getMethods() {
        return $this->methods;
    }

    public function getVarsNames() {
        preg_match_all('/{[^}]*}/', $this->path, $matches);
        $reset = reset($matches);
        return $reset ? $reset : [];
    }

    public function hasVars() {
        return $this->getVarsNames() !== [];
    }

    public function getVars() {
        return $this->vars;
    }

    public static function trimPath($path) {
        return '/' . rtrim(ltrim(trim($path), '/'), '/');
    }
    public function parseVar($var,$value) {
        if(($callable = $this->router->getVariableCallback($var)) !== false) {
            if (!is_callable($callable)) {
                return $value;
            }
            return call_user_func($callable, $value);
        }
        return $value;
    }

    private function parseVarCallbacks() {
        foreach($this->varCallbacks as $varName=>$callbacks) {
            if(isset($this->vars[$varName])) {
                $value = $this->vars[$varName];
                foreach($callbacks as $callback) {
                    if(($callable = $this->router->getVariableCallback($callback)) !== false) {
                        if (!is_callable($callable)) {
                            continue;
                        }
                        $value = call_user_func($callable, $value);
                    }
                }
                $this->vars[$varName] = $value;
            }
        }
    }

    public static function fromArray($path,$methods,$callable,$requireAuth = false) {
        return new Route($path,$path,$methods,$callable,$requireAuth);
    }
    public static function fromClass($classNameOrObject) {
        if(!is_object($classNameOrObject)) {
            $classNameOrObject = new $classNameOrObject();
        }
        if(is_object($classNameOrObject) && !($classNameOrObject instanceof ClassControllerInterface)) {
            throw new \Exception("Class must implement ClassControllerInterface");
        }
        $routeArray = $classNameOrObject->routes();
        $routes = [];
        foreach($routeArray as $route) {
            if(!($route instanceof Route) && !is_array($route)) {
                throw new \Exception("Class must return array of Route objects");
            }
            if(is_array($route)) {
                $className = get_class($classNameOrObject);
                $callable = [$className,$route[2]];
                $route = self::fromArray($route[0],$route[1],$callable,isset($route[3]) ? $route[3] : false);
            }
            $routes[] = $route;
        }
        return $routes;
    }
}