<?php
namespace boru\dhcache;

use boru\dhcache\backends\Memcached;
use boru\dhcache\backends\CacheInterface;
use \boru\dhutils\dhGlobal;

/**
 * A wrapper class used to store values transparently in memcache
 * Can be enabled or disabled through site configuration
 * Also sets a default time if no time is passed to it to enforce caching
 **/
class dhCache {
	/** @var CacheInterface */
	private $cache;
	public $enabled=true;
    public $expirationDefault=60;
    public $expirationPlay=5;

	public static function withMemcached($options=[]) {
		$memcached = new Memcached($options);
		$instance = new self(["backend"=>$memcached]);
		return $instance;
	}

	public function __construct($options=[]) {
        if(($expirationDefault = dhGlobal::getVal($options,"expirationDefault",false)) !== false) {
			$this->expirationDefault = $expirationDefault;
		}
		if(($expirationPlay = dhGlobal::getVal($options,"expirationPlay",false)) !== false) {
			$this->expirationPlay = $expirationPlay;
		}
		$enabled = dhGlobal::getVal($options,"enabled",null);
		if(!is_null($enabled)) {
			$this->enabled = $enabled;
		}
		if(($backend = dhGlobal::getVal($options,"backend",false)) !== false) {
			$this->backend($backend);
		} else {
			if(($memcachedOpts = dhGlobal::getVal($options,"memcached",false)) !== false) {
				$memcached = new Memcached($memcachedOpts);
				$this->backend($memcached);
			}
		}
		if(is_null($this->cache)) {
			if(class_exists("\\Memcached")) {
				$memcached = new Memcached();
				$this->backend($memcached);
			}
		}
	}

	protected function backend($obj) {
		$this->cache = $obj;
	}

	public function set($key, $value, $expiration=NULL) {
		if (!$this->enabled) return false;
		return $this->cache->set($key, $value, $this->expiration($expiration));
	}

	public function get($key, $default=false) {
		if (!$this->enabled) return false;
		return $this->cache->get($key, $default);
	}

	public function lockSet($key, $value, $expiration=NULL) {
		if (!$this->enabled) return false;
		return $this->cache->atomicSet($key,$value,$this->expiration($expiration));
	}

	public function lockGet($key) {
		if (!$this->enabled) return false;
		return $this->cache->atomicGet($key);
	}
	public function atomicSet($key, $value, $expiration=NULL) {
		if (!$this->enabled) return false;
		return $this->cache->atomicSet($key,$value,$this->expiration($expiration));
	}

	public function atomicGet($key) {
		if (!$this->enabled) return false;
		return $this->cache->atomicGet($key);
	}
	
	public function lockVar($key, $value = 1, $expiration = 0) {
		if (!$this->enabled) return false;
		return $this->cache->lockVar($key,$value,$expiration);
	}
	public function unlockVar($key) {
		if (!$this->enabled) return false;
		return $this->cache->unlockVar($key);
	}
	
	public function increment($key, $expiration = null, $value = 1) {
		if (!$this->enabled) return false;
		return $this->cache->increment($key,$this->expiration($expiration),$value);
	}
	
	public function decrement($key, $expiration = null, $value = 1) {
		if (!$this->enabled) return false;
		return $this->cache->decrement($key,$this->expiration($expiration),$value);
	}
	
	public function __call($name, $arguments) {
		if (! $this->enabled) return false;
		return call_user_func_array(array($this->cache, $name), $arguments);
	}

	protected function expiration($expiration=null) {
		if (is_null($expiration)) {
			$expiration = $this->expirationDefault + rand( -$this->expirationPlay, $this->expirationPlay);
		}
		return $expiration;
	}
}