<?php
namespace boru\dhcli;

use \boru\dhutils\dhGlobal;
use \boru\dhcli\Option;
use \boru\dhutils\base\Args;

class Command {
    public $command;
    public $description;
    public $options = [];
    public $optionValues = [];
    public $errors = [];

    public $callbackUsed = false;
    public $isDefaultCommand = false;
    
    protected $status;

    //output formatting and organization variables
    public $optsRequired = [];
    public $optsOptional = [];
    public $optsFlags = [];

    public $valPlaceHolder = "<value>";
    protected $maxlenLong = 0;
    protected $maxlenShort = 0;
    protected $valPlaceHolderLen = 0;

    protected $dhargs;
    /**
     * all args from $dhargs
     */
    protected $argsAll = [];
    /**
     * unused args from $dhargs
     */
    protected $argsUnused = [];

    protected $callback;

    /**
     * Shortcut to create a new CLI Command
     * @param mixed $command The command name, used to execute
     * @param mixed $description Description of the command, used in help
     * @param \boru\dhcli\Option[] $options array of \boru\dhcli\Option objects
     * @param array $extraConf additional configuration items, if any. (commonOptions,valPlaceHolder,callback)
     * @return static 
     */
    public static function create($command,$description,$options=[],$extraConf=[]) {
        $arr = ["command"=>$command,"description"=>$description];
        if(!empty($options)) {
            $arr["opts"]=$options;
        }
        if(!empty($extraConf)) {
            foreach($extraConf as $k=>$v) {
                $arr[$k]=$v;
            }
        }
        return new static($arr);
    }

    public function __construct($options=[]) {
        $this->command = dhGlobal::getMultiVal($options,["name"=>false,"command"=>false],true);
        $this->description = dhGlobal::getVal($options,"description","");
        $this->valPlaceHolder = dhGlobal::getVal($options,"valPlaceHolder","<value>");
        $this->valPlaceHolderLen = strlen($this->valPlaceHolder);
        $opts = dhGlobal::getMultiVal($options,["opts"=>false,"opt"=>false,"options"=>false,"option"=>false,"commonOptions"=>false,"commonOpts"=>false]);
        foreach($opts as $optVal) {
            if($optVal !== false) {
                if(is_array($optVal)) {
                    $this->options($optVal);
                } elseif(is_array($optVal) || is_object($optVal)) {
                    $this->option($optVal);
                }
            }
        }
        $this->callback = dhGlobal::getVal($options,"callback",null);
        $default = dhGlobal::getVal($options,"default",false);
        if($default !== false) {
            $this->isDefaultCommand = true;
        }
    }
    public function name() {
        return $this->command;
    }
    public function addNewOpt($array) {
        $opt = new Option($array);
        if($opt->getName() != "") {
            $this->addOpt($opt);
        }
        return $this;
    }
    public function options(array $opts) {
        foreach($opts as $opt) {
            if(is_array($opt)) {
                $this->addNewOpt($opt);
            } elseif(is_object($opt)) {
                $this->option($opt);
            }
        }
        return $this;
    }
    public function option($opt) {
        if(is_array($opt)) {
            $this->addNewOpt($opt);
        } elseif(is_object($opt)) {
            return $this->addOpt($opt);
        }
    }
    public function addOpt(Option $opt) {
        $this->options[$opt->getName()] = $opt;
        if($opt->isRequired()) {
            $this->optsRequired[$opt->getName()] = $opt->getName();
        } else {
            $this->optsOptional[$opt->getName()] = $opt->getName();
        }
        if($opt->isLong()) {
            $len = strlen($opt->getLong());
            if($opt->needsValue()) {
                $len+=$this->valPlaceHolderLen+1;
            }
            $this->maxlenLong = max($this->maxlenLong,$len+2);
        }
        if($opt->isShort()) {
            $len = strlen($opt->getShort());
            if($opt->needsValue()) {
                $len+=$this->valPlaceHolderLen+2;
            } else {
                $this->optsFlags[$opt->getName()] = $opt->getName();
            }
            $this->maxlenShort = max($this->maxlenShort,$len+1);
        }
        return $this;
    }
    public function isValid() {
        foreach($this->options as $name=>$opt) {
            if($opt->isRequired()) {
                if(!$opt->hasValue()) {
                    $this->errors[$name] = "required";
                }
            }
            if($opt->needsValue() && !empty($opt->getInvalidOptions())) {
                $this->errors[$name] = "invalid value";
            } elseif($opt->needsValue() && $opt->getValue() === false) {
                $this->errors[$name] = "missing value";
            }
        }
        return empty($this->errors);
    }
    public function getOptionValues() {
        if(!$this->isValid()) {
            return false;
        }
        if(is_null($this->optionValues) || empty($this->optionValues)) {
            foreach($this->options as $name=>$opt) {
                if($opt->hasValue()) {
                    $this->optionValues[$name] = $opt->getValue();
                }
            }
        }
        return $this->optionValues;
    }
    public function getErrors() {
        return empty($this->errors) ? false : $this->errors;
    }
    public function getErrorOutput($syntaxPrefix="") {
        $this->syntax($syntaxPrefix);
        dhGlobal::outLine("");
        if(($errors = $this->getErrors()) !== false) {
            foreach($errors as $opt=>$err) {
                $opt = $this->options[$opt];
                if($err == "required") {
                    $info = "";
                    if($opt->isShort()) {
                        $info .= $opt->getInputShort()." ";
                    }
                    if($opt->isLong()) {
                        $info .= $opt->getInputLong();
                    }
                    dhGlobal::outLine("Missing required option:",$info);
                } elseif($err == "missing value") {
                    $info = "";
                    if($opt->isShort()) {
                        $info .= $opt->getInputShort()." ";
                    }
                    if($opt->isLong()) {
                        $info .= $opt->getInputLong();
                    }
                    dhGlobal::outLine("Missing value for non-required option:",$info);
                } elseif($err == "invalid value") {
                    $invalidOptions = $opt->getInvalidOptions();
                    $selArg = $opt->getParsedArg();
                    if(is_array($selArg)) {
                        $selArg = $selArg[0];
                    }
                    foreach($invalidOptions as $invalidOpt) {

                        dhGlobal::outLine("$selArg: invalid value of \"".$invalidOpt["value"]."\" -- ".$invalidOpt["reason"]);
                    }
                }
            }
        }
    }
    public function help($syntaxPrefix="") {
        dhGlobal::outLine($this->name());
        $this->syntax("Usage: ".$syntaxPrefix,"  ");
        dhGlobal::outLine("  {$this->description}");
        if(!empty($this->optsRequired)) {
            dhGlobal::outLine("   * Required Options:");
            foreach($this->optsRequired as $optName) {
                $this->_printOpt($this->options[$optName]);
            }
        }
        if(!empty($this->optsOptional)) {
            dhGlobal::outLine("   * Optional Options:");
            foreach($this->optsOptional as $optName) {
                $this->_printOpt($this->options[$optName]);
            }
        }        
    }
    public function syntax($syntaxPrefix="",$indent="") {
        dhGlobal::outLine($indent.$syntaxPrefix."{$this->command} ".$this->getSyntaxArgs());
    }
    public function getSyntaxArgs() {
        $syntaxArgs = [];
        if(!empty($this->optsRequired)) {
            foreach($this->optsRequired as $optName) {
                if($this->options[$optName]->isShort()) {
                    $syntaxArgs[] = "-".$this->options[$optName]->getShort().$this->valPlaceHolder;
                } else {
                    $syntaxArgs[] = "--".$this->options[$optName]->getLong()."=".$this->valPlaceHolder;
                }
            }
        }
        if(!empty($this->optsOptional)) {
            $syntaxArgs[] = "[optionals]";
        }
        return implode(" ",$syntaxArgs);
    }
    public function _printOpt($opt) {
        $val = $opt->needsValue();
        $short = str_pad("",$this->maxlenShort," ",STR_PAD_RIGHT);
        $long = str_pad("",$this->maxlenLong," ",STR_PAD_RIGHT);
        $hasShort = $opt->isShort();
        $hasLong = $opt->isLong();
        if($val) {
            if($opt->isShort()) {
                $short = $opt->getShort().' '.$this->valPlaceHolder;
                if($hasLong) {
                    $short.=",";
                }
                $short = str_pad("-".$short,$this->maxlenShort," ",STR_PAD_RIGHT);
            }
            if($opt->isLong()) {
                $long = str_pad("--".$opt->getLong()."=".$this->valPlaceHolder,$this->maxlenLong," ",STR_PAD_RIGHT);
            }
        } else {
            if($opt->isShort()) {
                $short = $opt->getShort();
                if($hasLong) {
                    $short.=",";
                }
                $short = str_pad("-".$short,$this->maxlenShort," ",STR_PAD_RIGHT);
            }
            if($opt->isLong()) {
                $long = str_pad("--".$opt->getLong(),$this->maxlenLong," ",STR_PAD_RIGHT);
            }
        }
        //$name = $opt->getName();
        $desc = $opt->getDescription();
        
        dhGlobal::outLine("  ",$short,"  ",$long,"  "," -- ",$desc);
    }
    public function process($args) {
        if(!is_object($args)) {
            if(is_array($args) || is_string($args)) {
                $this->args = new Args($args);
            } else {
                throw new \Exception("Invalid args for ".__CLASS__."->process()");
            }
        } else {
            $this->args = $args;
        }
        $this->dhargs = $args;
        $this->setArgs($args->getArgs(true),true);
        foreach($this->options as $name=>$opt) {
            $this->parseOptionArgs($this->options[$name]);
        }
        if(!$this->isValid()) {
            $this->setStatus(false);
            return false;
        }

        foreach($this->options as $name=>$opt) {
            if($opt->hasValue()) {
                $this->optionValues[$name] = $opt->getValue();
            }
        }
        $this->setArgs($args->getArgs());
        $this->setStatus(true);
        if(!is_null($this->callback)) {
            $result = [
                "commandName"=>$this->name(),
                "command"=>&$this,
                "options"=>$this->getOptionValues(),
                "args"=>[
                    "all"=>$this->getArgs(true),
                    "unused"=>$this->getArgs(false),
                ]
            ];
            $var = $this->callback;
            $callbackResult = $var($this->getResult());
            if(!is_null($callbackResult) && $callbackResult !== false && $callbackResult !== true) {
                if(is_array($callbackResult)) {
                    foreach($callbackResult as $line) {
                        dhGlobal::outLine($line);
                    }
                } else {
                    dhGlobal::outLine($$callbackResult);
                }
            }
            $this->callbackUsed = true;
            return true;
        }
        //returned because there was no callback
        return null;
    }
    public function getResult() {
        $result = [
            "commandName"=>$this->name(),
            "command"=>&$this,
            "options"=>$this->getOptionValues(),
            "args"=>[
                "all"=>$this->getArgs(true),
                "unused"=>$this->getArgs(false),
            ]
        ];
        return $result;
    }

    /**
     * Get the value of status
     */ 
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set the value of status
     *
     * @return  self
     */ 
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    public function setArgs($args,$all=false) {
        if(!$all) {
            $this->argsUnused = $args;
        } else {
            $this->argsAll = $args;
        }
    }
    public function getArgs($all=false) {
        if(!$all) {
            return $this->argsUnused;
        } else {
            return $this->argsAll;
        }
    }
    public function getOption($key,$default=false) {
        return isset($this->optionValues[$key]) ? $this->optionValues[$key] : $default;
    }

    public function parseOptionArgs(Option &$opt) {
        $optName = $opt->getName();

        //reset the cursor
        $this->args->cursor=0;

        //loop through all the args and process any options we have defined
        while(($arg = $this->args->next()) !== false) {
            if($this->isOptArg($arg,$opt)) {
                $this->args->markUsed();
                if($this->args->isLongArg($arg)) {
                    $opt->setParsedArg("--".$opt->getLong());
                    //$this->args->debug("isLongArg $arg");
                    if($opt->needsValue()) {
                        if(($value = $this->args->getArgValue($arg)) !== false) {
                            //$this->debug("arg found, setting value to $value");
                            $opt->setValue($value);
                        }
                    } else {
                        //$this->debug("$arg found");
                        $opt->setValue(true);
                    }
                } elseif($this->args->isShortArg($arg)) {
                    $opt->setParsedArg("-".$opt->getShort());
                    if($opt->needsValue()) {
                        if(($value = $this->args->getArgValue($arg)) !== false) {
                            //$this->debug("arg found, setting value to $value");
                            $opt->setValue($value);
                        }
                    } else {
                        //$this->debug("$arg found");
                        $opt->setValue(true);
                    }
                }
            }
        }
        return $this;
    }
    public function isOptArg($arg,$opt) {
        if(($long = $opt->getLong()) !== false) {
            $o = substr($arg,2);
            $p = strpos($o, '=');
            if($p) {
                return $long == substr($o, 0, $p);
            }
        }
        if(($short = $opt->getShort()) !== false) {
            $o = substr($arg,1);
            if(!$opt->needsValue()) {
                //if it doesn't need a value
                //eg -s
                if($short == $o) {
                    return true;
                }
                //if we don't need a value, we may be a flag that can be grouped
                //eg -sTmv
                if(strlen($o)>1) {
                    for($i=0;$i<strlen($o);$i++) {
                        if($short == $o[$i]) {
                            return true;
                        }
                    }
                }
                return false;
            } else {
                //we need a value
                //if we need a value but dont have one.. lets fail it
                if(!$this->args->getArgValue($arg)) {
                    return false;
                }
                if($short == $o[0]) {
                    return true;
                }
                return false;
            }
        }
        if($this->args->isLongArg($arg) || $this->args->isShortArg($arg)) {
            
        }
        return false;
    }
}