<?php
namespace boru\dhcli;

use \boru\dhutils\dhGlobal;
use \boru\dhcli\Command;
use \boru\dhcli\Option;
use \boru\dhutils\base\Args;

class dhCLI {
    public $commands = [];
    public $name;// = "dhCLI 1.0";
    public $description;//"A CLI utility library that parses commands for use in other sripts\nGenerally not usable by itself, but embedded in other functions";
    public $prescript = ""; //eg: php -f somescript.php
    public $defaultCommand = false;

    /** @var self */
    private static $dhcli;

    protected $cmdObj;
    protected $command;
    /**
     * Status of the process execution
     * -1 :: no matching command
     *  0 :: command error
     *  1 :: success
     */
    protected $status = 0;

    use \boru\dhutils\traits\GetOption;

    public static function makeCommand($options=[]) {
        return new Command($options);
    }
    public static function makeOption($options=[]) {
        return new Option($options);
    }

    /**
     * Returns a new Option configured for flags (no value)
     * @param string $description 
     * @param string $short 
     * @param string $long 
     * @param array $extraConfig additional options (default,valueFormat,options)
     * @return \boru\dhcli\Option 
     */
    public static function makeFlagOption($description,$short,$long,$extraConfig=[]) {
        return Option::flag($description,$short,$long,$extraConfig);
    }

    /**
     * Returns a new Option configured as a single value option
     * @param string $description 
     * @param string $short 
     * @param string $long 
     * @param bool $required 
     * @param array $extraConfig additional options (default,valueFormat,options)
     * @return \boru\dhcli\Option 
     */
    public static function makeOptionOption($description,$short,$long,$required=true,$extraConfig=[]) {
        return Option::option($description,$short,$long,$required,$extraConfig);
    }

    /**
     * Returns a new Option configured as a multi-value option
     * @param string $description 
     * @param string $short 
     * @param string $long 
     * @param bool $required 
     * @param array $extraConfig additional options (default,valueFormat,options)
     * @return \boru\dhcli\Option 
     */
    public static function makeMultiOption($description,$short,$long,$required=true,$extraConfig=[]) {
        return Option::multiOption($description,$short,$long,$required,$extraConfig);
    }

    /**
     * Construct the CLI utility. Accepts an array of options (name,description,script) and an optional Object that will be called if a command matches.
     */
    public function __construct($options=[],&$cmdObj=null) {
        $this->name = $this->getOption($options,"name",null);
        $this->description = $this->getOption($options,"description",null);
        $this->prescript = $this->getOption($options,"script",null);
        $prefix = $this->getOption($options,"prefix",false);
        dhGlobal::out()->setPrefix($prefix);
        if(($prefixFormat = $this->getOption($options,"prefix",false))!==false) {
            dhGlobal::out()->setPrefixFormat($prefixFormat);
        }
        if(!is_null($cmdObj)) {
            $this->cmdObj = &$cmdObj;
        }

    }
    public function classCommand($className) {
        if(class_exists($className)) {
            $this->command(Command::create($className::commandName(),$className::commandDescription(),$className::commandOptions(),['callback'=>function($result) use ($className) {
                $instance = new $className($result);
            }]));
        }
    }
    public function command(Command $command) {
        $this->commands[$command->name()] = $command;
        if($command->isDefaultCommand) {
            $this->setDefaultCommand($this->commands[$command->name()]);
        }
    }
    public function createOptFromArray($arrayOptions=[]) {
        $opt = new Option($arrayOptions);
        return $opt;
    }
    public function addCommandOpt($command,$opts) {
        if(is_object($opts)) {
            $this->commands[$command]->addOpt($opts);
        } elseif(is_array($opts)) {
            foreach($opts as $opt) {
                if(is_object($opt)) {
                    $this->commands[$command]->addOpt($opt);
                }
            }
        }
    }
    public function addCommandOptFromArray($command,$arrayOptions=[]) {
        if(is_array($arrayOptions)) {
            $opt = $this->createOptFromArray($arrayOptions);
            if($opt->name != "") {
                $this->addCommandOpt($command,$opt);
            }
        }
    }
    public function commandStandard($command="",$description="",$opts="") {
        $this->commands[$command] = new Command(["command"=>$command,"description"=>$description]);
        if(!empty($opts)) {
            $this->addCommandOpt($command,$opts);
        }
        return $this->commands[$command];
    }

    public function process($input,$separator=" ") {
        $args = $input;
        if(!is_array($input)) {
            $args = explode($separator,$input);
        }
        //print_r($args);
        if(empty($args) && $this->defaultCommand !== false) {
            $this->command = $this->defaultCommand;
        } else {
            if(empty($args) || $args[0] == "-h" || $args[0] == "--help" || $args[0] == "help") {
                $this->printBanner();
                $cmdFilter = "";
                if(isset($args[1])) {
                    $cmdFilter = $args[1];
                }
                $this->help($cmdFilter);
                $this->setStatus(-1);
                return -1;
            }
            if(!isset($this->commands[$args[0]])) {
                $this->printBanner();
                dhGlobal::outLine("Unknown command: {$args[0]}");
                $this->setStatus(-1);
                return -1;
            }
            $this->command = $this->commands[$args[0]];
            array_shift($args);
        }
        $dhargs = new Args($args);
        $worked = $this->command->process($dhargs);
        if($this->command->getStatus() === true) {
            if(!is_null($this->cmdObj)) {
                if(method_exists($this->cmdObj,"commandRun")) {
                    $this->cmdObj->commandRun($this->command->getResult());
                    $worked = true;
                }
            }
            $this->setStatus(1);
            if(is_null($worked)) {
                dhGlobal::info($this->command->name(),"processed but had no execution");
            }
            return 1;
        }
        $this->setStatus(0);
        $this->printErrors();
        return 0;
    }
    public function help($commandFilter="") {
        dhGlobal::outLine("Syntax: {$this->prescript}<command> [options]");
        dhGlobal::outLine("");
        if(!empty($commandFilter) && isset($this->commands[$commandFilter])) {
            $this->commands[$commandFilter]->help($this->prescript);
        } else {
            dhGlobal::outLine("Available commands:");
            $first=true;
            foreach($this->commands as $cmdName=>$cmd) {
                dhGlobal::outLine(" ",$cmdName,"-",$cmd->description);
            }
            dhGlobal::outLine("");
            dhGlobal::outLine("use 'help <command>' for more information");
        }
    }
    public function getCommandHelp($command="") {
        if(!empty($command)) {
            if(!isset($this->commands[$command])) {
                return false;
            }

        }
    }
    public function getResult() {
        if($this->getStatus()>=1) {
            $arr = [];
            $arr["command"] = $this->command->command;
            $arr["options"] = $this->command->getOptionValues();
            $arr["args"]["all"] = $this->command->getArgs(true);
            $arr["args"]["unused"] = $this->command->getArgs(false);
            $arr["callbackUsed"] = $this->command->callbackUsed;
            return $arr;
        }
        return false;
    }
    public function printErrors() {
        if($this->getStatus() === -1) {
            return -1;
        } elseif($this->getStatus() === 0) {
            $this->printBanner();
            $this->command->getErrorOutput($this->prescript);
            return 1;
        }
        return 0;
    }
    public function printName() {
        dhGlobal::outLine($this->name);
    }
    public function printDesc() {
        dhGlobal::outLine($this->description);
    }
    public function printBanner() {
        dhGlobal::outLine("");
        $this->printName();
        $this->printDesc();
        dhGlobal::outLine("");
    }
    public function resetOutput() {
        //$this->dhoutput = dhGlobal::out();
    }

    /**
     * Get the value of command
     */ 
    public function getCommand()
    {
        return $this->command;
    }

    /**
     * Set the value of command
     *
     * @return  self
     */ 
    public function setCommand($command)
    {
        $this->command = $command;

        return $this;
    }


    /**
     * Get status of the process execution
     */ 
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set status of the process execution
     *
     * @return  self
     */ 
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get the value of defaultCommand
     */ 
    public function getDefaultCommand()
    {
        return $this->defaultCommand;
    }

    /**
     * Set the value of defaultCommand
     *
     * @return  self
     */ 
    public function setDefaultCommand(&$defaultCommand)
    {
        $this->defaultCommand = &$defaultCommand;

        return $this;
    }

    public static function init($name,$description=null,$classCommands=[],$args=null,$script=null) {
        if(is_null($args)) {
            global $argv;
            $args = $argv;
        }
        if(is_null($script)) {
            $sn = array_shift($args);
            $script = "php -f $sn ";
        }
        if(is_null($description)) {
            $description = "No description set";
        }
        static::$dhcli = new self(["name"=>$name,"description"=>$description,"script"=>$script]);
        foreach($classCommands as $className) {
            static::$dhcli->classCommand($className);
        }
        return static::$dhcli->process($args);
    }
}