<?php
namespace boru\dhcli;

use \boru\dhutils\dhGlobal;

class Option {
    public $name = "";
    private $short;
    private $long;
    private $parsedArg;
    private $description;
    private $needsValue = false;
    private $required = false;
    private $multiple = false;
    private $value;
    private $isSet;
    private $options;
    private $valueFormat = '/^(.)+$/'; //allows any input value
    private $invalidOptions = [];
    private $isPositional = false;


    public static function positional($name,$description,$required=true,$extraConfig=[]) {
        $opts = ["long"=>$name,"description"=>$description];
        $opts["required"] = $required;
        $opts["positional"]=true;
        if(!empty($extraConfig)) {
            foreach($extraConfig as $k=>$v) {
                $opts[$k] = $v;
            }
        }
        return new static($opts);
    }
    /**
     * Returns a new Option configured for flags (no value)
     * @param string $description description of the option
     * @param string $short short arg
     * @param string $long long arg name
     * @param array $extraConfig additional options (default,valueFormat,options)
     * @return static 
     */
    public static function flag($short,$long,$description,$extraConfig=[]) {
        $opts = ["short"=>$short,"long"=>$long,"description"=>$description];
        $opts["needsValue"] = false;
        $opts["required"] = false;
        $opts["multiple"] = false;
        if(!empty($extraConfig)) {
            foreach($extraConfig as $k=>$v) {
                $opts[$k] = $v;
            }
        }
        return new static($opts);
    }

    /**
     * Returns a new Option configured as a single value option
     * @param string $description description of the option
     * @param string $short short arg
     * @param string $long long arg name
     * @param bool $required
     * @param array $extraConfig additional options (default,valueFormat,options)
     * @return static 
     */
    public static function option($short,$long,$description,$required=true,$extraConfig=[]) {
        $opts = ["short"=>$short,"long"=>$long,"description"=>$description];
        $opts["needsValue"] = true;
        $opts["required"] = $required;
        $opts["multiple"] = false;
        if(!empty($extraConfig)) {
            foreach($extraConfig as $k=>$v) {
                $opts[$k] = $v;
            }
        }
        return new static($opts);
    }

    /**
     * Returns a new Option configured as a multi-value option
     * @param string $description description of the option
     * @param string $short short arg
     * @param string $long long arg name
     * @param bool $required
     * @param array $extraConfig additional options (default,valueFormat,options)
     * @return static 
     */
    public static function multiOption($short,$long,$description,$required=true,$extraConfig=[]) {
        $opts = ["short"=>$short,"long"=>$long,"description"=>$description];
        $opts["needsValue"] = true;
        $opts["required"] = $required;
        $opts["multiple"] = true;
        if(!empty($extraConfig)) {
            foreach($extraConfig as $k=>$v) {
                $opts[$k] = $v;
            }
        }
        return new static($opts);
    }

    public function __construct($arrayOptions=[]) {
        if(!empty($arrayOptions) && is_array($arrayOptions)) {
            if(isset($arrayOptions["name"])) {
               $this->name = $arrayOptions["name"];
            }
            if(isset($arrayOptions["short"])) {
                $this->setShort($arrayOptions["short"]);
            }
            if(isset($arrayOptions["long"])) {
                $this->setLong($arrayOptions["long"]);
            }
            if(isset($arrayOptions["description"])) {
                $this->setDescription($arrayOptions["description"]);
            }
            if(isset($arrayOptions["required"])) {
                $this->setRequired($arrayOptions["required"]);
            }
            if(isset($arrayOptions["multiple"])) {
                $this->setMultiple($arrayOptions["multiple"]);
            }
            if(isset($arrayOptions["needsValue"])) {
                $this->setNeedsValue($arrayOptions["needsValue"]);
            }
            if(isset($arrayOptions["options"])) {
                $this->setOptions($arrayOptions["options"]);
            }
            if(isset($arrayOptions["default"])) {
                $this->setValue($arrayOptions["default"]);
            }
            if(isset($arrayOptions["valueFormat"])) {
                $this->setValueFormat($arrayOptions["valueFormat"]);
            }
            if(isset($arrayOptions["positional"])) {
                $this->setIsPositional($arrayOptions["positional"]);
            }
        }
    }

    public function getInputShort() {
        if(!$this->isShort()) {
            return false;
        }
        if($this->needsValue()) {
            return "-".$this->getShort()."<value>";
        } else {
            return "-".$this->getShort();
        }
    }
    public function getInputLong() {
        if(!$this->isLong()) {
            return false;
        }
        if($this->needsValue()) {
            return "--".$this->getLong()."=<value>";
        } else {
            return "--".$this->getLong();
        }
    }

    //Boolean checks
    public function isShort() {
        if($this->isPositional) {
            return false;
        }
        if(!empty($this->short) && !is_null($this->short)) {
            return true;
        }
        return false;
    }
    public function isLong() {
        if($this->isPositional) {
            return false;
        }
        if(!empty($this->long) && !is_null($this->long)) {
            return true;
        }
        return false;
    }
    public function isPositional() {
        return $this->isPositional;
    }
    public function isRequired() {
        return $this->required ? true : false;
    }
    public function isMultiple() {
        return $this->multiple ? true : false;
    }
    public function needsValue() {
        return $this->needsValue ? true : false;
    }
    public function hasValue() {
        return !is_null($this->value);
    }
    public function isValidOption($value) {
        if(empty($this->options)) {
            if(empty($this->valueFormat)) {
                return true;
            }
            if(preg_match($this->valueFormat,$value) === 1) {
                return true;
            } else {
                $this->invalidOptions[] = array("value"=>$value,"reason"=>"invalid format, must match {$this->valueFormat} ".preg_match($this->valueFormat,$value));
                return false;
            }
        }
        if(in_array($value,$this->options)) {
            return true;
        } else {
            $this->invalidOptions[] = array("value"=>$value,"reason"=>"must be one of [".implode(', ',$this->options)."])");
            return false;
        }
        return in_array($value,$this->options);
    }

    //Getters and Setters
    public function getName() {
        if(empty($this->name)) {
            if(($long = $this->getLong()) !== false) {
                return $long;
            }
            return $this->getShort();
        }
        return $this->name;
    }
    public function getShort() {
        return !empty($this->short) ? $this->short : false;
    }
    public function setShort($short) {
        $this->short = $short;
        return $this;
    }
    public function getLong() {
        return !empty($this->long) ? $this->long : false;
    }
    public function setLong($long) {
        $this->long = $long;
        return $this;
    }
    public function getDescription() {
        return $this->description;
    }
    public function setDescription($description) {
        $this->description=$description;
        return $this;
    }
    public function setRequired($req) {
        $this->required = $req;
        return $this;
    }
    public function setNeedsValue($nv) {
        $this->needsValue = $nv;
        return $this;
    }
    public function getValue() {
        return $this->value;
    }
    public function setValue($value) {
        if($this->isValidOption($value)) {
            if($this->isMultiple()) {
                if(!is_array($this->value)) {
                    if(!is_null($this->value)) {
                        $this->value = [$this->value,$value];
                    } else {
                        $this->value = $value;
                    }
                } else {
                    $this->value[] = $value;
                }
            } else {
                $this->value = $value;
            }
            return $this;
        }
        //throw here?
        return false;
    }
    public function getOptions() {
        return $this->options;
    }
    public function setOptions($options) {
        $this->options = $options;
        return $this;
    }


    /**
     * Get the value of multiple
     */ 
    public function getMultiple()
    {
        return $this->multiple;
    }

    /**
     * Set the value of multiple
     *
     * @return  self
     */ 
    public function setMultiple($multiple)
    {
        $this->multiple = $multiple;

        return $this;
    }

    /**
     * Get the value of valueFormat
     */ 
    public function getValueFormat()
    {
        return $this->valueFormat;
    }

    /**
     * Set the value of valueFormat
     *
     * @return  self
     */ 
    public function setValueFormat($valueFormat)
    {
        $this->valueFormat = $valueFormat;

        return $this;
    }

    /**
     * Get the value of invalidOptions
     */ 
    public function getInvalidOptions()
    {
        return $this->invalidOptions;
    }

    /**
     * Set the value of invalidOptions
     *
     * @return  self
     */ 
    public function setInvalidOptions($invalidOptions)
    {
        $this->invalidOptions = $invalidOptions;

        return $this;
    }

    /**
     * Get the value of parsedArg
     */ 
    public function getParsedArg()
    {
        return $this->parsedArg;
    }

    /**
     * Set the value of parsedArg
     *
     * @return  self
     */ 
    public function setParsedArg($parsedArg)
    {
        if(!is_null($this->parsedArg)) {
            if(!is_array($this->parsedArg)) {
                $this->parsedArg = [$this->parsedArg];                
            }
            $this->parsedArg[] = $parsedArg;
        } else {
            $this->parsedArg = $parsedArg;
        }
        return $this;
    }

    /**
	 * Get the value of isPositional
     * @return  mixed
     */
    public function getIsPositional() {
        return $this->isPositional;
    }

    /**
     * Set the value of isPositional
     * @param   mixed  $isPositional  
     * @return  self
	 */
    public function setIsPositional($isPositional) {
        $this->isPositional = $isPositional;
        return $this;
    }
}