<?php
namespace boru\dhdb\core;

use boru\dhdb\Query;
use boru\dhdb\contract\ResultInterface;

/**
 * Service that executes operations using a dhDB-like facade.
 * PHP 5.6 compatible.
 */
class DbRunner {

    /** @var object */
    protected $db;

    public function __construct($db) {
        $this->db = $db;
    }

    /**
     * @param string $sql
     * @param array $params
     * @return Statement|bool
     */
    public function prepare($sql, $params = []) {
        if (!$this->db->connect()) {
            return false; // connect() handles legacy vs exception mode
        }
        return new Statement($this->db, $sql, $params);
    }

    /**
     * @param string|Query $sql
     * @param array $params
     * @param array $options
     * @return ResultInterface|bool
     */
    public function query($sql, $params = [], $options = []) {
        if (!$this->db->connect()) {
            return false;
        }

        if ($sql instanceof Query) {
            $sql = $this->db->fromQuery($sql); // keep your existing behavior
            if(!$sql) {
                return false;
            }
        }

        $stmt = $this->prepare($sql, $params);
        if (!$stmt) {
            return false;
        }
        return $stmt->run();
    }

    /**
     * @param string|Query $sql
     * @param array $params
     * @param array $options
     * @return ResultInterface|bool
     */
    public function run($sql, $params = [], $options = []) {
        return $this->query($sql, $params, $options);
    }

    /**
     * @param Statement|ResultInterface $handler
     * @param int $mode
     * @param int $cursorOrientation
     * @param int $cursorOffset
     * @return RowInterface|false
     */
    public function next($handler = null, $mode = \PDO::FETCH_ASSOC, $cursorOrientation = \PDO::FETCH_ORI_NEXT, $cursorOffset = 0) {
        if (is_object($handler)) {
            return $handler->next($mode, $cursorOrientation, $cursorOffset);
        }
        return false;
    }

    /**
     * @param Statement|ResultInterface $handler
     * @param bool $assoc
     * @param bool $object
     * @return RowInterface|false
     */
    public function nextRow($handler = null, $assoc = true, $object = true) {
        if (is_object($handler)) {
            return $handler->nextRow($assoc, $object);
        }
        return false;
    }

    /**
     * @param string $name
     * @return mixed|false
     */
    public function lastInsertId($name = null) {
        if (!$this->db->connect()) {
            return false;
        }
        return $this->db->pdo()->lastInsertId($name);
    }
}
