<?php
namespace boru\dhdb;

use boru\dhdb\core\DbConnector;
use boru\dhdb\core\Debugger;
use boru\dhdb\core\query\QueryParser;
use boru\dhdb\core\Statement;
use boru\dhdb\core\Error;
use boru\dhdb\Query;
use boru\dhdb\traits\DhDbConnectionTrait;
use boru\dhdb\traits\DhDbErrorTrait;
use boru\dhdb\traits\DhDbDebugTrait;


class dhDB {

    // ---------------------------------------------------------------------
    // Connection lifecycle + PDO delegation
    // ---------------------------------------------------------------------
    use DhDbConnectionTrait;

    // ---------------------------------------------------------------------
    // Error state
    // ---------------------------------------------------------------------
    use DhDbErrorTrait;

    // ---------------------------------------------------------------------
    // Debugger proxies
    // ---------------------------------------------------------------------
    use DhDbDebugTrait;

    /** @var \PDO|null */
    protected $pdo = null;

    /** @var array */
    protected $config = [];

    /** @var Error|null */
    protected $error = null;

    /** @var bool */
    protected $throwExceptions = false;

    /** @var bool */
    protected $legacyMode = true;

    /** @var Debugger|null */
    private $debugger = null;

    /** @var self|null */
    private static $instance = null;

    /** @var \boru\dhdb\core\DbRunner|null */
    protected $runner = null;

    public function runner() {
        if (!$this->runner) {
            $this->runner = new \boru\dhdb\core\DbRunner($this);
        }
        return $this->runner;
    }

    public function setRunner($runner) {
        $this->runner = $runner; return $this;
    }


    // ---------------------------------------------------------------------
    // Static factories / adapters
    // ---------------------------------------------------------------------

    public static function instance($config = []) {
        if (is_null(static::$instance) || !empty($config)) {
            static::$instance = new static($config);
        }
        return static::$instance;
    }

    public static function fromVtigerConfig($configIncPhpFile, $static = true, $options = []) {
        $connection = DbConnector::fromVtigerConfig($configIncPhpFile);
        return new static($connection, $static);
    }

    // ---------------------------------------------------------------------
    // Construction / config
    // ---------------------------------------------------------------------

    /**
     * @param array|\PDO $options
     * @param bool $static
     */
    public function __construct($options = [], $static = true) {
        if ($options instanceof \PDO) {
            $this->pdo = $options;
            return;
        }

        // Maintain backward compatibility with the old config format:
        if (isset($options["config"])) {
            $this->setConfig($options["config"]);
        } else {
            $this->setConfig($options);
        }

        $debuggerOpts = [];
        if (isset($_SERVER["HTTP_USER_AGENT"])) {
            $debuggerOpts["isBrowser"] = true;
        }
        $this->debugger = new Debugger($debuggerOpts);

        if ($static) {
            static::$instance = $this;
        }
    }

    /** PDO/Wrapper methods (old 'extends Mysql' syntax) */
    public function setConfig($config) {
        $this->config = $config;
    }

    public function getConfig() {
        return $this->config;
    }

    // ---------------------------------------------------------------------
    // Query execution / statement helpers
    // ---------------------------------------------------------------------

    /** @return \boru\dhdb\core\Statement */
    public function prepare($sql, $params = []) {
        return $this->runner()->prepare($sql, $params);
    }

    /** @return \boru\dhdb\core\Statement */
    public function query($sql, $params = [], $options = []) {
        return $this->runner()->query($sql, $params, $options);
    }

    /** @return \boru\dhdb\core\Statement */
    public function run($sql, $params = [], $options = []) {
        return $this->runner()->run($sql, $params, $options);
    }

    public function next($handler = null, $mode = \PDO::FETCH_ASSOC, $cursorOrientation = \PDO::FETCH_ORI_NEXT, $cursorOffset = 0) {
        return $this->runner()->next($handler, $mode, $cursorOrientation, $cursorOffset);
    }

    public function nextRow($handler = null, $assoc = true, $object = true) {
        return $this->runner()->nextRow($handler, $assoc, $object);
    }

    public function lastInsertId($name = null) {
        return $this->runner()->lastInsertId($name);
    }


    public function getDumpCommand($db = null, $table = null) {
        $cmd = "mysqldump";
        $cmd .= " --host=" . $this->config["dbhost"];
        $cmd .= " --user=" . $this->config["dbuser"];
        $cmd .= " --password=" . $this->config["dbpass"];
        $cmd .= " ";
        if (!is_null($db)) {
            $cmd .= $db . " ";
            if (!is_null($table)) {
                $cmd .= "'" . $table . "' ";
            }
        }
        return $cmd;
    }

    // ---------------------------------------------------------------------
    // Query-builder helpers (legacy Mysql surface area)
    // ---------------------------------------------------------------------

    public function newQuery() {
        return new Query();
    }

    public function newParser() {
        return new QueryParser();
    }

    public function makeQuery() {
        return new Query();
    }

    // ---------------------------------------------------------------------
    // Static utilities
    // ---------------------------------------------------------------------

    public static function interpolateQuery($query = "", $params = []) {
        if (empty($params)) {
            return $query;
        }
        $keys = array();
        $values = $params;

        // build a regular expression for each parameter
        foreach ($params as $key => $value) {
            if (is_string($key)) {
                $keys[] = '/:' . $key . '/';
            } else {
                $keys[] = '/[?]/';
            }

            if (is_array($value))
                $values[$key] = implode(',', $value);

            if (is_null($value))
                $values[$key] = 'NULL';
        }

        // Walk the array to see if we can add single-quotes to strings
        array_walk($values, function (&$v) {
            if (!is_numeric($v) && $v != "NULL") {
                $v = "\"" . $v . "\"";
            }
        });

        $query = preg_replace($keys, $values, $query, 1, $count);

        return $query;
    }

    /**
     * Generate an array of [?,?,?,...]
     */
    public static function generateQs($array, $glue = ",") {
        $qarr = array_fill(0, count($array), "?");
        return implode($glue, $qarr);
    }

    public static function parse($queryString, $opts = []) {
        $parser = new Parser($queryString);
        $options = [];
        if (isset($opts["params"])) {
            $options["constantAsQuestionMark"] = true;
        }
        if (isset($opts["part"])) {
            $options["part"] = $opts["part"];
        }
        if (isset($opts["array"])) {
            return $parser->toArray($options);
        }
        $data = $parser->toSql($options);
        if (isset($opts["details"])) {
            $data["details"] = $parser->details($options);
        }
        return $data;
    }


    /**
     * Generate a Schema for a table. Optionally provide an existing Schema for comparison
     * 
     * @param string $table The tablename to parse
     * @param null|Schema $schema optional schema to compare
     * @return \boru\dhdb\Schema
     */
    public function getTable($table,$schema=null) {
        return $this->table($table,$schema);
    }
    public function table($tableName,$schema=null) {
		$meta = [];
		if(is_null($schema)) {
			$schema = new Schema($tableName);
		}
		$sth = $this->run("SHOW TABLE STATUS LIKE ?",[$tableName]);
		while($row = $sth->next()) {
			$meta["engine"] = $row->get("Engine");
			$meta["collate"] = $row->get("Collation");
			if(!empty($row->get("Auto_increment"))) {
				$meta["auto_increment"] = $row->get("Auto_increment");
			}
			if(!empty($row->get("Comment"))) {
				$meta["comment"]= $row->get("Comment");
			}
		}
		foreach($meta as $k=>$v) {
			$schema->meta($k,$v);
		}
		$sth = $this->run("SHOW FULL FIELDS FROM `".$tableName."`");
		while($row = $sth->next()) {
			$extra = ["null"=>true];
			if(strtolower($row->get("Null")) == "no") {
				$extra["null"] = false;
			}
			if(!empty($row->get("Collation"))) {
				$extra["collate"] = $row->get("Collation");
			}
			if(!empty($row->get("Default"))) {
				$extra["default"] = $row->get("Default");
			}
			if(!empty($row->get("Extra")) && $row->get("Extra") == "auto_increment") {
				$extra["auto_increment"] = true;
			}
			$schema->add($row->get("Field"),$row->get("Type"),$extra);
		}
		$sth = $this->run("SHOW INDEX FROM `".$tableName."`");
		$indexArr = [];
		while($row = $sth->next()) {
			if(!isset($indexArr[$row->get("Key_name")])) {
				if($row->get("Key_name") == "PRIMARY" && $row->get("Non_unique")<=0) {
					$type = "primary";
				} elseif($row->get("Index_type") == "FULLTEXT") {
					$type="fulltext";
				} elseif($row->get("Non_unique") <= 0) {
					$type = "unique";
				} else {
					$type = "key";
				}
				$indexArr[$row->get("Key_name")] = [
					"name"=>$row->get("Key_name"),
					"type"=>$type,
					"extra"=>[]
				];
			}
			$indexArr[$row->get("Key_name")]["extra"][] = $row->get("Column_name").($row->get("Cardinality") == "D" ? " DESC" : "");
		}
		foreach($indexArr as $name=>$index) {
			$schema->addIndex($name,$index["extra"],$index["type"]);
		}
		return $schema;
	}
}
