<?php
namespace boru\dhdb;

use boru\dhdb\parser\expr\Expression;
use boru\dhdb\parser\expr\Params;
use boru\dhdb\parser\SQLParserWrapper;

class Parser implements \JsonSerializable {

    private $query = "";
    private $options = [];
    private $wrapper;

    private $arrayParsed = [];
    private $expressions = [];

    public function __construct($query=null,$options=[]) {
        $this->wrapper = new SQLParserWrapper();
        if(!empty($options)) {
            $this->setOptions($options);
        }
        if(!is_null($query)) {
            $this->setQuery($query);
            $this->parse();
        }
    }

    public function setQuery($query) {
        $this->query = $query;
        $this->parse();
        return $this;
    }
    public function setOptions($options=[],$merge=true) {
        if($merge) {
            $this->options = array_merge($this->options,$options);
        } else {
            $this->options = $options;
        }
        return $this;
    }

    public function getExpressions($type=null) {
        if(!is_null($type)) {
            $type = strtoupper($type);
            if(isset($this->expressions[$type])) {
                return $this->expressions[$type];
            }
        }
        return $this->expressions;
    }

    public function parse($sql=null,$options=[]) {
        if(!is_null($sql)) {
            $this->setQuery($sql);
        }
        if(!empty($options)) {
            $this->setOptions($options);
        }
        $this->wrapper->parse($this->query, $this->options);
        $parsed = $this->wrapper->get();
        foreach($parsed as $key=>$value) {
            $key = strtoupper($key);
            $this->expressions[$key] = [];
            foreach($value as $expr) {
                $this->expressions[$key][] = Expression::fromArray($expr);
            }
        }
    }

    public function get($whatPart,$asSql=false) {
        $whatPart = strtoupper($whatPart);
        if(isset($this->expressions[$whatPart])) {
            if($asSql) {
                $sql = [];
                foreach($this->expressions[$whatPart] as $expr) {
                    $sql[] = $expr->toSql();
                }
                return implode(" ",$sql);
            } else {
                return $this->expressions[$whatPart];
            }
        }
        return false;
    }

    public function toArray($options=[]) {
        $sqlArray = [];
        $part = isset($options["part"]) ? $options["part"] : null;
        if(is_null($part)) {
            foreach($this->expressions as $key=>$value) {
                $sqlArray[$key] = [];
                foreach($value as $expr) {
                    $sqlArray[$key][] = $expr->toArray($options);
                }
            }
        } else {
            $part = strtoupper($part);
            if(isset($this->expressions[$part])) {
                foreach($this->expressions[$part] as $expr) {
                    $sqlArray[] = $expr->toArray($options);
                }
            }
        }
        return $sqlArray;
    }
    public function jsonSerialize() {
        return $this->toArray();
    }
    public function __toString() {
        return json_encode($this->toArray(),JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
    }
    public function toString($options=[]) {
        $sqlArray = $this->toArray($options);
        return json_encode($sqlArray,JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
    }
    public function toSql($options=[]) {
        $params = new Params();
        $sqlArray = [];
        $part = isset($options["part"]) ? $options["part"] : null;
        if(is_null($part)) {
            foreach($this->expressions as $key=>$value) {
                $sqlArray[$key] = [];
                foreach($value as $expr) {
                    if($expr instanceof Expression) {
                        $sqlArray[$key][] = $expr->toSql($options,$params);
                    } else {
                        $sqlArray[$key][] = $expr;
                    }
                }
            }
            foreach($sqlArray as $key=>$value) {
                $sqlArray[$key] = $this->keyToSql($key)." ".implode(" ",$value);
            }
        } else {
            $part = strtoupper($part);
            if(isset($this->expressions[$part])) {
                foreach($this->expressions[$part] as $expr) {
                    if($expr instanceof Expression) {
                        $sqlArray[] = $expr->toSql($options,$params);
                    } else {
                        $sqlArray[] = $expr;
                    }
                }
            }
        }
        return ["sql"=>implode(" ",$sqlArray),"params"=>$params->get()];
    }

    private function keyToSql($key) {
        $key = strtoupper($key);
        if($key == "ORDER") {
            return "ORDER BY";
        }
        return $key;
    }
}