<?php
namespace boru\dhdb\core;

use boru\dhutils\tools\DebugTrace;

class Debugger {
    /**
     * True if called from a browser
     * @var bool
     */
    private $isBrowser = false;

    /**
     * Whether to interpolate the query or print sql/params separately
     */
    private $useInterpolate = true;

    /**
     * Whether to save the query to the database
     * @var bool
     */
    private $saveToTable = false;

    /**
     * Whether to print the query to the browser/screen
     * @var bool
     */
    private $printDebug = false;

    /**
     * Whether to include the trace in the output
     * @var bool
     */
    private $includeTrace = false;

    /**
     * The number of seconds to consider a query slow
     * @var float
     */
    private $slowLimit = 1;

    /**
     * Whether to log slow queries to the database
     * @var bool
     */
    private $logSlow = false;

    /**
     * Whether to log slow queries to the database
     * @var bool
     */
    private $printSlow = false;

    /**
     * The reference to use for the query
     * @var string
     */
    private $reference = "";

    private $callback = null;

    public function __construct($options=[]) {
        if(!empty($options)) {
            $this->setOptions($options);
        }
    }
    public function setOptions($options=[]) {
        foreach($options as $key=>$value) {
            if(method_exists($this,$key)) {
                $this->$key($value);
            }
        }
    }
    public function getOptions() {
        return [
            "isBrowser" => $this->isBrowser,
            "useInterpolate" => $this->useInterpolate,
            "saveToTable" => $this->saveToTable,
            "printDebug" => $this->printDebug,
            "includeTrace" => $this->includeTrace,
            "slowLimit" => $this->slowLimit,
            "logSlow" => $this->logSlow,
            "printSlow" => $this->printSlow,
            "reference" => $this->reference,
            "callback" => $this->callback,
        ];
    }

    public function isBrowser($isBrowser=null) {
        if(is_null($isBrowser)) return $this->isBrowser;
        $this->isBrowser = $isBrowser;
    }
    public function useInterpolate($useInterpolate=null) {
        if(is_null($useInterpolate)) return $this->useInterpolate;
        $this->useInterpolate = $useInterpolate;
    }
    public function saveToTable($saveToTable=null) {
        if(is_null($saveToTable)) return $this->saveToTable;
        $this->saveToTable = $saveToTable;
    }
    public function printDebug($printDebug=null) {
        if(is_null($printDebug)) return $this->printDebug;
        $this->printDebug = $printDebug;
    }
    public function includeTrace($includeTrace=null) {
        if(is_null($includeTrace)) return $this->includeTrace;
        $this->includeTrace = $includeTrace;
    }
    public function slowLimit($slowLimit=null) {
        if(is_null($slowLimit)) return $this->slowLimit;
        $this->slowLimit = $slowLimit;
    }
    public function logSlow($logSlow=null) {
        if(is_null($logSlow)) return $this->logSlow;
        $this->logSlow = $logSlow;
    }
    public function printSlow($printSlow=null) {
        if(is_null($printSlow)) return $this->printSlow;
        $this->printSlow = $printSlow;
    }
    public function reference($reference=null) {
        if(is_null($reference)) return $this->reference;
        $this->reference = $reference;
    }
    public function callback($callback=null) {
        if(is_null($callback)) return $this->callback;
        $this->callback = $callback;
    }

    /**
     * 
     * @param Statement $statement 
     * @return void 
     */
    public function log($statement) {
        if(is_callable($this->callback)) {
            call_user_func($this->callback,$statement);
        }
        if($this->saveToTable || ($this->logSlow && $this->isSlow($statement))) {
            if($this->includeTrace && empty($statement->getTrace())) {
                $statement->setTrace(DebugTrace::getTraceInfo(3));
            }
            //$this->logToTable($statement);
        }
        if($this->printDebug || ($this->printSlow && $this->isSlow($statement))) {
            if($this->includeTrace && empty($statement->getTrace())) {
                $statement->setTrace(DebugTrace::getTraceInfo(3));
            }
            if($this->isBrowser) {
                DebugPrinter::printHtml($statement,$this->isSlow($statement),$this->useInterpolate);
            } else {
                DebugPrinter::printText($statement,$this->isSlow($statement),$this->useInterpolate);
            }
        }
    }

    private function isSlow($statement) {
        return $statement->getExecTime() >= $this->slowLimit;
    }
}