<?php
namespace boru\dhdb\core;

use boru\dhdb\core\query\QueryParser;
use boru\dhdb\dhDB;
use boru\dhdb\Query;
use boru\dhdb\Schema;
use boru\dhutils\dhGlobal;

class Mysql {
    /** @var \PDO */
    private $pdo;

    private $config = [];
    private $connected = false;

    public function setConfig($config) {
        $this->config = $config;
    }

    public function connect($config=null) {
        if($this->connected) {
            return true;
        }
        if(is_null($config)) {
            $config = $this->config;
        }
        try {
            if(!isset($this->config["dbtype"])) {
                $this->config["dbtype"] = "mysql";
            }
            $dsn=$this->config["dbtype"].':host='.$this->config["dbhost"].";port=".$this->config["dbport"].";dbname=".$this->config["dbname"];
            $user=$this->config["dbuser"];
            $passwd=$this->config["dbpass"];
            $options = array(
                \PDO::ATTR_PERSISTENT => true, 
                \PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION
            );
            $this->pdo = new \PDO($dsn, $user, $passwd, $options);
            $this->connected = true;
            return true;
        } catch (\PDOException $e) {
            return false;
        }
    }
    public function pdo() {
        return $this->pdo;
    }

    public function prepare($sql,$params=[]) {
        if(!$this->connect()) {
            throw new \Exception("Could not connect to database");
        }
        return new Statement($this,$sql,$params);
    }
    public function getDumpCommand($db=null,$table=null) {
        $cmd = "mysqldump";
        $cmd.=" --host=".$this->config["dbhost"];
        $cmd.=" --user=".$this->config["dbuser"];
        $cmd.=" --password=".$this->config["dbpass"];
        $cmd.=" ";
        if(!is_null($db)) {
            $cmd.=$db." ";
            if(!is_null($table)) {
                $cmd.="'".$table."' ";
            }
        }
        return $cmd;
    }
    
    /**
     * Wrapper for ResultInterface->next() so no need to check if result is proper or not before calling next()
     * 
     * @param StatementInterface|null $handler 
     * @return bool|ResultInterface 
     */
    public function next($handler=null,$mode=\PDO::FETCH_ASSOC,$cursorOrientation = \PDO::FETCH_ORI_NEXT,$cursorOffset = 0) {
        if(is_object($handler)) {
            return $handler->next($mode,$cursorOrientation,$cursorOffset);
        }
        return false;
    }
    /**
     * Wrapper for ResultInterface->next() so no need to check if result is proper or not before calling next()
     * 
     * @param StatementInterface|null $handler 
     * @return bool|ResultInterface 
     */
    public function nextRow($handler=null,$assoc=true,$object=true) {
        if(is_object($handler)) {
            return $handler->nextRow($assoc,$object);
        }
        return false;
    }

    public function __call($method,$args) {
        if(!$this->connected) {
            return false;
        }
        return call_user_func_array( [$this->pdo,$method], $args);
    }


    //BC functions

    public function driver($driver=null) {
        return $this;
    }
    public function newQuery() {
		return new Query();
	}
	public function newParser() {
		return new QueryParser();
	}
    /**
     * Returns a new Query object for ORM style operations
     * 
     * @return Query
     */
    public function makeQuery() {
        return new Query();
    }

    /**
     * Returns the SQL for a query object. If Interpolate is true, returns the interpolated query (default, true)
     * 
     * @param Query $query the query object to parse
     * @param boolean $interpolate wether or not to interpolate the returned sql
     * @return string
     */
    public function fromQuery(Query $query,$interpolate=true) {
        $parser = new QueryParser($query);
		$parser->parse();
		$sql = $parser->getQueryString();
        if($interpolate) {
            return dhDB::interpolateQuery($sql,$query->values);
        }
        return $sql;
    }
    /**
     * Generate a Schema for a table. Optionally provide an existing Schema for comparison
     * 
     * @param string $table The tablename to parse
     * @param null|Schema $schema optional schema to compare
     * @return \boru\dhdb\Schema
     */
    public function getTable($table,$schema=null) {
        return $this->table($table,$schema);
    }
    public function table($tableName,$schema=null) {
		$meta = [];
		if(is_null($schema)) {
			$schema = new Schema($tableName);
		}
		$sth = $this->run("SHOW TABLE STATUS LIKE ?",[$tableName]);
		while($row = $sth->next()) {
			$meta["engine"] = $row->get("Engine");
			$meta["collate"] = $row->get("Collation");
			if(!empty($row->get("Auto_increment"))) {
				$meta["auto_increment"] = $row->get("Auto_increment");
			}
			if(!empty($row->get("Comment"))) {
				$meta["comment"]= $row->get("Comment");
			}
		}
		foreach($meta as $k=>$v) {
			$schema->meta($k,$v);
		}
		$sth = $this->run("SHOW FULL FIELDS FROM `".$tableName."`");
		while($row = $sth->next()) {
			$extra = ["null"=>true];
			if(strtolower($row->get("Null")) == "no") {
				$extra["null"] = false;
			}
			if(!empty($row->get("Collation"))) {
				$extra["collate"] = $row->get("Collation");
			}
			if(!empty($row->get("Default"))) {
				$extra["default"] = $row->get("Default");
			}
			if(!empty($row->get("Extra")) && $row->get("Extra") == "auto_increment") {
				$extra["auto_increment"] = true;
			}
			$schema->add($row->get("Field"),$row->get("Type"),$extra);
		}
		$sth = $this->run("SHOW INDEX FROM `".$tableName."`");
		$indexArr = [];
		while($row = $sth->next()) {
			if(!isset($indexArr[$row->get("Key_name")])) {
				if($row->get("Key_name") == "PRIMARY" && $row->get("Non_unique")<=0) {
					$type = "primary";
				} elseif($row->get("Index_type") == "FULLTEXT") {
					$type="fulltext";
				} elseif($row->get("Non_unique") <= 0) {
					$type = "unique";
				} else {
					$type = "key";
				}
				$indexArr[$row->get("Key_name")] = [
					"name"=>$row->get("Key_name"),
					"type"=>$type,
					"extra"=>[]
				];
			}
			$indexArr[$row->get("Key_name")]["extra"][] = $row->get("Column_name").($row->get("Cardinality") == "D" ? " DESC" : "");
		}
		foreach($indexArr as $name=>$index) {
			$schema->addIndex($name,$index["extra"],$index["type"]);
		}
		return $schema;
	}
}