<?php
namespace boru\dhdb\parser;

use boru\dhutils\dhGlobal;
use PHPSQLParser\exceptions\UnsupportedFeatureException;

class SQLParserWrapper implements \JsonSerializable {

    private $query = "";
    private $parsed = [];
    private $parser;
    private $creator;
    private $options = [];

    public function __construct($query=null,$options=[]) {
        $this->parser = new \PHPSQLParser\PHPSQLParser();
        $this->creator = new \PHPSQLParser\PHPSQLCreator();
        if(!empty($options)) {
            $this->setOptions($options);
        }
        if(!is_null($query)) {
            $this->setQuery($query);
            $this->parse();
        }
    }

    public function setQuery($query) {
        $this->query = $query;
        $this->parse();
        return $this;
    }
    public function setOptions($options=[],$merge=true) {
        if($merge) {
            $this->options = array_merge($this->options,$options);
        } else {
            $this->options = $options;
        }
        return $this;
    }

    public function parse($sql=null,$options=[]) {
        if(!is_null($sql)) {
            $this->setQuery($sql);
        }
        if(!empty($options)) {
            $this->setOptions($options);
        }
        $this->parser->parse($this->query, true);
        $this->parsed = $this->parser->parsed;
        return $this;
    }

    public function jsonSerialize() {
        return $this->parsed;
    }
    public function __toString() {
        return json_encode($this->parsed,JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
    }

    public function get($key=null,$default=null) {
        if(is_null($key)) {
            return $this->parsed;
        }
        if(isset($this->parsed[$key])) {
            return $this->parsed[$key];
        }
        return dhGlobal::getDot($this->parsed,$key,$default);
    }
    public function set($key,$value) {
        dhGlobal::dotAssign($this->parsed,$key,$value);
        return $this;
    }

    public function getQuery() {
        return $this->query;
    }
    public function getSelect($asSql=false) {
        $part = $this->get('SELECT',false);
        if($asSql) {
            return $this->toQuery(["SELECT"=>$part]);
        }
        return $part;
    }
    public function getFrom($asSql=false) {
        $part = $this->get('FROM',false);
        if($asSql) {
            return $this->toQuery(["FROM"=>$part]);
        }
        return $part;
    }
    public function getWhere($asSql=false) {
        $part = $this->get('WHERE',false);
        if($asSql) {
            return $this->toQuery(["WHERE"=>$part]);
        }
        return $part;
    }
    public function getGroupBy($asSql=false) {
        $part = $this->get('GROUP',false);
        if($asSql) {
            return $this->toQuery(["GROUP"=>$part]);
        }
        return $part;
    }
    public function getOrderBy($asSql=false) {
        $part = $this->get('ORDER',false);
        if($asSql) {
            return $this->toQuery(["ORDER"=>$part]);
        }
        return $part;
    }
    public function toQuery($partArray=null) {
        if(is_null($partArray)) {
            $partArray = $this->parsed;
        }
        $sql = "";
        try {
            $sql = $this->creator->create($partArray);
        } catch (UnsupportedFeatureException $e) {
            $partType = key($partArray);
            try {
                $sql = $this->partQuery($partType,$partArray);
            } catch (\Exception $e) {
                throw new \Exception("Unsupported: ".$e->getMessage());
            }
        }
        return $sql;
    }
    public function partQuery($partType,$partArray) {
        $partValue = $partArray[$partType];
        $partType = strtolower($partType);
        if($partType       == "select") {
            $builder = new \PHPSQLParser\builders\SelectStatementBuilder();
        } elseif($partType == "from") {
            $builder = new \PHPSQLParser\builders\FromBuilder();
        } elseif($partType == "where") {
            $builder = new \PHPSQLParser\builders\WhereBuilder();
        } elseif($partType == "group") {
            $builder = new \PHPSQLParser\builders\GroupByBuilder();
        } elseif($partType == "order") {
            $builder = new \PHPSQLParser\builders\OrderByBuilder();
        } elseif($partType == "limit") {
            $builder = new \PHPSQLParser\builders\LimitBuilder();
        } elseif($partType == "having") {
            $builder = new \PHPSQLParser\builders\HavingBuilder();
        } elseif($partType == "union") {
            $builder = new \PHPSQLParser\builders\UnionStatementBuilder();
        } elseif($partType == "union all") {
            $builder = new \PHPSQLParser\builders\UnionAllStatementBuilder();
        } else {
            throw new \Exception("Unknown part type: ".$partType);
        }
        return $builder->build($partValue);
    }
}