<?php
namespace boru\dhdb;

use \boru\dhutils\dhGlobal;
use \boru\dhutils\filesys\File;

class Schema implements \JsonSerializable {
    protected $tableName;
    protected $meta;
    protected $columns;
    protected $keys;
    protected $history = [];
    protected $version = 0;
    protected $versions = [];

    protected $changes = [];

    protected $file;
    

    public function __construct($tableName=null,$file=null) {
        if(!is_null($tableName)) {
            $this->setTableName($tableName);
        }
        if(!is_null($file)) {
            $this->setFile($file);
            if(!is_null($this->file)) {
                $this->load();
            }
        }
    }

    public function get($version=null) {
        $array["name"] = $this->getTableName();
        $array["meta"] = $this->getMeta($version);
        $array["columns"] = $this->getColumns($version);
        $array["keys"] = $this->getKeys($version);
        return $array;
    }
    public function getAll() {
        $array["tableName"] = $this->tableName;
        $array["meta"] = $this->meta;
        $array["columns"] = $this->columns;
        $array["keys"] = $this->keys;
        $array["history"] = $this->history;
        $array["versions"] = $this->versions;
        $array["changes"] = $this->changes;
        return $array;
    }

    public function save($makeVersion=true) {
        if(!empty($this->changes) && $makeVersion) {
            $this->version++;
            $versionString = $this->getVersionString();
            $this->meta("comment","dhDB::".$versionString);
            $this->history[$versionString] = $this->changes;
            $this->versions[$versionString]["columns"] = $this->columns;
            $this->versions[$versionString]["keys"] = $this->keys;
            $this->versions[$versionString]["meta"] = $this->meta;
            $this->changes = [];
        }
        if(!is_null($this->file)) {
            $this->file->write(json_encode($this->getAll(),JSON_PRETTY_PRINT));
        }
        return $this;
    }
    public function load($clean=true) {
        if(!is_null($this->file)) {
            $data = $this->file->content(["json"=>true]);
            if(is_array($data)) {
                if(!$clean) {
                    $this->meta = dhGlobal::getVal($data,"meta",null);
                    $this->columns = dhGlobal::getVal($data,"columns",null);
                    $this->keys = dhGlobal::getVal($data,"keys",null);
                }
                $this->history = dhGlobal::getVal($data,"history",null);
                $this->versions = dhGlobal::getVal($data,"versions",null);
                $this->version = 0;
                if(is_array($this->versions)) {
                    foreach($this->versions as $version=>$d) {
                        $version = intval(dhGlobal::trimString("v_",$version,dhGlobal::TRIM_START));
                        if($version>$this->version) {
                            $this->version = $version;
                        }
                    }
                }
            }
        }
        return $this;
    }

    public function getColumnHistory($column) {
        $columnHistory = [];
        if(!empty($this->history)) {
            foreach($this->history as $version=>$changes) {
                foreach($changes as $change) {
                    if($change["column"] == $column && strpos($change["action"],"Index") === false) {
                        $columnHistory[$version][] = $change;
                    }
                }
            }
        }
        return $columnHistory;
    }
    public function replayHistory($version=null) {
        if(!is_null($version)) {
            $version = $this->getVersionString($version);
            if(isset($this->history[$version])) {
                $this->replayHistoryVersion($version);
                return true;
            }
            return false;
        } else {
            if(!empty($this->history)) {
                $versions = array_keys($this->history);
                foreach($versions as $version) {
                    $this->replayHistoryVersion($version);
                }
                return true;
            }
            return false;
        }
        return false;
    }
    protected function replayHistoryVersion($version) {
        foreach($this->history[$version] as $entry) {
            $this->schemaColumnEntry($entry);
        }
    }


    public function version($version=null) {
        if(is_null($version)) {
            return $this->version;
        }
        if($version === true) {
            $this->version++;
        } else {
            $this->version = $version;
        }
        return $this;
    }

    public function add($col,$type=null,$extra=null) {
        return $this->schemaColumn("add",$col,$type,$extra);
    }
    public function remove($col) {
        return $this->schemaColumn("drop",$col);
    }
    public function change($col,$newcol,$type=null,$extra=null) {
        return $this->schemaColumn("change",[$col,$newcol],$type,$extra);
    }
    public function addIndex($name,$cols=[],$type="key") {
        return $this->schemaColumn("addIndex",$name,strtolower($type),$cols);
    }
    public function key($name,$cols=[],$type="key") {
        return $this->schemaColumn("addIndex",$name,strtolower($type),$cols);
    }
    public function index($name,$cols=[],$type="key") {
        return $this->schemaColumn("addIndex",$name,strtolower($type),$cols);
    }
    public function dropIndex($name) {
        return $this->schemaColumn("dropIndex",$name);
    }
    public function meta($key,$value=null,$version=null) {
        if($version == null) {
            $this->meta[$key] = ["key"=>$key,"value"=>$value];
        } else {
            $version = $this->getVersionString($version);
            $this->versions[$version]["meta"][$key] = ["key"=>$key,"value"=>$value];
        }
        
        return $this;
    }
    
    protected function parseColExtra($extra,$glue=" ") {
        $null = dhGlobal::getVal($extra,"null",false);
        $collate = dhGlobal::getVal($extra,"collate",false);
        $charset = dhGlobal::getVal($extra,"charset",false);
        $default = dhGlobal::getVal($extra,"default",false);
        $auto_incr = dhGlobal::getVal($extra,"auto_increment",false);
        if(!$charset && $collate !== false) {
            list($charset,$unused) = explode("_",$charset,2);
        }
        $parts = [];
        if($charset !== false) {
            $parts[] ="CHARACTER SET ".$charset;
        }
        if($collate !== false) {
            $parts[] ="COLLATE ".$collate;
        }
        if($null) {
            if($default === false) {
                $parts[] = "DEFAULT NULL";
            } else {
                $parts[] = 'DEFAULT "'.$default.'"';
            }
        } else {
            $parts[] = "NULL NULL";
            if($default !== false) {
                $parts[] = 'DEFAULT "'.$default.'"';
            }
        }
        if($auto_incr !== false) {
            $parts[] = "AUTO_INCREMENT";
        }
        return implode($glue,$parts);
    }

    public function schemaColumn($action="add",$col,$type=null,$extra=null) {
        $entry = ["action"=>$action,"column"=>$col,"type"=>$type, "extra"=>$extra];
        return $this->schemaColumnEntry($entry);
    }
    protected function schemaColumnEntry(array $entry) {
        if(!is_array($this->changes)) {
            $this->changes = [];
        }
        $changed = $this->processEntry($entry);
        if($changed) {
            $this->changes[] = $entry;
        }
        return $this;
    }
    protected function processEntry(array $entry) {
        $action = $entry["action"];
        $column  = $entry["column"];
        $type = $entry["type"];
        $extra = $entry["extra"];
        if($action == "add") {
            if(!isset($this->columns[$column])) {
                $this->columns[$column] = ["type"=>$type,"extra"=>$extra];
                return true;
            }
            return false;
        } elseif($action == "drop") {
            if(isset($this->columns[$column])) {
                unset($this->columns[$column]);
                return true;
            }
            return false;
        } elseif($action == "change") {
            $old = $column[0];
            $new = $column[1];
            if(isset($this->columns[$old])) {
                unset($this->columns[$old]);
                $this->columns[$new] = ["type"=>$type,"extra"=>$extra];
                return true;
            }
            if(!isset($this->columns[$new])) {
                $entry["action"] == "add";
                return $this->processEntry($entry);
            }
            return false;
        } elseif($action == "dropIndex") {
            if(isset($this->keys[$column])) {
                unset($this->keys[$column]);
                return true;
            }
            return false;
        } elseif($action == "addIndex") {
            if(!isset($this->keys[$column])) {
                $this->keys[$column] = ["type"=>$type,"columns"=>$extra];
                return true;
            }
            return false;
        }
    }

    public function toQuery($patch=false) {
        $query = new Query();
        $table = $this->getTableName();

        if($patch && $this->version>=1) {
            return $this->toPatchQuery();
        }

        $query->create($table);
        $after = null;
        foreach($this->getColumns($this->version) as $col=>$entry) {
            //if(!is_null($after)) {
                //$entry["extra"]["after"] = $after;
            //}
            if(isset($entry["extra"]["after"])) {
                unset($entry["extra"]["after"]);
            }
            $query->add($col,$entry["type"],$entry["extra"]);
            $after = $col;
        }
        foreach($this->getKeys($this->version) as $name=>$entry) {
            $query->addIndex($name,$entry["columns"],$entry["type"]);
        }
        foreach($this->getMeta($this->version) as $key=>$entry) {
            $noBlank = ["auto_increment","comment"];
            if(in_array($entry["key"],$noBlank) && (is_null($entry["value"]) || empty($entry["value"]))) {

            } else {
                $query->meta($entry["key"],$entry["value"]);
            }
        }
        return $query;
    }
    public function toPatchQuery($reorder=false) {
        if($this->version<=0) {
            return $this->toQuery(false);
        }

        $changes = 0;

        $savedCols = $this->getColumns($this->version);
        $curCols = $this->getColumns();
        $savedOrder = array_keys($savedCols);
        $curOrder = array_keys($curCols);
        $columnAfter = [];
        $needAfter = [];
        foreach($savedOrder as $colname) {
            $columnAfter[$colname] = $this->getColumnAfter($colname,$this->version);
        }
        unset($savedOrder);
        foreach($curOrder as $colname) {
            if($columnAfter[$colname] != $this->getColumnAfter($colname)) {
                $needAfter[$colname] = $columnAfter[$colname];
            }
        }
        unset($curOrder);
        


        $query = new Query();
        $table = $this->getTableName();
        $query->alter($table);
        foreach($curCols as $col=>$entry) {
            if(!isset($savedCols[$col])) {
                $origCol = $col;
                //currentColumn isn't part of the savedCols.. now we see if it changed
                $changed = $dropped = false;
                $colHist = $this->getColumnHistory($col);
                foreach($colHist as $ver=>$changes) {
                    foreach($changes as $change) {
                        if($change["action"] == "change" && $change["column"][0] == $col) {
                            $col = $change["column"][1];
                            $entry = ["type"=>$change["type"],"extra"=>$change["extra"]];
                            $changed=true;
                        }
                        if($changed && $change["action"] == "drop" && $change["column"] == $col) {
                            $changed = false;
                        }
                    }
                }
                if($changed) {
                    if(isset($savedCols[$col])) {
                        $extra = $savedCols[$col]["extra"];
                        if(isset($columnAfter[$col])) { 
                            $extra["after"] = $columnAfter[$col];
                            if(isset($needAfter[$col])) {
                                unset($needAfter[$col]);
                            }
                        }
                        $query->change($origCol,$col,$savedCols[$col]["type"],$extra);
                        $changes++;
                    } else {
                        throw new \Exception("Couldn't process change for $origCol to $col.. $col not part of current columns. History fail?");
                    }
                } else {
                    $query->remove($origCol);
                    $changes++;
                }
            }
        }
        $addedCols = [];
        foreach($savedCols as $sCol=>$entry) {
            if(!isset($curCols[$sCol])) {
                $entry["extra"]["after"] = $this->getColumnAfter($sCol,$this->version);
                $query->add($sCol,$entry["type"],$entry["extra"]);
                $addedCols[] = $sCol;
                $changes++;
            } else {
                $changed = false;
                $versionTypeDiff = false;
                if(strpos($entry["type"],"(") === false) {
                    if(substr($curCols[$sCol]["type"],0,strlen($entry["type"])+1) == $entry["type"]."(") {
                        $versionTypeDiff = true;
                    }
                }
                if($entry["type"] != $curCols[$sCol]["type"] && !$versionTypeDiff) {
                    $changed = true;
                } else {
                    if(!is_array($entry["extra"]) && is_array($curCols[$sCol]["extra"])) {
                        $changed = true;
                    } elseif(is_array($entry["extra"]) && !is_array($curCols[$sCol]["extra"])) {
                        $changed = true;
                    } elseif(count($entry["extra"]) !== count($curCols[$sCol]["extra"])) {
                        $changed = true;
                    } else {
                        foreach($entry["extra"] as $e=>$ev) {
                            if(!isset($curCols[$sCol]["extra"][$e]) || $curCols[$sCol]["extra"][$e] != $ev) {
                                $changed = true;
                            }
                        }
                    }
                }
                if($changed) {
                    $entry["extra"]["after"] = $this->getColumnAfter($sCol,$this->version);
                    if(isset($needAfter[$sCol])) {
                        unset($needAfter[$sCol]);
                    }
                    $query->change($sCol,$sCol,$entry["type"],$entry["extra"]);
                    $changes++;
                }
            }
        }
        if(!empty($needAfter)) {
            foreach($needAfter as $col=>$after) {
                if(in_array($after,$addedCols)) {

                } else {
                    $extra = $savedCols[$col]["extra"];
                    $extra["after"] = $after;
                    $query->change($col,$col,$savedCols[$col]["type"],$extra);
                    $changes++;
                }
            }
        }
        //TODO: order check //$entry["extra"]["after"] = $this->getColumnAfter($sCol,$this->version);
        if($changes>0) {
            foreach($this->getMeta($this->version) as $key=>$entry) {
                if($entry["key"] == "auto_increment")
                $query->meta($entry["key"],$entry["value"]);
            }
            return $query;
        }
        return false;
    }

    protected function getColumnAfter($colName,$version=null) {
        $columns = $this->getColumns($version);
        $colKeys = array_keys($columns);
        $after = false;
        foreach($colKeys as $sCol) {
            if($sCol == $colName) {
                return $after;
            }
            $after = $sCol;
        }
        return $after;
    }

    public function jsonSerialize($array=null) {
        if(is_null($array)) {
            
        }
        return $array;
    }

    public function __toString() {
        return json_encode($this);
    }

    /**
     * Get the value of tableName
     */ 
    public function getTableName() {
        return $this->tableName;
    }

    /**
     * Get the value of columns
     */ 
    public function getColumns($version=null) {
        if($version == null) {
            return $this->columns;
        } else {
            $version = $this->getVersionString($version);
            return dhGlobal::getVal($this->versions[$version],"columns",null);
        }
    }
    /**
     * Get the value of keys
     */ 
    public function getKeys($version=null) {
        if($version == null) {
            return $this->keys;
        } else {
            $version = $this->getVersionString($version);
            return dhGlobal::getVal($this->versions[$version],"keys",null);
        }
    }
    /**
     * Get the value of meta
     */ 
    public function getMeta($version=null) {
        if($version == null) {
            return $this->meta;
        } else {
            $version = $this->getVersionString($version);
            return dhGlobal::getVal($this->versions[$version],"meta",null);
        }
    }
    /**
     * Get the value of versions
     */ 
    public function getVersions() {
        return $this->versions;
    }
    /**
     * Get the value of history
     */ 
    public function getHistory($version=null) {
        if($version == null) {
            return $this->history;
        } else {
            $version = $this->getVersionString($version);
            return dhGlobal::getVal($this->history,$version,null);
        }
    }
    /**
     * Get the value of changes
     *
     * @return  mixed
     */
    public function getChanges() {
        return $this->changes;
    }
    /**
     * Set the value of columns
     *
     * @return  self
     */ 
    public function setColumns($columns) {
        $this->columns = $columns;
        return $this;
    }
    /**
     * Set the value of tableName
     *
     * @return  self
     */ 
    public function setTableName($tableName) {
        $this->tableName = $tableName;
        return $this;
    }
    /**
     * Set the value of keys
     *
     * @return  self
     */ 
    public function setKeys($keys) {
        $this->keys = $keys;
        return $this;
    }
    /**
     * Set the value of meta
     *
     * @return  self
     */ 
    public function setMeta($meta) {
        $this->meta = $meta;
        return $this;
    }

    /**
     * Get the value of file
     *
     * @return  mixed
     */
    public function getFile() {
        return $this->file;
    }

    /**
     * Set the value of file
     *
     * @param   File|String  $file  
     * @return  self
     */
    public function setFile($file) {
        if(!is_object($file)) {
            if(is_string($file)) {
                $this->file = new File(["path"=>$file]);
            } elseif(is_array($file)) {
                $this->file = new File($file);
            }
        } else {
            $this->file = $file;
        }
        return $this;
    }

    protected function getVersionString($version=null) {
        if(is_null($version)) {
            $version = $this->version;
        }
        if(!is_numeric($version)) {
            return $version;
        }
        return "v_".$version;
    }
}