<?php
namespace boru\dhdb\core;

use boru\dhutils\tools\DebugTrace;

class Debugger {
    /**
     * True if called from a browser
     * @var bool
     */
    private $isBrowser = false;

    /**
     * Whether to interpolate the query or print sql/params separately
     */
    private $useInterpolate = true;

    /**
     * Whether to save the query to the database
     * @var bool
     */
    private $logDebug = false;

    /**
     * Whether to print the query to the browser/screen
     * @var bool
     */
    private $printDebug = false;

    /**
     * Whether to print the error to the browser/screen
     * @var bool
     */
    private $printError = false;

    /**
     * Whether to log the error to the database
     * @var bool
     */
    private $logError = false;

    /**
     * Whether to include the trace in the output
     * @var bool
     */
    private $includeTrace = false;

    /**
     * The number of seconds to consider a query slow
     * @var float
     */
    private $slowLimit = 1;

    /**
     * Whether to log slow queries to the database
     * @var bool
     */
    private $logSlow = false;

    /**
     * Whether to log slow queries to the database
     * @var bool
     */
    private $printSlow = false;

    /**
     * The reference to use for the query
     * @var string
     */
    private $reference = "";

    private $callback = null;

    public function __construct($options=[]) {
        if(!empty($options)) {
            $this->setOptions($options);
        }
    }
    public function setOptions($options=[]) {
        foreach($options as $key=>$value) {
            if(method_exists($this,$key)) {
                $this->$key($value);
            }
        }
    }
    public function getOptions() {
        return [
            "isBrowser" => $this->isBrowser,
            "useInterpolate" => $this->useInterpolate,
            "logDebug" => $this->logDebug,
            "printDebug" => $this->printDebug,
            "includeTrace" => $this->includeTrace,
            "slowLimit" => $this->slowLimit,
            "logSlow" => $this->logSlow,
            "printSlow" => $this->printSlow,
            "printError" => $this->printError,
            "logError"  => $this->logError,
            "reference" => $this->reference,
            "callback" => $this->callback,
        ];
    }

    public function isBrowser($isBrowser=null) {
        if(is_null($isBrowser)) return $this->isBrowser;
        $this->isBrowser = $isBrowser;
    }
    public function useInterpolate($useInterpolate=null) {
        if(is_null($useInterpolate)) return $this->useInterpolate;
        $this->useInterpolate = $useInterpolate;
    }
    public function logDebug($logDebug=null) {
        if(is_null($logDebug)) return $this->logDebug;
        $this->logDebug = $logDebug;
    }
    public function printDebug($printDebug=null) {
        if(is_null($printDebug)) return $this->printDebug;
        $this->printDebug = $printDebug;
    }
    public function printError($printError=null) {
        if(is_null($printError)) return $this->printError;
        $this->printError = $printError;
    }
    public function logError($logError=null) {
        if(is_null($logError)) return $this->logError;
        $this->logError = $logError;
    }
    public function includeTrace($includeTrace=null) {
        if(is_null($includeTrace)) return $this->includeTrace;
        $this->includeTrace = $includeTrace;
    }
    public function slowLimit($slowLimit=null) {
        if(is_null($slowLimit)) return $this->slowLimit;
        $this->slowLimit = $slowLimit;
    }
    public function logSlow($logSlow=null) {
        if(is_null($logSlow)) return $this->logSlow;
        $this->logSlow = $logSlow;
    }
    public function printSlow($printSlow=null) {
        if(is_null($printSlow)) return $this->printSlow;
        $this->printSlow = $printSlow;
    }
    public function reference($reference=null) {
        if(is_null($reference)) return $this->reference;
        $this->reference = $reference;
    }
    public function callback($callback=null) {
        if(is_null($callback)) return $this->callback;
        $this->callback = $callback;
    }

    /**
     * 
     * @param Statement $statement 
     * @return void 
     */
    public function log($statement) {
        if(is_callable($this->callback)) {
            call_user_func($this->callback,$statement);
        }
        $isSlow = $statement->isSlow($this->slowLimit);
        $isError = $statement->isError();
        $shouldLog = $this->shouldLog($isSlow,$isError);
        $shouldPrint = $this->shouldPrint($isSlow,$isError);

        $includeTrace = $this->includeTrace || $isError;

        //should we include the trace?
        if(empty($statement->getTrace()) && $includeTrace && ($shouldLog || $shouldPrint)) {
            $statement->setTrace(DebugTrace::getTraceInfo(4,true));
        }


        if($shouldLog) {
            $this->doLog($statement);
        }
        if($shouldPrint) {
            $this->doPrint($statement);
        }
    }

    private function doLog($statement) {
        //TODO: log to database
        //$this->logToTable($statement);
    }

    private function doPrint($statement) {
        if($this->isBrowser) {
            DebugPrinter::printHtml($statement,$this->isSlow($statement),$this->useInterpolate);
        } else {
            DebugPrinter::printText($statement,$this->isSlow($statement),$this->useInterpolate);
        }
    }

    private function isSlow($statement) {
        return $statement->getExecTime() >= $this->slowLimit;
    }

    private function shouldLog($isSlow=false,$isError=false) {
        if($this->logDebug) {
            return true;
        }
        if($this->logSlow && $isSlow) {
            return true;
        }
        if($this->logError && $isError) {
            return true;
        }
        return false;
    }

    private function shouldPrint($isSlow=false,$isError=false) {
        if($this->printDebug) {
            return true;
        }
        if($this->printSlow && $isSlow) {
            return true;
        }
        if($this->printError && $isError) {
            return true;
        }
        return false;
    }
}