<?php
namespace boru\dhdb\core;

use boru\dhdb\dhDB;
use boru\dhutils\tools\DebugTrace;

class Statement {
    /** @var Mysql */
    private $mysql;
    /** @var \PDO */
    private $pdo;
    /** @var string */
    private $query = "";
    /** @var string */
    /** @var array */
    private $params = [];
    /** @var \PDOStatement */
    private $stmt;
    private $bindPos = 1;

    private $startTime = 0;
    private $endTime = 0;
    private $execTime = 0;
    private $completed = false;
    /** @var bool|Error */
    private $error = false;

    private $trace = null;

    private $options = [
        "debug" => false,
        "debugOnError" => false,
        "queryCallback" => null,
        "trace" => false,
    ];

    public function __construct($mysql,$query=null,$params=[]) {
        $this->mysql = $mysql;
        $this->pdo = $mysql->pdo();
        $this->query = $query;
        if(!is_null($query)) {
            $this->prepare($query);
        }
        if(!is_null($params)) {
            $this->bind($params);
        }
    }
    public function __destruct() {
        $this->stmt = null;
    }

    public function run($query=null,$params=[]) {
        if(!$this->hasStmt()) {
            return $this;
        }
        
        if(!is_null($query) || !empty($params)) {
            $query = is_null($query) ? $this->query : $query;
            $params = is_null($params) ? $this->params : $params;
            $this->prepare($query,$params);
        }
        $this->startTime = microtime(true);
        $this->completed = false;
        try { 
            $this->stmt->execute();
            $this->completed = true;
            $this->error = false;
        } catch (\PDOException $e) {
            $this->error = new Error($this->query,$this->params,$this->interpolate(),$e);
        } catch (\Exception $e) {
            $this->error = new Error($this->query,$this->params,$this->interpolate(),$e);
        }
        $this->endTime = microtime(true);
        $this->execTime = $this->endTime - $this->startTime;
        $this->logQuery();
        return $this;
    }

    private function logQuery() {
        $this->mysql->getDebugger()->log($this);
    }

    public function isSlow($slowTime=1) {
        return $this->execTime > $slowTime;
    }
    public function isError() {
        return $this->error !== false;
    }

    public function getQuery() {
        return $this->query;
    }
    public function getParams() {
        return $this->params;
    }
    public function getExecTime() {
        return $this->execTime;
    }
    public function getError() {
        return $this->error;
    }
    public function getTrace() {
        return $this->trace;
    }
    public function setTrace($trace) {
        $this->trace = $trace;
        return $this;
    }
    public function getCompleted() {
        return $this->completed;
    }
    public function interpolate() {
        if(!empty($this->query)) {
            return dhDB::interpolateQuery($this->query,$this->params);
        }
        return "";
    }
    public function getPDO() {
        return $this->pdo;
    }
    public function toArray() {
        return [
            "query" => $this->query,
            "params" => $this->params,
            "interpolated" => $this->interpolate(),
            "error" => $this->error,
            "trace" => $this->trace,
            "execTime" => $this->execTime,
            "completed" => $this->completed,
        ];
    }


    /**
     * Prepare the query
     * @param mixed $query 
     * @param array $params 
     * @return $this 
     */
    public function prepare($query,$params=[]) {
        $this->query = $query;
        $this->stmt = $this->mysql->pdo()->prepare($query);
        if(!is_null($this->stmt) && $this->stmt !== false) {
            $this->stmt->setFetchMode(\PDO::FETCH_CLASS | \PDO::FETCH_PROPS_LATE,"\\boru\\dhdb\\core\\Row");
        }
        if(!is_null($params)) {
            $this->bind($params);
        }
        return $this;
    }
    
    /**
     * Set the parameters for the query
     * @param mixed $array 
     * @param mixed $array2 
     * @return bool 
     */
    public function bind($array=null) {
        if(is_null($array) || $array === false) {
            return false;
        }
        if(!is_array($array)) {
            $array = [$array];
        }
        $this->params = array_merge($this->params,$array);
        foreach($array as $value) {
            $this->bindParam($value);
        }
        return true;
    }
    
    public function bindParam($value,$type=null) {
        $this->bindPos($this->bindPos,$value,$type);
        $this->bindPos++;
	}
    public function bindPos($pos,$value,$type=null) {
        if(!$this->hasStmt()) {
            return false;
        }
		if( is_null($type) ) {
			switch( true ) {
				case is_int($value):
					$type = \PDO::PARAM_INT;
					break;
				case is_bool($value):
					$type = \PDO::PARAM_BOOL;
					break;
				case is_null($value):
					$type = \PDO::PARAM_NULL;
					break;
				default:
					$type = \PDO::PARAM_STR;
			}
		}
		$this->stmt->bindValue($pos,$value,$type);
    }

    public function hasStmt() {
        return !is_null($this->stmt) && $this->stmt !== false;
    }
    public function nextRow($assoc=true,$object=true) {
        if(!$this->hasStmt()) {
            return false;
        }
        if(!is_null($this->stmt) && $this->stmt !== false) {
            if($object) {
                return $this->stmt->fetch($assoc ? \PDO::FETCH_ASSOC : \PDO::FETCH_NUM);
            } else {
                return $this->stmt->fetch($assoc ? \PDO::FETCH_ASSOC : \PDO::FETCH_NUM);
            }
        }
        $this->stmt = null;
        unset($this->stmt);
        return false;
    }
    public function next($mode=\PDO::FETCH_ASSOC,$cursorOrientation = \PDO::FETCH_ORI_NEXT,$cursorOffset = 0) {
        if(!$this->hasStmt()) {
            return false;
        }
        if(!is_null($this->stmt) && $row = Row::fromStatement($this->stmt->fetch($mode,$cursorOrientation,$cursorOffset))) {
            return $row;
        }
        $this->stmt = null;
        unset($this->stmt);
        return false;
    }
    public function all($object=true) {
        if(!$this->hasStmt()) {
            return false;
        }
        if($object) {
            return $this->stmt->fetchAll(\PDO::FETCH_CLASS | \PDO::FETCH_PROPS_LATE,"\\boru\\dhdb\\drivers\\mysql\\Row");
        } else {
            return $this->stmt->fetchAll(\PDO::FETCH_ASSOC);
        }
    }

    public function __call($method,$args) {
        if(!$this->hasStmt()) {
            return false;
        }
        return call_user_func_array( [$this->stmt,$method], $args);
    }
}