<?php
namespace boru\dhdb\core\query;

use \boru\dhutils\dhGlobal;

class QueryParser {
    protected $query;
    protected $queryString;
    
    public function __construct($query=null) {
        if(!is_null($query)) {
            $this->setQuery($query);
        }
    }
    public function parse($query=null) {
        if(is_null($query)) {
            $query = $this->getQuery();
        }

        //raw query string
        if(!is_null($query->rawQueryString)) {
			return $this->setQueryString($query->rawQueryString);
		}
        
        
        if(isset($query->type) && !is_null($query->type)) {
            //select query
            if($query->type == "select") {
                return $this->parseSelect($query);
            }

            //update query
            if($query->type == "update") {
                return $this->parseUpdate($query);
            }
            
            //insert query
            if($query->type == "insert") {
                return $this->parseInsert($query);
            }

            //delete query

            //union query

            //create table
            if($query->type == "create") {
                return $this->parseCreateTable($query);
            }

            //alter table
            if($query->type == "alter") {
                return $this->parseAlterTable($query);
            }
        }

        
    }
    protected function parseSelect($query) {
        $sql = "";
        $sql.= $this->parseColumnString($query," ");
        $sql.= $this->parseTableString($query," FROM ");
        $sql.= $this->parseConditionString($query," WHERE ");
        $sql.= $this->parseGroupByString($query);
        $sql.= $this->parseOrderByString($query);
        $sql.= $this->parseLimitString($query);
        if(!empty($sql)) {
            return $this->setQueryString("SELECT".$sql);
        }
        return false;
    }

    protected function parseUpdate($query) {
        $sql = "";
        $sql.= $this->parseTableString($query,"UPDATE ");
        $sql.= $this->parseColumnString($query," SET ",false);
        $sql.= $this->parseConditionString($query," WHERE ");
        $sql.= $this->parseGroupByString($query);
        $sql.= $this->parseOrderByString($query);
        $sql.= $this->parseLimitString($query);
        if(!empty($sql)) {
            return $this->setQueryString($sql);
        }
        return false;
    }

    protected function parseInsert($query) {
        $sql = "";
        $sql.= $this->parseTableString($query,"INSERT INTO ");
        $sql.= " (";
        $sql.= $this->parseColumnString($query);
        $sql.= " ) ";
        $sql.= $this->parseInsertValues($query,"VALUES ");
        if(!empty($sql)) {
            return $this->setQueryString($sql);
        }
        return false;
    }

    protected function parseCreateTable($query) {
        $sql = "";
        if($query->ifNotExists) {
            $sql.= $this->parseTableString($query,"CREATE TABLE IF NOT EXISTS ");
        } else {
            $sql.= $this->parseTableString($query,"CREATE TABLE ");
        }
        $sql.= " (";
        $sql.= $this->parseSchemaColumnsCreate($query);
        $sql.= ") ";
        $sql.= $this->parseTableMeta($query," ");
        if(!empty($sql)) {
            return $this->setQueryString($sql);
        }
        return false;
    }
    protected function parseAlterTable($query) {
        $tableAlter = $this->parseTableString($query,"ALTER TABLE ");
        $columnAlter = $this->parseSchemaColumnsAlter($query);
        $metaAlter = $this->parseTableMeta($query,",");
        $sql = $tableAlter." ";
        $sqlInner = [];
        if(!empty($columnAlter)) {
            $sqlInner[]=$columnAlter;
        }
        if(!empty($metaAlter)) {
            $sqlInner[]=$metaAlter;
        }
        if(!empty($sqlInner)) {
            return $this->setQueryString($sql.implode(", ",$sqlInner));
        }
        return false;
    }

    protected function parseGroupByString($query) {
        if(!is_null($query->groupBy)) {
            if(is_array($query->groupBy)) {
                return " GROUP BY ".implode(",",$query->groupBy);
            } else {
                return " GROUP BY ".$query->groupBy;
            }
        }
        return "";
    }
    protected function parseOrderByString($query) {
        if(!is_null($query->orderBy)) {
            if(is_array($query->orderBy)) {
                return " ORDER BY ".implode(",",$query->orderBy);
            } else {
                return " ORDER BY ".$query->orderBy;
            }
        }
        return "";
    }
    protected function parseLimitString($query) {
        if(!is_null($query->limit)) {
            if(is_array($query->limit)) {
                return " LIMIT ".implode(",",$query->limit);
            } else {
                return " LIMIT ".$query->limit;
            }
        }
        return "";
    }
    protected function parseColumnString($query,$prefix="",$sanitize=true) {
        if(empty($query->columns)) {
            return "";
        }
        if($sanitize) {
            $columns = array_map([$this,'sanitizeColumnName'],$query->columns);
        } else {
            $columns = $query->columns;
        }
        return $prefix.implode(",",$columns);
    }
    protected function parseConditionString($query,$prefix="") {
        if(empty($query->conditions)) {
            return "";
        }
        $conds = [];
        foreach($query->conditions as $cond) {
            $conds[] = "($cond)";
        }
        
        return $prefix.implode(" AND ",$conds);
    }
    protected function parseTableString($query,$prefix="") {
        if(empty($query->tables)) {
            return "";
        }
        $tables = $query->tables;
        $string = $this->sanitizeColumnName($tables[0]["name"]);
        array_shift($tables);
        if(!empty($tables)) {
            $tableStrings = [];
            foreach($tables as $table) {
                $tstring = "";
                if(!is_null($table["type"])) {
                    $tstring.=" ".$table["type"]." ";
                } else {
                    $tstring.=",";
                }
                $tstring .= $this->sanitizeColumnName($table["name"]);
                if(!is_null($table["on"])) {
                    $tstring.=" ON ".$table["on"];
                }
                $tableStrings[] = $tstring;
            }
            $string.=implode(" ",$tableStrings);
        }
        return $prefix.$string;
    }
    public function parseInsertValues($query,$prefix = "") {
        $valueGroups = [];
        if(!empty($query->values)) {
            if(is_array($query->values[0])) {
                //TODO
                
            } else {
                $valueGroups[] = "(".$this->generateQs($query->values).")";
            }
        }
        return $prefix.implode(",",$valueGroups);
    }
    public function parseSchemaColumnsCreate($query,$prefix = "") {
        if(empty($query->schemaColumns)) {
            return "";
        }
        $coldefs = [];
        foreach($query->schemaColumns as $def) {
            $action = $def["action"];
            $col = $def["column"];
            $type = $def["type"];
            $extra = $def["extra"];
            if($action == "add") {
                $coldefs[] = $this->sanitizeColumnName($col)." $type".(!empty($extra) ? " ".$this->parseColExtra($extra,$type) : "");
            } elseif($action == "addIndex") {
                $indexCols = [];
                if(is_array($extra)) {
                    foreach($extra as $v) {
                        $indexCols[] = $this->sanitizeColumnName($v);
                    }
                } else {
                    $indexCols[] = $this->sanitizeColumnName($extra);
                }
                if(empty($type) || $type == "key") {
                    $coldefs[] = "KEY ".$this->sanitizeColumnName($col)." (".implode(",",$indexCols).")";
                } elseif($type == "primary") {
                    $coldefs[] = "PRIMARY KEY (".implode(",",$indexCols).")";
                } elseif($type == "fulltext") {
                    $coldefs[] = "FULLTEXT KEY ".$this->sanitizeColumnName($col)." (".implode(",",$indexCols).")";
                } elseif($type == "unique") {
                    $coldefs[] = "UNIQUE KEY ".$this->sanitizeColumnName($col)." (".implode(",",$indexCols).")";
                }
            }
        }
        return implode(",",$coldefs);
    }
    public function parseSchemaColumnsAlter($query,$prefix = "") {
        if(empty($query->schemaColumns)) {
            return "";
        }
        $out = [];
        foreach($query->schemaColumns as $def) {
            //$this->schemaColumns[] = ["action"=>$action,"column"=>$col,"type"=>$type, "extra"=>$extra];
            $action = $def["action"];
            $col = $def["column"];
            $type = $def["type"];
            $extra = $def["extra"];
            if($action == "add") {
                $out[] = "ADD COLUMN ".$this->sanitizeColumnName($col)." $type".(!empty($extra) ? " ".$this->parseColExtra($extra,$type) : "");
            } elseif($action == "drop") {
                $out[] = "DROP COLUMN ".$this->sanitizeColumnName($col);
            } elseif($action == "change") {
                $out[] = "CHANGE COLUMN ".$this->sanitizeColumnName($col[0])." ".$this->sanitizeColumnName($col[1])." $type".(!empty($extra) ? " ".$this->parseColExtra($extra,$type) : "");
            } elseif($action == "dropIndex") {
                $out[] = "DROP INDEX ".$this->sanitizeColumnName($col);
            } elseif($action == "addIndex") {
                $indexCols = [];
                if(is_array($extra)) {
                    foreach($extra as $v) {
                        $indexCols[] = $this->sanitizeColumnName($v);
                    }
                } else {
                    $indexCols[] = $this->sanitizeColumnName($extra);
                }
                if(empty($type) || $type == "key") {
                    $out[] = "ADD KEY ".$this->sanitizeColumnName($col)." (".implode(",",$indexCols).")";
                } elseif($type == "primary") {
                    $out[] = "ADD PRIMARY KEY (".implode(",",$indexCols).")";
                } elseif($type == "fulltext") {
                    $out[] = "ADD FULLTEXT INDEX ".$this->sanitizeColumnName($col)." (".implode(",",$indexCols).")";
                } elseif($type == "unique") {
                    $out[] = "ADD UNIQUE INDEX ".$this->sanitizeColumnName($col)." (".implode(",",$indexCols).")";
                }
            }
        }
        return $prefix.implode(",",$out);
    }
    public function parseTableMeta($query,$implode=",") {
        //tableMeta
        if(empty($query->tableMeta)) {
            return "";
        }
        $out = [];
        foreach($query->tableMeta as $def) {
            $key = $def["key"];
            $value = $def["value"];
            if($key != "auto_increment") {
                if(is_null($value) || empty($value)) {
                    $out[] = $key;
                } else {
                    $noEmpty = ["comment","auto_increment"];
                    if($key == "comment" && !empty($value)) {
                        $out[] = $key.'="'.$value.'"';
                    } else {
                        $out[] = $key."=".$value;
                    }
                }
            }
        }
        return implode($implode,$out);
    }
    protected function parseColExtra($extra,$type,$glue=" ") {
        if(!is_array($extra)) {
            if(empty($extra)) {
                return "";
            } else {
                return $extra;
            }
        }
        $null = dhGlobal::getVal($extra,"null",false);
        $collate = dhGlobal::getVal($extra,"collate",false);
        $charset = dhGlobal::getVal($extra,"charset",false);
        $default = dhGlobal::getVal($extra,"default",false);
        $auto_incr = dhGlobal::getVal($extra,"auto_increment",false);
        $after = dhGlobal::getVal($extra,"after",false);
        if(!$charset && $collate !== false) {
            list($charset,$unused) = explode("_",$collate,2);
        }
        $parts = [];
        if($charset !== false) {
            $parts[] ="CHARACTER SET ".$charset;
        }
        if($collate !== false) {
            $parts[] ="COLLATE ".$collate;
        }
        if($null) {
            if($default === false) {
                $parts[] = "DEFAULT NULL";
            } else {
                if($type == "timestamp" && $default == "CURRENT_TIMESTAMP") {
                    $parts[] = 'DEFAULT CURRENT_TIMESTAMP';
                } else {
                    $parts[] = 'DEFAULT "'.$default.'"';
                }
            }
        } else {
            $parts[] = "NOT NULL";
            if($default !== false) {
                if($type == "timestamp" && $default == "CURRENT_TIMESTAMP") {
                    $parts[] = 'DEFAULT CURRENT_TIMESTAMP';
                } else {
                    $parts[] = 'DEFAULT "'.$default.'"';
                }
            }
        }
        if($auto_incr !== false) {
            $parts[] = "AUTO_INCREMENT";
        }
        if($after !== false) {
            $parts[] = "AFTER ".$this->sanitizeColumnName($after);
        }
        return implode($glue,$parts);
    }

    public function sanitizeColumnName($colString) {
        if(!is_array($colString) && $colString == "*") {
            return $colString;
        }
        if(!is_array($colString)) {
            //$columns = explode(",",$colString);
            $columns = [$colString];
        } else {
            $columns = $colString;
        }

        return $this->sanitize_arrayOfColumns($columns);
    }
    public function sanitize_arrayOfColumns($columns) {
        foreach($columns as $k=>$colString) {
            if(strpos($colString,"(") !== false && strpos($colString,")") !== false) {
                //do nothing.. it's a function or subquery, and i'm not about to go down that rabbit hole.
                echo "skipping $colString\n";
            } elseif(strpos($colString," ") !== false) {
                $columns[$k] = $this->sanitize_colWithAlias($colString);
            } elseif(strpos($colString,".") !== false) {
                $columns[$k] = $this->sanitize_colWithSub($colString);
            } else {
                $columns[$k] = $this->sanitize_colName($colString);
            }
        }
        return implode(",",$columns);
    }
    public function sanitize_colWithAlias($colString) {
        $parts = preg_split('/\sas\s|\s/', strtolower($colString));
        foreach($parts as $k=>$part) {
            if(strpos($parts[$k],".") !== false) {
                $parts[$k] = $this->sanitize_colWithSub($part);
            } else {
                $parts[$k] = $this->sanitize_colName($part);
            }
        }
        return implode(" ",$parts);
    }
    public function sanitize_colWithSub($colString) {
        $parts = explode(".",strtolower($colString));
        foreach($parts as $k=>$part) {
            $parts[$k] = $this->sanitize_colName($part);
        }
        return implode(".",$parts);
    }
    
    public function sanitize_colName($col) {
        if($col == "*") {
            return $col;
        }
        return "`".preg_replace("/[^A-Za-z0-9\-_\.\(\)\*]/", '', $col)."`";
    }
    public function sanitize_quoteCol($field) {
        if($field == "*") {
            return $field;
        }
        return preg_replace("/[^\`A-Za-z0-9\-_\.\(\)\*\ ]/", '', $field);
    }

    /**
     * Get the value of query
     */ 
    public function getQuery()
    {
        return $this->query;
    }

    /**
     * Set the value of query
     *
     * @return  self
     */ 
    public function setQuery($query)
    {
        $this->query = $query;

        return $this;
    }

    /**
     * Get the value of queryString
     */ 
    public function getQueryString()
    {
        return $this->queryString;
    }

    /**
     * Set the value of queryString
     *
     * @return  self
     */ 
    public function setQueryString($queryString)
    {
        $this->queryString = $queryString;

        return $this;
    }

    /**
     * Generate an array of [?,?,?,...]
     * 
     * @param array $array the array to count/replace with '?'
     * @param string $glue optional glue to use on the returned string (default ',');
     * @return string
     */
    public function generateQs($array,$glue=",") {
		$qarr = array_fill(0,count($array),"?");
        return implode($glue,$qarr);
	}
}