<?php
namespace boru\dhfw\base;

use boru\dhutils\filesys\Directory;
use ReflectionClass;

abstract class BaseModule {
    /**
     * The name of the module
     * @var string
     */
    protected static $moduleName = "Module";

    /**
     * The name of the table in the database
     * @var string
     */
    protected static $moduleTable;

    /**
     * The name of the primary key fields in the database
     * * createdTime and updatedTime can be blank to disbale automatic timestamping
     * @var string
     */
    protected static $defaultFieldNames = [
        "id"=>"id",
        "createdTime"=>"created",
        "updatedTime"=>"updated",
    ];

    /**
     * If true, the module will use the table above to store data, otherwise no data will be stored/expected 
     * @var bool
     */
    protected static $isRecordModule = true;

    /**
     * The default view to use for this module
     * @var string
     */
    protected static $defaultView = "ListView";

    protected static $hasInit = false;

    protected static $reflectionClass;


    public static function initModule() {
        if(static::$hasInit) {
            return;
        }
        Directory::fromPathString(static::getModuleDirectory(),true,["scan"=>false]);
        Directory::fromPathString(static::getTemplateDirectory(),true,["scan"=>false]);
        Directory::fromPathString(static::getViewDirectory(),true,["scan"=>false]);
        Directory::fromPathString(static::getActionDirectory(),true,["scan"=>false]);
        static::$hasInit = true;
    }

    public static function getModuleDirectory() {
        static::initModule();
        return \boru\dhfw\DHFW::dir("app/".strtolower(static::$moduleName));
    }
    public static function getTemplateDirectory() {
        return static::getModuleDirectory()."/templates";
    }
    public static function getViewDirectory() {
        return static::getModuleDirectory()."/views";
    }
    public static function getActionDirectory() {
        return static::getModuleDirectory()."/actions";
    }

    /**
     * Get the ReflectionClass for this module
     * @return ReflectionClass<$this> 
     */
    public static function reflectionClass() {
        if(is_null(static::$reflectionClass)) {
            static::$reflectionClass = new \ReflectionClass(new static());
        }
        return static::$reflectionClass;
    }
    public static function getNamespace() {
        return static::reflectionClass()->getNamespaceName();
    }
    public static function getNamespacedClassName() {
        return static::getNamespace()."\\Module";
    }


    public static function getView($viewName) {
        $className = static::getNamespace()."\\views\\".$viewName;
        $fallback = "\\FWAPP\\core\\views\\".$viewName;
        if(class_exists($className)) {
            return new $className();
        } elseif(class_exists($fallback)) {
            return new $fallback();
        }else {
            return new \FWAPP\core\views\NotFound();
        }
    }
    public static function getAction($actionName) {
        $className = static::getNamespace()."\\actions\\".$actionName;
        $fallback = "\\FWAPP\\core\\actions\\".$actionName;
        if(class_exists($className)) {
            return new $className();
        } elseif(class_exists($fallback)) {
            return new $fallback();
        }else {
            return new \FWAPP\core\actions\NotFound();
        }
    }


    public function getInfo() {
        return [
            "name"=>static::$moduleName,
            "table"=>static::$moduleTable,
            "fields"=>static::$defaultFieldNames,
            "isRecordModule"=>static::$isRecordModule,
            "defaultView"=>static::$defaultView,
            "namespace"=>static::getNamespace(),
            "viewNamespace"=>static::getNamespace()."\\views",
            "actionNamespace"=>static::getNamespace()."\\actions",
            "namespaceClass"=>static::getNamespacedClassName(),
        ];
    }
}