<?php
namespace boru\dhfw\base;

use boru\dhcli\dhCLI;
use boru\dhcli\ClassCommandInterface;
use boru\dhcli\Option;
use boru\dhfw\DHFW;
use boru\dhfw\FWCli;
use boru\dhfw\util\FindDir;
use boru\dhutils\dhGlobal;

abstract class BaseCommand {

    public static $commandDescription = "Base command";
    public static function commandOptions() {
        return [];
    }
    public $lockFile = false;

    /** @var FWCli */
    private $FWCli = null;

    /** @var dhCLI */
    private $dhCli = null;

    private $commandParsed = false;

    public function __destruct() {
        $this->unlock();
    }
    public function __construct() {
    }
    public function getFWCli() {
        return $this->FWCli;
    }
    public function preProcess() {
        $this->lock();
    }
    
    abstract function process();
    public function postProcess() {
        $this->unlock();
    }

    public function trigger($FWCli) {
        $this->FWCli = $FWCli;
        //
        $command = dhCli::makeCommand([
            "command"=>$this->FWCli->commandName(),
            "description"=>static::$commandDescription,
            "opts"=>static::commandOptions(),
            "callback"=>function($result) {
                $this->setCommandParsed($result);
                $this->preProcess();
                $this->process();
                $this->postProcess();
            }
        ]);
        $commandArgs = $this->FWCli->args();
        $isHelp=false;
        if(!empty($commandArgs) && strtolower($commandArgs[0]) == "help") {
            $isHelp=true;
            array_shift($commandArgs);
        }
        array_unshift($commandArgs,$this->FWCli->commandName());
        if($isHelp) {
            array_unshift($commandArgs,"help");
        }
        FWCli::dhCli()->command($command);
        FWCli::dhCli()->process($commandArgs);
        return;
    }

    public function setCommandParsed($result) {
        $this->commandParsed = $result;
    }

    public function get($opt,$default=null) {
        return dhGlobal::getVal($this->commandParsed["options"],$opt,$default);
    }
    public function getProcessed() {
        return $this->commandParsed;
    }

    public static function commonOptions() {
        return [
            Option::option( "d","debug","Debug level to print",false,["default"=>"none","options"=>["none","error","debug","all"]]),
        ];
    }
    public static function commandName() {
        return static::class;
    }
    /** Lockfile execution.. public so it can be overridden */
    public function isLocked() {
        if($this->lockFile === false) {
            return false;
        }
		if(file_exists($this->lockFile)) {
			$lockingPID = trim(file_get_contents($this->lockFile));
			$pids = explode("\n",trim( `ps -e | awk '{print $1}'` ));
			if(in_array($lockingPID,$pids)) return true;
			unlink($this->lockFile);
		}
		return false;
	}
    public function unlock() {
        if($this->lockFile !== false && file_exists($this->lockFile)) {
            unlink($this->lockFile);
        }
    }
	public function lock() {
        if($this->lockFile === false) {
            return false;
        }
        if($this->lockFile === true) {
            $this->lockFile = $this->getLockFileTempDir().preg_replace("/[^a-zA-Z0-9]/","_",static::class).".lock";
        }
        if($this->isLocked()) {
            throw new \Exception("Already running");
        }
		file_put_contents($this->lockFile,getmypid()."\n");
		return true;
	}
    private function getLockFileTempDir() {
        if(($cacheDir = DHFW::getConfig("cacheDir",false)) === false) {
            $cacheDir = sys_get_temp_dir();
        }
        if(substr($cacheDir,-1) != "/") {
            $cacheDir .= "/";
        }
        return $cacheDir;
    }

    private $columnLengths = [];
    private $rows = [];

    public function addRow($columns=[]) {
        $columns = array_values($columns);
        $this->rows[] = $columns;
        foreach($columns as $col=>$val) {
            $curMax = isset($this->columnLengths[$col]) ? $this->columnLengths[$col] : 0;
            $this->columnLengths[$col] = max($curMax,strlen($val));
        }
    }
    public function printTable($separator=" | ") {
        foreach($this->rows as $row) {
            $this->printRow($row,$separator);
        }
    }
    public function printRow($row,$separator=" | ") {
        $outArray = [];
        foreach($row as $col=>$val) {
            $outArray[] = dhGlobal::padRight($val,$this->columnLengths[$col]+2," ");
        }
        dhGlobal::outLine(implode($separator,$outArray));
    }

    public function makeBarLine($text,$char="-") {
        $minLength = 75;
        $colLength = 0;
        if(!empty($this->columnLengths)) {
            $colLength = array_sum($this->columnLengths)+count($this->columnLengths)*2;
        }
        $length = max($minLength,$colLength);
        return dhGlobal::padRight($text,$length,$char);
    }
}