<?php
namespace boru\dhfw;

use boru\dhapi\core\Container;
use boru\dhapi\core\Dispatcher;
use boru\dhapi\API;
use boru\dhapi\routing\Route;
use boru\dhapi\routing\Router;
use boru\dhfw\util\Debug;
use boru\dhutils\dhGlobal;
use boru\dhutils\filesys\Directory;

class AppRouter {
    /** @var Container */
    private $args;
    /** @var Container */
    private $request;
    /** @var Dispatcher */
    private $api;

    private $moduleName;
    private $viewName;
    private $actionName;
    private $recordIds = [];
    private $rest = [];

    private $baseUrl;
    private $module;
    private $view;
    private $action;
    private $apiItem;
    private $record;

    private $user;
    private $isAuthed = false;

    private static $baseNS = null;

    /**
     * @param Dispatcher $api 
     * @return void 
     */
    public function processWeb($api) {
        $this->api = $api;
        $timer = Debug::add(999);
        $this->baseUrl = $api->baseUrl();
        $this->args = $api->getArgs();
        $this->request = new Container($api->getRequest()->body());
        $this->parseVars($this->args);
        if(is_null($this->moduleName) || $this->moduleName == "") {
            $this->moduleName = "basemodule";
        }
        $this->checkLogin();
        try {
            if(($handler = $this->getView()) !== false) {
            } elseif(($handler = $this->getAction()) !== false) {
            } elseif(($handler = $this->getView(true)) !== false) {
                $this->viewName = $this->getModule()->getDefaultViewName();
                $this->view = $handler;
            } else {
                $handler = new \boru\dhfw\app\basemodule\views\NotFound();
            }
        
    
            if(is_object($handler)) {
                $handler->trigger($this);
            }
        } catch (\Exception $e) {
            $view = new \boru\dhfw\app\basemodule\views\ExceptionView();
            $view->setException($e);
            $view->trigger($this);
        }
        exit();
    }

    /**
     * @param Dispatcher $api 
     * @return void 
     */
    public function processApi($api) {

        $this->api = $api;
        $timer = Debug::add(999);
        $this->baseUrl = $api->baseUrl();
        $this->args = $api->getArgs();
        $this->request = new Container($api->getRequest()->body());
        $this->parseVars($this->args);
        if(is_null($this->moduleName) || $this->moduleName == "") {
            $this->moduleName = "basemodule";
        }
        try {
            if(($handler = $this->getApi()) !== false) {
            } else {
                $handler = new \boru\dhfw\app\basemodule\apis\NotFound();
            }
        
    
            if(is_object($handler)) {
                $handler->trigger($this);
            }
        } catch (\Exception $e) {
            $view = new \boru\dhfw\app\basemodule\apis\ExceptionApi();
            $view->setException($e);
            $view->trigger($this);
        }
        exit();
    }

    public function getUser() {
        if(is_null($this->user)) {
            if(($className = static::searchClass("user","User",null,false)) !== false) {
                $this->user = new $className();
            } elseif(($className = static::searchClass("user","Record",null,false)) !== false) {
                $this->user = new $className();
            } else {
                $this->user = new \boru\dhfw\app\user\User();
            }
        }
        if(is_null($this->user)) return false;
        return $this->user;
    }

    /**
     * @return Module|false
     */
    public function getModule() {
        if(is_null($this->module)) {
            if(($className = static::searchClass($this->moduleName,ucfirst($this->moduleName))) !== false) {
                $this->module = new $className();
            } elseif(($className = static::searchClass($this->moduleName,$this->moduleName)) !== false) {
                $this->module = new $className();
            } elseif(($className = static::searchClass($this->moduleName,"Module")) !== false) {
                $this->module = new $className();
            } else {
                $this->module = new \boru\dhfw\app\basemodule\Module();
            }
        }
        if(is_null($this->module)) return false;
        return $this->module;
    }
    /**
     * @return View|false
     */
    public function getView($default=false) {
        //get the caller line and filename
        if(!$default && (is_null($this->viewName) || empty($this->viewName))) {
            return false;
        }
        if($default) {
            $viewName = $this->getModule()->getDefaultViewName();
        } else {
            if(!is_null($this->view)) {
                return $this->view;
            }
            $viewName = $this->viewName;
        }
        $view = null;
        if(($className = static::searchClass($this->moduleName,$viewName,"view")) !== false) {
            $view = new $className();
        } else {
            $view = $this->getNotFoundView();
        }
        if(!$default) {
            $this->view = $view;
        }
        return $view;
    }

    /**
     * @return View|false
     */
    public function getLoginView() {
        if(($className = static::searchClass("User","Login","view")) !== false) {
            return new $className();
        } else {
            return new \boru\dhfw\app\user\views\LoginView();
        }
    }
    /**
     * @return View|false
     */
    public function getExceptionView() {
        if(($className = static::searchClass($this->moduleName,"Exception","view")) !== false) {
            return new $className();
        } else {
            return new \boru\dhfw\app\basemodule\views\ExceptionView();
        }
    }
    /**
     * @return View|false
     */
    public function getNotFoundView() {
        if(($className = static::searchClass($this->moduleName,"NotFound","view")) !== false) {
            return new $className();
        } else {
            return new \boru\dhfw\app\basemodule\views\NotFound();
        }
    }
    /**
     * @return View|false
     */
    public function getNotAuthorizedView() {
        if(($className = static::searchClass($this->moduleName,"NotAuthorized","view")) !== false) {
            return new $className();
        } else {
            return new \boru\dhfw\app\basemodule\views\NotAuthorized();
        }
    }
    /**
     * @return Action|false
     */
    public function getAction() {
        if(is_null($this->actionName) || empty($this->actionName)) {
            return false;
        }
        if(is_null($this->action)) {
            if(($className = static::searchClass($this->moduleName,$this->actionName,"action")) !== false) {
                $this->action = new $className();
            } else {
                $this->action = new \boru\dhfw\app\basemodule\actions\NotFound();
            }
        }
        return $this->action;
    }
    /**
     * @return Api|false
     */
    public function getApi() {
        if(is_null($this->actionName) || empty($this->actionName)) {
            return false;
        }
        if(is_null($this->apiItem)) {
            if(($className = static::searchClass($this->moduleName,$this->actionName,"api")) !== false) {
                $this->apiItem = new $className();
            } else {
                $this->apiItem = new \boru\dhfw\app\basemodule\apis\NotFound();
            }
        }
        return $this->apiItem;
    }

    private function parseVars($vars) {
        $this->moduleName   = $vars->get( "module", null);
        $this->viewName     = $vars->get( "view",   null);
        $this->actionName   = $vars->get( "action", null);
        $this->recordIds     = $vars->get( "id",     null);
        $rest               = $vars->get( "rest",   []);
        if(!empty($rest)) {
            $this->rest = explode("/",$rest);
        }
    }

    public function getAll() {
        return $this->request->get();
    }
    public function get($key,$default=null) {
        return $this->request->get($key,$default);
    }
    public function request($key,$default=null) {
        return $this->request->get($key,$default);
    }
    public function getAllVars() {
        return $this->getAllArgs();
    }
    public function getVar($key,$default=null) {
        return $this->arg($key,$default);
    }
    public function getAllArgs() {
        return $this->args->get();
    }
    public function arg($key,$default=null) {
        return $this->args->get($key,$default);
    }
    public function header($key,$default=null) {
        return $this->api->header($key,$default);
    }
    public function attr($key,$default=null) {
        return $this->api->attr($key,$default);
    }
    public function baseUrl() {
        if(substr($this->baseUrl,-1) == "/") {
            return substr($this->baseUrl,0,-1);
        }
        return $this->baseUrl;
    }
    public function getPath() {
        return $this->api->getPath();
    }
    public function getMethod() {
        return $this->api->getRequest()->getMethod();
    }
    public function getArgs() {
        return $this->request;
    }
    public function getModuleName() {
        return $this->moduleName;
    }
    public function getViewName() {
        return $this->viewName;
    }
    public function getActionName() {
        return $this->actionName;
    }
    public function getRecordId($i=0) {
        return $this->recordIds[$i];
    }
    public function getRecordIds() {
        return $this->recordIds;
    }
    public function isAuthed() {
        return $this->isAuthed;
    }
    public function getRest() {
        return $this->rest;
    }

    private function checkLogin() {
        if($this->isAuthed) return true;
        if($this->getUser()->isAuthed()) {
            $this->isAuthed = true;
            return true;
        }
        return false;
    }


    private static $classNameFallback = [
        "{APPNS}\\{MODULE}",
        "\\boru\\dhfw\\app\\{MODULE}",
    ];
    private static $classNameBaseModule = [
        "{APPNS}\\basemodule",
        "\\boru\\dhfw\\app\\basemodule",
    ];

    public function getTemplateDirectories($moduleName=null) {
        if(is_null($moduleName)) $moduleName = $this->moduleName;
        $dirs = [];
        foreach(static::$classNameFallback as $className) {
            $classPath = static::parseClassToDir($className,$moduleName);
            $dirs[] = $classPath."/templates";
        }
        foreach(static::$classNameBaseModule as $className) {
            $classPath = static::parseClassToDir($className,$moduleName);
            $dirs[] = $classPath."/templates";
        }
        return $dirs;
    }

    public static function getModulePartDirs($moduleName,$part,$fallbackBaseModule=false) {
        $dirs = [];
        foreach(static::$classNameFallback as $className) {
            $classPath = static::parseClassToDir($className,$moduleName);
            $classPath.="/".strtolower($part)."s";
            if(is_dir($classPath)) {
                $dirs[] = $classPath;
            }
        }
        if($fallbackBaseModule) {
            foreach(static::$classNameBaseModule as $className) {
                $classPath = static::parseClassToDir($className,$moduleName);
                $classPath.="/".strtolower($part)."s";
                if(is_dir($classPath)) {
                    $dirs[] = $classPath;
                }
            }
        }
        return $dirs;
    }

    public static function searchClass($moduleName,$class,$type=null,$includeBaseModuleFallback=true) {
        $parts = !is_null($type) ? static::makePartSearch($class,$type) : [];
        $classSearch = static::$classNameFallback;
        if($includeBaseModuleFallback) {
            $classSearch = array_merge($classSearch,static::$classNameBaseModule);
        }
        foreach($classSearch as $fallbackClassNameTemplate) {
            if(!empty($parts)) {
                foreach($parts as $part) {
                    $nameSearch = static::parseClassNameString($fallbackClassNameTemplate,$moduleName);
                    $nameSearch .= "\\".strtolower($type)."s\\".$part;
                    Debug::addNote(990,"checking $nameSearch");
                    if(class_exists($nameSearch)) {
                        Debug::addNote(990,"found $nameSearch");
                        return $nameSearch;
                    }
                }
            } else {
                $nameSearch = static::parseClassNameString($fallbackClassNameTemplate,$moduleName);
                $nameSearch .= "\\".$class;
                Debug::addNote(990,"checking $nameSearch");
                if(class_exists($nameSearch)) {
                    Debug::addNote(990,"found $nameSearch");
                    return $nameSearch;
                }
            }
        }
        return false;
    }

    /**
     * 
     * @param string $moduleName 
     * @param string|null $type if null, will return the parent module directory, otherwise the type directory if it exists
     * @param bool $directoryObject return a Directory object(default) or just the path(false)
     * @param bool $includeBaseModuleFallback 
     * @return Directory|false|string 
     */
    public static function moduleDirectory($moduleName,$directoryObject=true,$includeBaseModuleFallback=true) {
        $classSearch = static::$classNameFallback;
        if($includeBaseModuleFallback) {
            $classSearch = array_merge($classSearch,static::$classNameBaseModule);
        }
        foreach($classSearch as $fallbackClassNameTemplate) {
            $nameSearch = static::parseClassNameString($fallbackClassNameTemplate,$moduleName);
            if(($dir = dhGlobal::dirIfExists(static::parseClassToDir($nameSearch,$moduleName),false)) !== false) {
                if($directoryObject) {
                    return $dir;
                }
                return $dir->path();
            }
        }
        return false;
    }

    /**
     * List all the modules available
     * @param string $type app|system|all
     * @return array 
     */
    public static function listModules($type="app") {
        if(strtolower($type) == "all") {
            $modules = static::listModules("system");
            $modules = array_merge($modules,static::listModules("app"));
            return $modules;
        } elseif(strtolower($type) == "system") {
            $directoryPath = DHFW::fwDir("src/app/");
        } else {
            $directoryPath = DHFW::dir("app/");
        }
        $directory = new Directory(["path"=>$directoryPath]);
        $modules = [];
        foreach($directory->dirs(false) as $mName=>$mPath) {
            if(($className = AppRouter::searchClass($mName,ucfirst($mName),null,false)) === false) {
                if(($className = AppRouter::searchClass($mName,"Module",null,false)) === false) {
                    continue;
                }
            }
            $modules[$mName] = $className;
        }
        return $modules;
    }

    private static $systemParts = [
        "\\boru\\dhfw\\app\\basemodule\\actions\\NotFoundAction",
        "\\boru\\dhfw\\app\\basemodule\\actions\\ExceptionAction",
        "\\boru\\dhfw\\app\\basemodule\\actions\\NotAuthorizedAction",

        "\\boru\\dhfw\\app\\basemodule\\apis\\NotAuthorizedApi",
        "\\boru\\dhfw\\app\\basemodule\\apis\\ExceptionApi",
        "\\boru\\dhfw\\app\\basemodule\\apis\\NotFoundApi",

        "\\boru\\dhfw\\app\\basemodule\\commands\\NotFoundCommand",
        "\\boru\\dhfw\\app\\basemodule\\commands\\ExceptionCommand",
        "\\boru\\dhfw\\app\\basemodule\\commands\\NotAuthorizedCommand",

        "\\boru\\dhfw\\app\\basemodule\\views\\NotFoundView",
        "\\boru\\dhfw\\app\\basemodule\\views\\ExceptionView",
        "\\boru\\dhfw\\app\\basemodule\\views\\NotAuthorizedView",
        
    ];
    /**
     * List all the types available for a module
     * @param string $module Module Name
     * @param string $type view|action|api|command|...
     * @return array 
     */
    public static function listModuleTypes($module,$type="view",$includeSystemParts=false) {
        $types = [];
        if(($moduleDir = static::moduleDirectory($module)) === false) {
            return $types;
        }
        $partDirs = static::getModulePartDirs($module,$type);
        if(empty($partDirs)) {
            return $types;
        }
        foreach($partDirs as $partDir) {
            if(($typeDir = dhGlobal::dirIfExists($partDir)) !== false) {
                foreach($typeDir->files(false) as $typeName=>$typePath) {
                    if(substr($typeName,-4) == ".php") {
                        $typeName = substr($typeName,0,-4);
                        if(($className = static::searchClass($module,$typeName,$type,false)) !== false) {
                            if(!$includeSystemParts && in_array($className,static::$systemParts)) {
                                continue;
                            }
                            $types[$typeName] = $className;
                        }
                    }
                }
            } else {
                
            }
        }
        ksort($types);
        return $types;
    }

    private static function makePartSearch($part,$type="view") {
        if(!is_null($type) && !empty($type)) {
            if(substr(strtolower($part),-(strlen($type))) == strtolower($type)) {
                $part = substr($part,0,-(strlen($type)));
            }
            $type = ucfirst($type);
        } else {
            $type = "";
        }
        $parts = [];
        $parts[] = ucfirst($part).$type;
        $parts[] = $part.$type;
        $parts[] = ucfirst($part);
        $parts[] = $part;
        return $parts;
    }
    public static function parseClassToDir($classString,$moduleName) {
        $rootDir = DHFW::getConfig("rootDirectory",__DIR__."/../");
        if(substr($rootDir,-1) !== "/") {
            $rootDir .= "/";
        }
        $className = static::parseClassNameString($classString,$moduleName);
        $className = str_replace("\\boru\\dhfw\\app\\",DHFW::fwDir("src/app/"),$className);
        $className = str_replace(static::getBaseNS(),DHFW::dir("app/"),$className);
        $className = str_replace("\\","/",$className);
        if(substr($className,0,1) == "/") {
            return $className;
        }
        return $rootDir.$className;
    }
    public static function parseClassNameString($classString,$moduleName) {
        $appns = self::getBaseNS();
        $className = str_replace("{APPNS}",$appns,$classString);
        $className = str_replace("{MODULE}",strtolower($moduleName),$className);
        return $className;
    }

    public static function setBaseNS($baseNS) {
        self::$baseNS = $baseNS;
    }
    public static function getBaseNS() {
        return self::$baseNS;
    }
    /**
     * Calls the DHFW::launch() method
     * @param string $baseNS
     * @param array $customRoutes
     * @deprecated use DHFW::launch() instead
     */
    public static function launch($baseNS=null,$customRoutes=[]) {
        DHFW::launch($baseNS,$customRoutes);
    }
}