<?php
namespace boru\dhfw;

use boru\dhapi\API;
use boru\dhapi\routing\Route;
use boru\dhapi\routing\Router;
use boru\dhdb\dhDB;
use boru\dhfw\util\Config;
use boru\dhfw\util\Debug;
use boru\dhfw\util\FWSetup;
use boru\dhsession\Session;
use \boru\dhutils\filesys\Directory;

class DHFW {
    /** @var Config */
    private static $config;
    /** @var Directory */
    private static $rootDir;

    public static $externalSmarty = false;
    /**
     * Initialize the framework
     * @param Directory $rootDirectory 
     * @return void 
     */
    public static function init($rootDirectory=null) {
        if(is_null($rootDirectory)) {
            $rootDirectory = new \boru\dhutils\filesys\Directory(["path"=>__DIR__.'/../']);
        } else {
            $rootDirectory == Directory::fromInput($rootDirectory);
        }
        if(!($rootDirectory instanceof Directory)) {
            throw new \Exception("DHFW::init() requires a Directory object");
        }
        static::$rootDir = $rootDirectory;
        static::$config = FWSetup::checkConfig(static::dir("config.json"));

        if(isset($_REQUEST)) {
            if(isset($_REQUEST["fwdebug"]) && $_REQUEST["fwdebug"]>0) {
                if($_REQUEST["fwdebug"]>1) {
                    Debug::enable($_REQUEST["fwdebug"]);
                } else {
                    Debug::enable();
                }
            }
        }
    }

    public static function initConfig() {
        if(is_null(static::$config)) {
            static::$config = FWSetup::checkConfig(static::dir("config.json"));
        }
    }


    // Config shortcuts
    /** 
     * Get a config value by dot notation, or the entire config array. Returns $default if not found
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function getConfig($key=null,$default=null) {
        static::initConfig();
        return static::$config->get($key,$default);
    }

    /**
     * Set a config value by dot notation
     * @param mixed $key 
     * @param mixed $value 
     * @return Config 
     */
    public static function setConfig($key,$value) {
        static::initConfig();
        return static::$config->set($key,$value);
    }

    /**
     * Save the config file
     * @return Config 
     */
    public static function saveConfig() {
        static::initConfig();
        return static::$config->save();
    }

    /**
     * Get the config object
     * @return Config 
     */
    public static function config() {
        static::initConfig();
        return static::$config;
    }

    /**
     * Get the root directory path string, or a path relative to the root directory
     * @param mixed $path 
     * @return string 
     */
    public static function dir($path=null) {
        return static::$rootDir->path($path);
    }

    private static $fwDir;
    public static function setFwDir($dir) {
        static::$fwDir = $dir;
    }
    public static function fwDir($path=null) {
        if(is_null(static::$fwDir)) {
            static::$fwDir = new Directory(["path"=>__DIR__."/../"]);
        }
        return static::$fwDir->path($path);
    }

    private static $db;
    public static function db($db=null) {
        if(!is_null($db)) {
            static::$db = $db;
        }
        if(is_null(static::$db)) {
            static::$db = new dhDB(["config"=>static::config()->get("db")]);
        }
        return static::$db;
    }

    public static function install($directory=null,$force=false) {
        //make sure we're running from cli and not a webserver
        if(php_sapi_name() !== "cli") {
            throw new \Exception("DHFW::install() must be run from the command line");
        }
        FWSetup::install($directory,$force);
    }

    private static $requestMiddleware = [];
    public static function addRequestMiddleware($routeType="api",$middleware) {
        if(!isset(static::$requestMiddleware[$routeType])) {
            static::$requestMiddleware[$routeType] = [];
        }
        static::$requestMiddleware[$routeType][] = $middleware;
    }

    public static function launch($baseNS=null,$customRoutes=[]) {
        if(is_null($baseNS)) {
            $baseNS = DHFW::getConfig("baseNamespace","\\FWAPP");
        }
        if(is_null($customRoutes)) {
            $customRoutes = DHFW::getConfig("customRoutes",[]);
        }

        $apiMethods = ["GET","POST","PUT","DELETE"];
        $apiCallable = ["\\boru\\dhfw\\AppRouter","processApi"];
        $apiOptions = [];
        if(isset(static::$requestMiddleware["api"])) {
            $apiOptions["middleware"] = static::$requestMiddleware["api"];
        }
        $webMethods = ["GET","POST","PUT"];
        $webCallable = ["\\boru\\dhfw\\AppRouter","processWeb"];
        $webOptions = [];
        if(isset(static::$requestMiddleware["web"])) {
            $webOptions["middleware"] = static::$requestMiddleware["web"];
        }

        $apiRoutes = [
            "/api/{module}/{id:idlist}/{action}/{rest:all}",
            "/api/{module}/{id:idlist}/{action}",
            "/api/{module}/{id:idlist}",
            "/api/{module}/{action}/{rest:all}",
            "/api/{module}/{action}",
            "/api/{module}",
            "/api"
        ];
        $webRoutes = [
            "/{module}/{id:idlist}/a/{action}/{rest:all}",
            "/{module}/{id:idlist}/a/{action}",
            "/{module}/{id:idlist}/a",

            "/{module}/a/{action}/{rest:all}",
            "/{module}/a/{action}",

            "/{module}/{id:idlist}/{view}/{rest:all}",
            "/{module}/{id:idlist}/{view}",
            "/{module}/{id:idlist}",

            "/{module}/{view}/{rest:all}",
            "/{module}/{view}",
            "/{module}",
            "/"
        ];
        Session::setHandler("dhdb",["db"=>DHFW::db(),"tableName"=>DHFW::getConfig("sessionTable","sessions"),"expiry"=>DHFW::getConfig("sessionExpiry",3600)]);

        AppRouter::setBaseNS($baseNS);
        $router = new Router();
        //API
        $i=0;
        foreach($apiRoutes as $apiRoute) {
            $i++;
            $router->add(new Route("api-$i", $apiRoute, $apiMethods, $apiCallable, $apiOptions));
        }
        //WEB
        $i=0;
        foreach($webRoutes as $webRoute) {
            $i++;
            $router->add(new Route("web-$i", $webRoute, $webMethods, $webCallable, $webOptions));
        }
        $request = API::requestFromGlobals();
        $api = new API($request,$router);
        $response = $api->process();
        $response->emit(true);
    }


    private static $moduleClassCache = [];
    public static function getModuleClass($moduleName) {
        if(isset(static::$moduleClassCache[$moduleName])) {
            return static::$moduleClassCache[$moduleName];
        }
        if(($className = AppRouter::searchClass($moduleName,ucfirst($moduleName))) !== false) {
            static::$moduleClassCache[$moduleName] = new $className();
        } elseif(($className = AppRouter::searchClass($moduleName,$moduleName)) !== false) {
            static::$moduleClassCache[$moduleName] = new $className();
        } elseif(($className = AppRouter::searchClass($moduleName,"Module")) !== false) {
            static::$moduleClassCache[$moduleName] = new $className();
        } else {
            static::$moduleClassCache[$moduleName] = new \boru\dhfw\app\basemodule\Module();
        }
        if(is_null(static::$moduleClassCache[$moduleName])) {
            static::$moduleClassCache[$moduleName] = false;
        }
        return static::$moduleClassCache[$moduleName];
    }

    //static record helpers
    /**
	 * @param string $moduleName
	 * @param string $field 
	 * @param string $value 
	 * @param int $offset 
	 * @param int $limit 
	 * @param string $orderby 
	 * @param string $orderdir 
	 * @param string $groupby 
	 * @param string $comparator 
	 * @return false|static[] 
	 */
	public static function search($moduleName,$field="",$value="",$offset=0,$limit=0,$orderby="",$orderdir="",$groupby="",$comparator="eq") {
        if(($moduleClass = static::getModuleClass($moduleName)) !== false) {
            return $moduleClass::search($field,$value,$offset,$limit,$orderby,$orderdir,$groupby,$comparator);
        } else {
            return false;
        }
    }
    /**
	 * @param string $moduleName
	 * @param mixed $field 
	 * @param int $offset 
	 * @param int $limit 
	 * @param string $orderby 
	 * @param string $orderdir 
	 * @return false|array 
	 */
	public static function searchDistinct($moduleName,$field,$offset=0,$limit=0,$orderby="",$orderdir="") {
        if(($moduleClass = static::getModuleClass($moduleName)) !== false) {
            return $moduleClass::searchDistinct($field,$offset,$limit,$orderby,$orderdir);
        } else {
            return false;
        }
    }

    /**
	 * @param string $moduleName
	 * @param mixed $field 
	 * @param string $value 
	 * @param string $comparator 
	 * @return false|static 
	 */
	public static function getOne($moduleName,$field,$value="",$comparator="eq") {
		return static::search($moduleName,$field,$value,0,1,"","","",$comparator);
    }

    /**
	 * @param string $moduleName
	 * @param array $array 
	 * @return static 
	 */
	public static function create($moduleName,$array=array()) {
        if(($moduleClass = static::getModuleClass($moduleName)) !== false) {
            return $moduleClass::create($array);
        } else {
            return false;
        }
    }

    /**
	 * 
	 * @param mixed $array 
	 * @param string $comparator 
	 * @return static 
	 */
	public static function getOrCreate($moduleName,$array,$comparator="eq") {
        if(($moduleClass = static::getModuleClass($moduleName)) !== false) {
            return $moduleClass::getOrCreate($array,$comparator);
        } else {
            return false;
        }
    }

    /**
	 * 
	 * @param mixed $value 
	 * @return false|static 
	 */
	public static function getById($moduleName,$value) {
        if(($moduleClass = static::getModuleClass($moduleName)) !== false) {
            return $moduleClass::getById($value);
        } else {
            return false;
        }
    }
}