<?php
namespace boru\dhfw\base;

use boru\dhfw\DHFW;
use boru\dhfw\util\Debug;
use boru\dhfw\util\QueryConditions;

abstract class BaseModuleRecord {
    /**
     * The name of the module
     * @var string
     */
    protected static $moduleName = "Module";
    /**
     * The name of the primary table in the database
     */
    protected static $sqlTable = "";

    /**
     * The name of the tables in the database
     * @var array<string>
     */
    protected static $sqlExtraTables = [];

    /**
     * The join clauses for the $sqlExtraTables
     * * Syntax: "tablename"=>"tablename.id = othertablename.id"
     * @var array
     */
    protected static $sqlTableJoins = [];

    /**
     * The name of the primary key field in the database
     * @var string
     */
    protected static $idField = "";

    /**
     * The name of the createdTime field in the database, blank to disable
     */
    protected static $createdTimeField = "";

    /**
     * The name of the modifiedTime field in the database, blank to disable
     */
    protected static $modifiedTimeField = "";

    /**
     * The name of the label field, or the sql to generate it
     */
    protected static $labelField = "";

    protected static $columnDefinitions = [];
    protected static $columns = [];
    protected static $fieldTableColumnMap = [];

    public static function initSqlDefaults() {
        if(static::$sqlTable == "") {
            static::$sqlTable = strtolower(static::$moduleName);
        }
        if(static::$idField == "") {
            static::$idField = "id";
        }
        if(static::$createdTimeField == "") {
            static::$createdTimeField = "createdTime";
        }
        if(static::$modifiedTimeField == "") {
            static::$modifiedTimeField = "modifiedTime";
        }
        if(static::$labelField == "") {
            static::$labelField = "label";
        }
        static::SQLInitColumnDefinitions();
    }

    public static function getColumns() {
        if(empty(static::$columns)) {
            static::SQLInitColumnDefinitions(true);
        }
        return static::$columns;
    }
    public static function getFieldMap() {
        if(empty(static::$fieldTableColumnMap)) {
            static::SQLInitColumnDefinitions(true);
        }
        return static::$fieldTableColumnMap;
    }

    private static function SQLInitColumnDefinitions($force=false) {
        if(count(static::$columnDefinitions) > 0 && !$force) {
            return;
        }
        if(empty(static::$sqlTable)) {
            return;
        }
        $tables = [static::$sqlTable];
        if(!empty(static::$sqlExtraTables)) {
            $tables = array_merge($tables, static::$sqlExtraTables);
        }
        foreach($tables as $table) {
            $db = DHFW::db();
            $sql = "SHOW columns from `".$table."`";
            $sth = $db->run($sql);
            while($row=$db->next($sth)) {
                Debug::addNote(1,"Column: ".$row["Field"]);
                $rowArr = $row->toArray();
                static::$columnDefinitions[$table][$rowArr["Field"]] = $rowArr;
            }
        }
        static::SQLColDefToMap();
        return static::$columnDefinitions;
    }
    private static function SQLColDefToMap() {
        if(empty(static::$columnDefinitions)) {
            static::SQLInitColumnDefinitions();
        }
        if(empty(static::$fieldTableColumnMap)) {
            foreach(static::$columnDefinitions as $table=>$cols) {
                foreach($cols as $col=>$def) {
                    static::$columns[$table][] = $col;
                    static::$fieldTableColumnMap[$col][] = $table;
                }
            }
        }
        return static::$fieldTableColumnMap;
    }
    
    public static function SQLTableWithJoins() {
        $joins = [];
        $joins[] = "`".static::$sqlTable."`";
        if(is_array(static::$sqlTableJoins) && !empty(static::$sqlTableJoins)) {
            foreach(static::$sqlTableJoins as $table=>$join) {
                $joins[] = "INNER JOIN `".$table."` ON ".$join;
            }
        }
        return implode(" ", $joins);
    }

    public static function SQLBuildSetClause($columns=[]) {
        $sql = "";
        $values = [];
        foreach($columns as $col=>$val) {
            if(isset(static::$fieldTableColumnMap[$col])) {
                $tables = static::$fieldTableColumnMap[$col];
                foreach($tables as $table) {
                    $sql .= "`".$table."`.`".$col."` = ?, ";
                    $values[] = $val;
                }
            }
        }
        $sql = rtrim($sql, ", ");
        return ["sql"=>$sql,"values"=>$values];
    }
    public static function SQLBuildConditionClause($conditions=[]) {
        $sql = "";
        $values = [];
        foreach($conditions as $col=>$val) {
            if(isset(static::$fieldTableColumnMap[$col])) {
                $tables = static::$fieldTableColumnMap[$col];
                foreach($tables as $table) {
                    $sql .= "`".$table."`.`".$col."` = ? AND ";
                    $values[] = $val;
                }
            }
        }
        $sql = rtrim($sql, " AND ");
        return ["sql"=>$sql,"values"=>$values];
    }
    public static function SQLInsertQuery($columns=[],$dateFields=false) {
        $sql = "INSERT INTO ".static::SQLTableWithJoins()." SET ";
        $setClause = static::SQLBuildSetClause($columns);
        $sql .= $setClause["sql"];
        $values = $setClause["values"];
        if($dateFields && !is_null(static::$createdTimeField) && !in_array(static::$createdTimeField, $columns)) {
            $sql .= ", `".static::$createdTimeField."` = NOW()";
        }
        if($dateFields && !is_null(static::$modifiedTimeField) && !in_array(static::$modifiedTimeField, $columns)) {
            $sql .= ", `".static::$modifiedTimeField."` = NOW()";
        }
        return ["sql"=>$sql,"values"=>$values];

    }
    public static function SQLUpdateQuery($columns=[], $conditions=[], $dateFields=false) {
        $sql = "UPDATE ".static::SQLTableWithJoins()." SET ";
        $setClause = static::SQLBuildSetClause($columns);
        $sql .= $setClause["sql"];
        $values = $setClause["values"];
        if($dateFields && !is_null(static::$modifiedTimeField) && !in_array(static::$modifiedTimeField, $columns)) {
            $sql .= ", `".static::$modifiedTimeField."` = NOW()";
        }
        $whereConditions = new QueryConditions();
		$whereConditions->addArray($conditions);
		
        $sql.=" ".$whereConditions->getSQL("WHERE");
		$values = array_merge($values,$whereConditions->getValues());
        $whereClause = static::SQLBuildConditionClause($conditions);
        return ["sql"=>$sql,"values"=>$values];

    }
    public static function SQLDeleteQuery($conditions=[]) {
        $sql = "DELETE FROM ".static::SQLTableWithJoins();
        $whereClause = static::SQLBuildConditionClause($conditions);
        $sql .= " WHERE ".$whereClause["sql"];
        $values = $whereClause["values"];
        return ["sql"=>$sql,"values"=>$values];
    }
}