<?php
namespace boru\dhfw\base;

use boru\dhfw\AppRouter;
use boru\dhfw\DHFW;
use boru\dhfw\util\Debug;

abstract class BaseView {

    /** @var AppRouter */
    protected $appRouter;

    protected $displayHeader = true;
    protected $displayFooter = true;
    protected $requiresAuth  = true;

    /** @var \boru\dhfw\viewer\Viewer */
    protected $viewer;

    protected $templates = [
        "header"=>"Header.tpl",
        "footer"=>"Footer.tpl",
        "preHeader"=>"HeaderPre.tpl",
        "postHeader"=>"HeaderPost.tpl",
        "preProcess"=>"BodyPre.tpl",
        "postProcess"=>"BodyPost.tpl",
        "preFooter"=>"FooterPre.tpl",
        "postFooter"=>"FooterPost.tpl",
        "javascripts"=>"Javascripts.tpl",
    ];

    private $stylesheets = [

    ];

    protected $headerJavascripts = [

    ];
    protected $footerJavascripts = [

    ];

    protected $javascriptVariables = [
        
    ];

    public function isPermissed() {
        return true;
    }
    abstract public function init();

    /** @param AppRouter $appRouter */
    public function trigger($appRouter) {
        $this->appRouter = $appRouter;

        if(!$this->get("isModal",false)) {
            $this->stylesheets = array_merge(DHFW::getConfig("stylesheets",[]),$this->stylesheets);
            $this->headerJavascripts = array_merge(DHFW::getConfig("headerJavascripts",[]),$this->headerJavascripts);
            $this->footerJavascripts = array_merge(DHFW::getConfig("footerJavascripts",[]),$this->footerJavascripts);
        }

        if($this->requiresAuth) {
            if(!$this->appRouter->isAuthed()) {
                $handler = $this->appRouter->getLoginView();
                $handler->setLoginMessage("You must be logged in to access this page.");
                $handler->trigger($this->appRouter);
                return;
            }
        }
        if(!$this->isPermissed()) {
            $handler = $this->appRouter->getNotAuthorizedView();
            $handler->trigger($this->appRouter);
            return;
        }

        if(is_null($this->viewer)) {
            $this->viewer = new \boru\dhfw\viewer\Viewer($this->appRouter);
        }
        //set default variables
        
        $this->assignVariables();
        $this->init();
        $this->assignPostInitVariables();
        if($this->displayHeader) {
            $this->preHeader();
            $this->header();
            $this->javascripts();
            $this->postHeader();
        }
        $this->preProcess();
        $this->process();
        $this->postProcess();
        if($this->displayFooter) {
            $this->preFooter();
            $this->footer();
            $this->postFooter();
        }
    }
    public function preHeader() {
        $this->displayPart("preHeader");
    }
    public function header() {
        $this->displayPart("header");
    }
    public function postHeader() {
        $this->displayPart("postHeader");
    }
    public function javascripts() {
        $this->viewer->assign("JSVARIABLES",$this->javascriptVariables);
        $this->displayPart("javascripts");
    }
    public function preProcess() {
        $this->displayPart("preProcess");
    }
    abstract public function process();
    public function postProcess() {
        $this->displayPart("postProcess");
    }
    public function preFooter() {
        $this->displayPart("preFooter");
    }
    public function footer() {
        $this->displayPart("footer");
    }
    public function postFooter() {
        $this->displayPart("postFooter");
    }


    public function setTemplate($key,$value) {
        $this->templates[$key] = $value;
    }
    public function getTemplate($key) {
        return $this->templates[$key];
    }

    protected function displayPart($part) {
        if(is_null($this->viewer)) {
            $this->viewer = new \boru\dhfw\viewer\Viewer($this->appRouter);
        }
        if(isset($this->templates[$part]) && $this->templates[$part] !== "") {
            $this->viewer->display($this->templates[$part]);
        }
    }

    public function assignVariables() {
        $viewName = $this->appRouter->getViewName();
        $actionName = $this->appRouter->getActionName();
        $moduleName = $this->appRouter->getModuleName();
        if(is_null($viewName)) {
            $viewName = "";
        }
        if(is_null($actionName)) {
            $actionName = "";
        }
        if(is_null($moduleName)) {
            $moduleName = "";
        }
        $this->viewer->assign("module",     $this->appRouter->getModule());
        $this->viewer->assign("user",       $this->appRouter->getUser());
        $this->viewer->assign("baseUrl",    $this->viewer->baseUrl());
        $this->viewer->assign("appRouter",  $this->appRouter);
        $this->viewer->assign("path",       $this->appRouter->getPath());
        $this->viewer->assign("viewName",   strtolower($viewName));
        $this->viewer->assign("actionName", strtolower($actionName));
        $this->viewer->assign("moduleName", strtolower($moduleName));

        $this->addJsVariable("APPURL",$this->viewer->baseUrl());
        $this->addJsVariable("APPVIEW",strtolower($viewName));
        $this->addJsVariable("APPACTION",strtolower($actionName));
        $this->addJsVariable("APPMODULE",strtolower($moduleName));
        $this->addJsVariable("APPPATH",$this->appRouter->getPath());

        
    }
    public function assignPostInitVariables() {
        $this->viewer->assign("STYLESHEETS",$this->stylesheets);
        $this->viewer->assign("HEADERJAVASCRIPTS",$this->headerJavascripts);
        $this->viewer->assign("FOOTERJAVASCRIPTS",$this->footerJavascripts);
        $this->viewer->assign("isModal",$this->get("isModal",false) ? true : false);
    }

    public function module() {
        return $this->appRouter->getModule();
    }
    public function recordId() {
        return $this->appRouter->getRecordId();
    }
    public function get($key,$defaultValue=null) {
        return $this->appRouter->get($key,$defaultValue);
    }
    public function request($key,$defaultValue=null) {
        return $this->appRouter->get($key,$defaultValue);
    }
    public function attr($key,$defaultValue=null) {
        return $this->appRouter->attr($key,$defaultValue);
    }
    public function getAll() {
        return $this->appRouter->getAll();
    }
    public function arg($key,$defaultValue=null) {
        return $this->appRouter->arg($key,$defaultValue);
    }
    public function getAllArgs() {
        return $this->appRouter->getAllArgs();
    }
    public function getVar($key,$defaultValue=null) {
        return $this->appRouter->arg($key,$defaultValue);
    }
    public function getAllVars() {
        return $this->appRouter->getAllArgs();
    }
    public function head($key,$defaultValue=null) {
        return $this->appRouter->header($key,$defaultValue);
    }

    public function addJsVariable($key,$value) {
        $this->javascriptVariables[$key] = $value;
    }
    public function addStylesheet($href,$attributes=[]) {
        $this->stylesheets[] = ["href"=>$href,"attributes"=>$attributes];
    }
    public function addHeaderJavascript($src,$attributes=[]) {
        $this->headerJavascripts[] = ["src"=>$src,"attributes"=>$attributes];
    }
    public function addFooterJavascript($src,$attributes=[]) {
        $this->footerJavascripts[] = ["src"=>$src,"attributes"=>$attributes];
    }
}