<?php
namespace boru\dhfw\viewer;

use boru\dhfw\DHFW;
use boru\dhfw\util\Debug;
use boru\dhutils\dhGlobal;
use Exception;

class Viewer {
    /** @var \Smarty */
    protected $smarty;

    /** @var \boru\dhfw\AppRouter */
    protected $appRouter;

    protected $directories = [];

    private static $instance;

    /**
     * @return Viewer
     */
    public static function getInstance() {
        return static::$instance;
    }

    /** @param \boru\dhfw\AppRouter $appRouter */
    public function __construct($appRouter,$oneOff=false) {
        $this->appRouter = $appRouter;
        $this->smarty = \boru\dhfw\util\FWSmarty::instance();
        $this->directories = $appRouter->getTemplateDirectories();
        $this->smarty->assign("viewer",$this);
        $this->smarty->assign("v",$this);
        $this->smarty->assign("_",$this);
        $this->smarty->assign("appRouter",  $this->appRouter);
        if(!$oneOff) {
            static::$instance = $this;
        }
    }

    public function templatesFor($moduleName) {
        $this->directories = $this->appRouter->getTemplateDirectories($moduleName);
    }

    public function assign($key,$value) {
        $this->smarty->assign($key,$value);
    }
    public function display($templateName,$fetch=false) {
        if($fetch) {
            return $this->fetch($templateName);
        }
        $template = $this->findTemplate($templateName);
        if(!$template) {
            throw new \Exception("Template not found: ".$templateName);
        }
        $this->smarty->display($template);
    }
    public function displayField($field,$viewType="Display") {
        $template = $this->fieldTemplate($field->type(),$viewType);
        $this->smarty->assign("field",$field);
        $this->smarty->display($template);
    }
    public function fetch($template) {
        $template = $this->findTemplate($template);
        if(!$template) {
            throw new \Exception("Template not found: ".$template);
        }
        return $this->smarty->fetch($template);
    }

    public function template($file) {
        return $this->findTemplate($file);
    }

    public function resource($file) {
        $url = DHFW::getConfig("publicUrl",false);
        if($url === false) {
            $url = DHFW::getConfig("rootUrl",$this->appRouter->baseUrl());
            if(substr($url,-1)!="/") {
                $url .= "/";
            }
            $pubPath = DHFW::getConfig("publicPath","");
            if($pubPath) {
                $url .= $pubPath;
            }
        }
        if(substr($url,-1)!="/") {
            $url .= "/";
        }
    
        $url .= $file;
        return $url;
    }
    public function baseUrl() {
        return DHFW::getConfig("rootUrl",$this->appRouter->baseUrl());
    }
    
    /**
     * Get the template to display for a fieldType for a viewType
     * @param mixed $fieldType Example: Text, Select, Checkbox, Textarea..
     * @param string $viewType Example: Display, Edit, List..
     * @return string template file to display/include
     * @throws Exception 
     */
    public function fieldTemplate($fieldType,$viewType="Display") {
        $template = $this->findTemplate("fields/".strtolower($fieldType)."/".$viewType.".tpl");
        if(!$template) {
            $template = $this->findTemplate("fields/".strtolower($fieldType)."/".ucfirst($viewType).".tpl");
            if(!$template) {
                $template = $this->findTemplate("fields/field/".ucfirst($viewType).".tpl");
                if(!$template) {
                    throw new \Exception("Template not found: fields/".strtolower($fieldType)."/".$viewType.".tpl");
                }
            }
        }
        return $template;
    }
    public function viewUrl($module,$view,$record=null,$params=[]) {
        $url = $this->baseUrl();
        if(substr($url,-1)!="/") {
            $url .= "/";
        }
        $url .= strtolower($module);
        if($record) {
            $url .= "/".$record;
        }
        $url .= "/".$view;
        $url.= $this->paramString($params);
        return $url;
    }
    public function actionUrl($module,$action,$record=null,$params=[]) {
        $url = $this->baseUrl();
        if(substr($url,-1)!="/") {
            $url .= "/";
        }
        $url .= $module;
        if($record) {
            $url .= "/".$record;
        }
        $url .= "/".$action;
        $url.= $this->paramString($params);
        return $url;
    }
    public function makeUrl($href="",$params=[]) {
        $url = $this->baseUrl();
        if(substr($url,-1)!="/" && substr($href,0,1)!="/") {
            $url .= "/";
        }
        $url .= $href;
        $url.= $this->paramString($params);
        return $url;
    }
    public function view() {
        $viewName = $this->appRouter->getViewName();
        if(is_null($viewName)) {
            return false;
        } else {
            return $viewName;
        }
    }
    public function action() {
        $actionName = $this->appRouter->getActionName();
        if(is_null($actionName)) {
            return false;
        } else {
            return $actionName;
        }
    }
    public function record() {
        $record = $this->appRouter->getRecordId();
        if(is_null($record)) {
            return false;
        } else {
            return $record;
        }
    }
    

    private function findTemplate($template) {
        if(file_exists($template)) {
            return $template;
        }
        foreach($this->directories as $dir) {
            Debug::addNote(990,"Searching in ".$dir." for ".$template);
            $path = $dir."/".$template;
            if(file_exists($path)) {
                Debug::addNote(990,"found: ".$path);
                return $path;
            }
        }
        foreach($this->directories as $dir) {
            Debug::addNote(990,"Searching in ".$dir." for ".ucfirst($template));
            $path = $dir."/".ucfirst($template);
            
            if(file_exists($path)) {
                Debug::addNote(990,"found: ".$path);
                return $path;
            }
        }
        Debug::addNote(990,"found nothing for: ".$template);
        return false;
    }

    private function paramString($params=[]) {
        $string = "";
        if(is_array($params) && count($params)>0) {
            $string .= "?";
            if(dhGlobal::isList($params)) {
                $string .= implode("&",$params);
            } else {
                $string .= http_build_query($params);
            }
        } elseif(is_string($params) && $params !== "") {
            $string .= "?".$params;
        }
        return $string;
    }
}