<?php
namespace boru\dhfw\base;

use boru\dhfw\app\basemodule\fields\Field;
use boru\dhfw\AppRouter;
use boru\dhfw\DHFW;
use boru\dhfw\util\Container;
use boru\dhfw\util\Debug;
use boru\dhfw\util\QueryBuilder;
use boru\dhfw\util\QueryConditions;
use boru\dhutils\dhGlobal;
use boru\dhutils\filesys\Directory;
use ReflectionClass;

abstract class BaseModule extends BaseModuleRecord {

    /**
     * this is version 2 of the moduleDef process.. instead of using a bunch of static variables, we use a moduleDef array.
     * this allows us to use the same module class for multiple modules, and also allows us to use the same module class for multiple tables
     * it will also eventually allow us to load the moduleDef from a file and/or table, which will allow us to create modules without having to create a class for them
     */
    protected $moduleDefinition = [];

    //all of these variables are commented and defined in BaseModuleRecord
    /**
     * These are version 1 of the moduleDef process.. they are still used in the BaseModuleRecord class.
     * The order of precedence is:
     * 1. moduleDefinition
     * 2. static variables
     */
    protected static $moduleName = "Module";
    protected static $isRecordModule = true;
    protected static $sqlTable = "";
    protected static $sqlExtraTables = [];
    protected static $sqlTableJoins = [];
    protected static $idField = "";
    protected static $createdTimeField = "";
    protected static $modifiedTimeField = "";
    protected static $labelField = "";

    protected static $jsonFields = [];
    protected static $serializedFields = [];

    protected static $fieldDefs = [];
    protected static $useFieldsTable = false;

    
    protected static $Static_SQLDebug = false;
    /**
     * The default view to use for this module
     * @var string
     */
    protected static $defaultView = "Home";

    public static function getDefaultViewName() {
        static::initDefaults();
        return static::$defaultView;
    }

    public static function setSqlDebug($debug) {
        static::$Static_SQLDebug = $debug;
    }

    public static function count($field="",$value="",$offset=0,$limit=0,$orderby="",$orderdir="",$groupby="",$comparator="eq") {
        return static::search($field,$value,$offset,$limit,$orderby,$orderdir,$groupby,$comparator,true);
    }

    public static function queryBuilder() {
        return new QueryBuilder(static::getModuleName());
    }

    /**
	 * 
	 * @param string $field 
	 * @param string $value 
	 * @param int $offset 
	 * @param int $limit 
	 * @param string $orderby 
	 * @param string $orderdir 
	 * @param string $groupby 
	 * @param string $comparator 
	 * @return false|static[] 
	 */
	public static function search($field="",$value="",$offset=0,$limit=0,$orderby="",$orderdir="",$groupby="",$comparator="eq",$asCount=false) {
	    $db = DHFW::db();
		$sql = "SELECT * FROM ";
        if($asCount) {
            $sql = "SELECT COUNT(*) as cnt FROM ";
        }
        $sql.= static::SQLTableWithJoins();
        $values = [];
        if($field instanceof QueryConditions) {
            $whereConditions = $field;
        } else {
            $whereConditions = new QueryConditions();
            if(is_array($field) || !empty($field)) {
                if(is_array($field)) {
                    $whereConditions->addArray($field);
                } else {
                    $whereConditions->add($field,$value,$comparator);
                }
                $sql.=" ".$whereConditions->getSQL("WHERE");
                $values = $whereConditions->getValues();
            } else {
                
            }
        }
		if($orderby != "") {
			$sql.=" ORDER BY $orderby $orderdir";
		}
		if($limit>=1) {
			$sql.=" LIMIT ?,?";
			$values[] = $offset;
			$values[] = $limit;
		}
        if(static::$Static_SQLDebug) {
            dhGlobal::outLine("SQLDebug(static)",$sql,"::::",$values);
        }
		$sth=$db->run($sql,$values);
        if(!$asCount) {
            $return = array();
            while($row=$db->next($sth)) {
                $obj = new static();
                $obj->populate($row->toArray());
                $return[] = $obj;
            }
            if(empty($return)) return false;
            return $return;
        } else {
            $row = $db->next($sth);
            return $row["cnt"];
        }
	}

    /**
	 * 
	 * @param mixed $field 
	 * @param int $offset 
	 * @param int $limit 
	 * @param string $orderby 
	 * @param string $orderdir 
	 * @return false|array 
	 */
	public static function searchDistinct($field,$offset=0,$limit=0,$orderby="",$orderdir="") {
		$db = DHFW::db();
		$sql = "select DISTINCT `".$field."` FROM ";
        $sql.= static::SQLTableWithJoins();
		$values = array();
		if($limit>=1) {
			$sql.=" LIMIT ?,?";
			$values[] = $offset;
			$values[] = $limit;
		}
		if($orderby != "") {
			$sql.=" ORDER BY $orderby $orderdir";
		}
        if(static::$Static_SQLDebug) {
            dhGlobal::outLine("SQLDebug(static)",$sql,"::::",$values);
        }
		$sth=$db->run($sql,$values);
		$return = array();
		while($row=$db->next($sth)) {
			$return[] = $row[$field];
		}
		if(empty($return)) return false;
		return $return;
	}

    /**
	 * 
	 * @param mixed $field 
	 * @param string $value 
	 * @param string $comparator 
	 * @return false|static 
	 */
	public static function getOne($field,$value="",$comparator="eq") {
		$data = static::search($field,$value,0,1,"","","",$comparator);
		if(empty($data)) return false;
		return $data[0];
	}

	/**
	 * 
	 * @param array $array 
	 * @return static 
	 */
	public static function create($array=array()) {
		global $dbh;
		$obj = new static();
		if(!empty($array)) {
			$obj->populate($array);
			$obj->save(true);
			$obj->setNew(true);
		}
		return $obj;
	}

    /**
	 * 
	 * @param mixed $array 
	 * @param string $comparator 
	 * @return static 
	 */
	public static function getOrCreate($array,$comparator="eq") {
		$data = static::getOne($array,"",$comparator);
		if(!$data) {
			$data = static::create();
			$data->populate($array);
			$data->save(true);
			$data->setNew(true);
		}
		return $data;
	}

    /**
	 * 
	 * @param mixed $value 
	 * @return false|static 
	 */
	public static function getById($value) {
		return static::getOne(static::$idField,$value,"eq");
	}

    private $debugSql = false;
    protected $id;
    protected $data;
    protected $isNew;
    protected $fields = [];
    public function getModuleDefinition($key=null,$default=null) {
        if(!is_array($this->moduleDefinition)) {
            $this->moduleDefinition = [];
        }
        if(is_null($key)) {
            return $this->moduleDefinition;
        }
        if(isset($this->moduleDefinition[$key])) {
            return $this->moduleDefinition[$key];
        }
        return $default;
    }
    public function get($key=null,$default=null) {
        return $this->data->get($key,$default);
    }
    public function set($key,$value) {
        $this->data->set($key,$value);
    }
    public function setNew($new) {
        $this->isNew = $new;
    }
    public function isNew() {
        return $this->isNew;
    }
    public function setId($id) {
        $this->id = $id;
    }
    public function id() {
        return $this->id;
    }
    public function label() {
        $labelField = static::$labelField;
        if($labelField == "") {
            return $this->id();
        }
        return $this->get($labelField,"");
    }
    public function moduleName() {
        return static::$moduleName;
    }
    public function debugSql($debug=true) {
        $this->debugSql = $debug;
    }
    /**
     * @param mixed $fieldName 
     * @return Field[] 
     */
    public function fields() {
        return $this->fields;
    }
    /**
     * @param mixed $fieldName 
     * @return Field|false 
     */
    public function field($fieldName) {
        if(isset($this->fields[$fieldName])) {
            return $this->fields[$fieldName];
        }
        return false;
    }

    public function __construct($moduleName=null) {
        if(!is_null($moduleName)) {
            static::$moduleName = $moduleName;
        }
        $this->data = new Container();
        static::initDefaults(static::$moduleName,$this);
    }

    public function delete() {
        $this->_delete();
    }
    public function _delete() {
        $db = DHFW::db();
        $sql = "DELETE FROM ".static::$sqlTable." WHERE ".static::$idField." = ?";
        $db->run($sql,[$this->id()]);
    }

    //Overridable save action
	public function save($date=true) {
		$this->_save($date);
		return $this;
	}
    public function _save($date=true,$forceId=false,$reload=true) {
        $db = DHFW::db();
        $data = $this->data->get();
        foreach($data as $k=>$v) {
            if($v === null) {
                unset($data[$k]);
            }
            if(is_array($v) || is_object($v)) {
                if(in_array($k,static::$jsonFields)) {
                    $data[$k] = json_encode($v,JSON_UNESCAPED_SLASHES);
                } elseif(in_array($k,static::$serializedFields)) {
                    $data[$k] = serialize($v);
                }
            }
        }

        if($this->isNew() && !$forceId) {
            $sqlInfo = static::SQLInsertQuery($data,$date);
            if($this->debugSql) {
                dhGlobal::outLine("SQLDebug(record)",$sqlInfo["sql"],"::::",$sqlInfo["values"]);
            }
            $db->run($sqlInfo["sql"],$sqlInfo["values"]);
            $this->setId($db->lastInsertId());
            $this->setNew(false);
        } else {
            $whereConditions = [static::$idField=>$this->id()];
            $sqlInfo = static::SQLUpdateQuery($data,$whereConditions,$date);
            if($this->debugSql) {
                dhGlobal::outLine("SQLDebug(record)",$sqlInfo["sql"],"::::",$sqlInfo["values"]);
            }
            $db->run($sqlInfo["sql"],$sqlInfo["values"]);
        }
        if($this->id()) {
            if($reload) {
                $this->load($this->id(),true);
            }
            return true;
        }
        return false;
    }

    public function populate($data) {
        $idField = static::$idField;
        if(isset($data[$idField])) {
            $this->setId($data[$idField]);
        }
        foreach($data as $k=>$v) {
            if(in_array($k,static::$jsonFields) && !is_array($v) && !is_object($v)) {
                $v = json_decode($v,true);
            } elseif(in_array($k,static::$serializedFields) && !is_array($v) && !is_object($v)) {
                $v = unserialize($v);
            }
            $this->set($k,$v);
        }
        $this->initFields();
    }

    public function load($id,$isReload=false) {
        $data = static::getById($id);
        if(!$data) return false;
        $this->populate($data->get());
        $this->setId($id);
        if(!$isReload) {
            $this->setNew(false);
        }
        return true;
    }

    private function initFields() {
        $this->fields = static::moduleFields();
        if(!is_array($this->fields)) {
            echo static::$moduleName."::moduleFields() did not return an array";
        }
        foreach($this->fields as $fieldName=>$field) {
            $this->captureFieldValue($field);
            $this->fields[$field->name()] = $field;
        }
    }
    private function captureFieldValue($field) {
        if($field->derivedCallable() === false) {
            if(($value = $this->get($field->name(),"ZfwZ-NO-VALUE-SET-YfwY")) !== "ZfwZ-NO-VALUE-SET-YfwY") {
                $field->value($value);
            }
            return $field->value();
        }
        $func = $field->derivedCallable();
        if(is_callable($func)) {
            $field->value($func($this));
        } elseif(method_exists($this,$func)) {
            $field->value($this->$func());
        } else {
            echo static::$moduleName."::".$field->name()." derivedCallable is not callable";
        }
        return $field->value();
    }
    private static $moduleFieldsCache = [];
    public static function moduleFields() {
        if(isset(self::$moduleFieldsCache[static::$moduleName])) {
            return self::$moduleFieldsCache[static::$moduleName];
        }
        $appRouter = AppRouter::getInstance();
        $fields = [];
        if(static::$useFieldsTable) {
            $db = DHFW::db();
            $sql = "select * from dhfw_fields where `module`=? order by `detailorder` ASC";
            $sth = $db->run($sql,[static::$moduleName]);
            while($row = $db->next($sth)) {
                if(($field = AppRouter::fieldFromDef(static::$moduleName,$row->toArray())) !== false)  {
                    $field->moduleName(static::$moduleName);
                    $fields[$field->name()] = $field;
                }
            }
        } else {
            foreach(static::$fieldDefs as $fieldName=>$fieldDef) {
                if(!isset($fieldDef["name"])) {
                    $fieldDef["name"] = $fieldName;
                }
                if(($field = AppRouter::fieldFromDef(static::$moduleName,$fieldDef)) !== false)  {
                    $field->moduleName(static::$moduleName);
                    $fields[$field->name()] = $field;
                }
            }
        }
        self::$moduleFieldsCache[self::$moduleName] = $fields;
        return $fields;
    }
    private static $moduleListFieldsCache = [];
    public static function moduleListFields($view=null) {
        if(is_null($view)) {
            $view = "default";
        }
        if(!isset(static::$moduleListFieldsCache[static::$moduleName][$view])) {
            static::$moduleListFieldsCache[static::$moduleName][$view] = [];
            $moduleFields = static::moduleFields();
            foreach($moduleFields as $fieldName=>$field) {
                if($field->listable()) {
                    static::$moduleListFieldsCache[static::$moduleName][$view][$fieldName] = $field;
                }
            }
        }
        return static::$moduleListFieldsCache[static::$moduleName][$view];
    }
    
}