<?php
namespace boru\dhfw\base;

use boru\dhdb\core\Error;
use boru\dhfw\DHFW;
use boru\dhfw\util\Debug;
use boru\dhfw\util\QueryConditions;
use boru\dhutils\dhGlobal;
use boru\dhutils\tools\DebugTrace;

abstract class BaseModuleRecord {
    /**
     * The name of the module
     * @var string
     */
    protected static $moduleName = "Module";
    /**
     * The name of the primary table in the database
     */
    protected static $sqlTable = "";

    /**
     * The name of the tables in the database
     * @var array<string>
     */
    protected static $sqlExtraTables = [];

    /**
     * The join clauses for the $sqlExtraTables
     * * Syntax: "tablename"=>"tablename.id = othertablename.id"
     * @var array
     */
    protected static $sqlTableJoins = [];

    /**
     * The name of the primary key field in the database
     * @var string
     */
    protected static $idField = "";

    /**
     * The name of the createdTime field in the database, blank to disable
     */
    protected static $createdTimeField = "";

    /**
     * The name of the modifiedTime field in the database, blank to disable
     */
    protected static $modifiedTimeField = "";

    /**
     * The name of the label field, or the sql to generate it
     */
    protected static $labelField = "";

    protected static $jsonFields = [];
    protected static $serializedFields = [];
    protected static $fieldDefs = [];
    protected static $useFieldsTable = false;
    protected static $isRecordModule = true;
    protected static $defaultView = "Home";


    protected static $columnDefinitions = [];
    protected static $columns = [];
    protected static $fieldTableColumnMap = [];
    

    
    private static $hasBeenInitialized = [];

    public static function getModuleName() {
        
        static::initDefaults();
        return static::$moduleName;
    }

    public static function initDefaults($moduleName=null,$obj=null) {
        if(is_null($moduleName) || $moduleName == "Core") {
            if(is_null($obj)) {
                $obj = new static();
            }
            $moduleName = $obj->getModuleDefinition("moduleName", $obj::$moduleName);
        }
        if(isset(self::$hasBeenInitialized[$moduleName])) {
            return;
        }
        if(is_null($obj)) {
            $obj = new static();
        }
        
        $moduleDefinition = $obj->getModuleDefinition();
        if(is_array($moduleDefinition) && !empty($moduleDefinition)) {
            static::$isRecordModule = $obj->getModuleDefinition("isRecordModule",static::$isRecordModule);
            static::$moduleName = $obj->getModuleDefinition("moduleName",$moduleName);
            static::$sqlTable = $obj->getModuleDefinition("sqlTable",static::$sqlTable);
            static::$sqlExtraTables = $obj->getModuleDefinition("sqlExtraTables",static::$sqlExtraTables);
            static::$sqlTableJoins = $obj->getModuleDefinition("sqlTableJoins",static::$sqlTableJoins);
            static::$idField = $obj->getModuleDefinition("idField",static::$idField);
            static::$createdTimeField = $obj->getModuleDefinition("createdTimeField",static::$createdTimeField);
            static::$modifiedTimeField = $obj->getModuleDefinition("modifiedTimeField",static::$modifiedTimeField);
            static::$labelField = $obj->getModuleDefinition("labelField",static::$labelField);
            static::$jsonFields = $obj->getModuleDefinition("jsonFields",static::$jsonFields);
            static::$serializedFields = $obj->getModuleDefinition("serializedFields",static::$serializedFields);
            static::$fieldDefs = $obj->getModuleDefinition("fieldDefs",static::$fieldDefs);
            static::$useFieldsTable = $obj->getModuleDefinition("useFieldsTable",static::$useFieldsTable);
            static::$defaultView = $obj->getModuleDefinition("defaultView",static::$defaultView);
        }
        self::$hasBeenInitialized[static::$moduleName] = true;
        if(static::$isRecordModule) {
            static::initSqlDefaults();
        }
    }

    public static function initSqlDefaults() {
        static::initDefaults();
        if(static::$sqlTable == "") {
            static::$sqlTable = strtolower(static::$moduleName);
        }
        if(static::$idField == "") {
            static::$idField = "id";
        }
        if(static::$createdTimeField == "") {
            static::$createdTimeField = "createdTime";
        }
        if(static::$modifiedTimeField == "") {
            static::$modifiedTimeField = "modifiedTime";
        }
        if(static::$labelField == "") {
            static::$labelField = "label";
        }
        static::_sqlinfo_ColumnDefinitions();
    }

    public static function getColumns() {
        static::initDefaults();
        return static::_sqlinfo_FieldTableColumnMap(true);
    }
    public static function getFieldMap() {
        static::initDefaults();
        return static::_sqlinfo_FieldTableColumnMap();
    }

    public static function labelField() {
        static::initDefaults();
        return static::$labelField;
    }
    public static function idField() {
        static::initDefaults();
        return static::$idField;
    }
    public static function createdTimeField() {
        static::initDefaults();
        return static::$createdTimeField;
    }
    public static function modifiedTimeField() {
        static::initDefaults();
        return static::$modifiedTimeField;
    }

    private static $missingTables = [];
    private static $skipColClass = [];
        
    public static function SQLTableWithJoins() {
        static::initDefaults();
        $joins = [];
        $joins[] = "`".static::$sqlTable."`";
        
        foreach(static::_sql_tableJoins() as $table=>$join) {
            $joins[] = "INNER JOIN `".$table."` ON ".$join;
        }
        
        return implode(" ", $joins);
    }

    public static function SQLBuildSetClause($columns=[]) {
        static::initDefaults();
        $sql = "";
        $values = [];
        $fieldTableColumnMap = static::_sqlinfo_FieldTableColumnMap();
        foreach($columns as $col=>$val) {
            if(isset($fieldTableColumnMap[$col])) {
                $tables = $fieldTableColumnMap[$col];
                foreach($tables as $table) {
                    $sql .= "`".$table."`.`".$col."` = ?, ";
                    $values[] = $val;
                }
            }
        }
        $sql = rtrim($sql, ", ");
        return ["sql"=>$sql,"values"=>$values];
    }
    public static function SQLBuildConditionClause($conditions=[]) {
        static::initDefaults();
        $sql = "";
        $values = [];
        $fieldTableColumnMap = static::_sqlinfo_FieldTableColumnMap();
        foreach($conditions as $col=>$val) {
            if(isset($fieldTableColumnMap[$col])) {
                $tables = $fieldTableColumnMap[$col];
                foreach($tables as $table) {
                    $sql .= "`".$table."`.`".$col."` = ? AND ";
                    $values[] = $val;
                }
            }
        }
        $sql = rtrim($sql, " AND ");
        return ["sql"=>$sql,"values"=>$values];
    }
    public static function SQLInsertQuery($columns=[],$dateFields=false) {
        static::initDefaults();
        $sql = "INSERT INTO ".static::SQLTableWithJoins()." SET ";
        $setClause = static::SQLBuildSetClause($columns);
        $sql .= $setClause["sql"];
        $values = $setClause["values"];
        if(static::shouldCreatedTime($dateFields) && !isset($columns[static::createdTimeField()])) {
            $sql .= ", `".static::createdTimeField()."` = NOW()";
        }
        if(static::shouldModifiedTime($dateFields) && !isset($columns[static::modifiedTimeField()])) {
            $sql .= ", `".static::modifiedTimeField()."` = NOW()";
        }
        return ["sql"=>$sql,"values"=>$values];

    }
    public static function SQLUpdateQuery($columns=[], $conditions=[], $dateFields=false) {
        static::initDefaults();
        $sql = "UPDATE ".static::SQLTableWithJoins()." SET ";
        $setClause = static::SQLBuildSetClause($columns);
        $sql .= $setClause["sql"];
        $values = $setClause["values"];
        if(static::shouldModifiedTime($dateFields) && !isset($columns[static::modifiedTimeField()])) {
            $sql .= ", `".static::modifiedTimeField()."` = NOW()";
        }
        $whereConditions = new QueryConditions();
		$whereConditions->addArray($conditions);
		
        $sql.=" ".$whereConditions->getSQL("WHERE");
		$values = array_merge($values,$whereConditions->getValues());
        $whereClause = static::SQLBuildConditionClause($conditions);
        return ["sql"=>$sql,"values"=>$values];

    }
    public static function SQLDeleteQuery($conditions=[]) {
        static::initDefaults();
        $sql = "DELETE FROM ".static::SQLTableWithJoins();
        $whereClause = static::SQLBuildConditionClause($conditions);
        $sql .= " WHERE ".$whereClause["sql"];
        $values = $whereClause["values"];
        return ["sql"=>$sql,"values"=>$values];
    }
    private static function shouldCreatedTime($dateFields=true) {
        return $dateFields && !is_null(static::createdTimeField()) && static::createdTimeField() !== false;
    }
    private static function shouldModifiedTime($dateFields=true) {
        return $dateFields && !is_null(static::modifiedTimeField()) && static::modifiedTimeField() !== false;
    }

    public static function _sql_table() {
        static::initDefaults();
        if(empty(static::$sqlTable)) {
            static::$sqlTable = "";
        }
        return static::$sqlTable;
    }
    public static function _sql_tableJoins() {
        static::initDefaults();
        if(!is_array(static::$sqlTableJoins)) {
            static::$sqlTableJoins = [];
        }
        return static::$sqlTableJoins;
    }

    public static function _sqlinfo_FieldTableColumnMap($colArray=false) {
        static::initDefaults();
        $modKey = static::getModuleName();
        if(!isset(static::$fieldTableColumnMap[$modKey])) {
            static::$fieldTableColumnMap[$modKey] = [];
        }
        if(!empty(static::$fieldTableColumnMap[$modKey])) {
            return static::$fieldTableColumnMap[$modKey];
        }
        $columnDefs = static::_sqlinfo_ColumnDefinitions();
        
        foreach($columnDefs as $table=>$cols) {
            foreach($cols as $col=>$def) {
                static::$columns[$modKey][$table][] = $col;
                static::$fieldTableColumnMap[$modKey][$col][] = $table;
            }
        }
        if($colArray) {
            return static::$columns[$modKey];
        }
        return static::$fieldTableColumnMap[$modKey];
    }
    public static function _sqlinfo_ColumnDefinitions($force=false) {
        static::initDefaults();
        $modKey = static::getModuleName();
        if(!isset(static::$columnDefinitions[$modKey])) {
            static::$columnDefinitions[$modKey] = [];
        }
        if(!empty(static::$columnDefinitions[$modKey]) && !$force) {
            return static::$columnDefinitions[$modKey];
        }
        if(empty(static::$sqlTable) || isset(self::$skipColClass[static::$moduleName])) {
            return;
        }
        $tables = [static::$sqlTable];
        if(!empty(static::$sqlExtraTables)) {
            $tables = array_merge($tables, static::$sqlExtraTables);
        }
        foreach($tables as $table) {
            $db = DHFW::db();
            $sql = "SHOW columns from `".$table."`";
            $sth = $db->run($sql);
            if($sth->isError()) {
                $error = $sth->getError();
                if($error instanceof Error) {
                    throw new \Exception("Error getting column definitions for table ".$table.": ".$error->getMessage());
                } else {
                    throw new \Exception("Error getting column definitions for table ".$table.": ".$error->getMessage());
                }
                self::$missingTables[] = $table;
                continue;
            }
            while($row=$db->next($sth)) {
                Debug::addNote(1,"Column: ".$row["Field"]);
                $rowArr = $row->toArray();
                static::$columnDefinitions[$modKey][$table][$rowArr["Field"]] = $rowArr;
            }
        }
        if(empty(static::$columnDefinitions[$modKey])) {
            self::$skipColClass[static::$moduleName] = static::$moduleName;
        }
        static::_sqlinfo_FieldTableColumnMap();
        return static::$columnDefinitions[$modKey];
    }


}