<?php
namespace boru\dhfw\util;

use boru\dhfw\DHFW;
use boru\dhfw\base\BaseModule;
use boru\dhfw\util\query\ConditionGroup;

class QueryBuilder extends ConditionGroup {
    /**
     * The module class, for static references to the module
     * @var \boru\dhfw\base\BaseModule
     */
    private $module;

    private $select = [];
    private $having;
    private $groupBy;
    private $orderBy;
    private $limit;
    private $offset;

    private $joins = "";

    public function __construct($moduleName=null) {
        if(!empty($moduleName)) {
            $this->forModule($moduleName);
        }
        $this->having = new ConditionGroup();
    }
    public function forModule($moduleName) {
        if($moduleName instanceof BaseModule) {
            $this->module = $moduleName;
        } else {
            $this->module = DHFW::getModuleClass($moduleName);
        }
        if(!$this->module instanceof BaseModule || $this->module->getModuleName() == "Core") {
            throw new \Exception("Invalid module/module name");
        }
        $this->joins = $this->module::SQLTableWithJoins();
        return $this;
    }
    public function select($selectInput,$quote=true) {
        return $this->addSelect($selectInput,$quote);
    }
    public function addSelect($selectString,$quote=true) {
        if(is_array($selectString)) {
            foreach($selectString as $select) {
                $this->addSelect($select,$quote);
            }
            return $this;
        }
        if($quote) {
            $this->select[] = "`".$selectString."`";
        } else {
            $this->select[] = $selectString;
        }
        return $this;
    }
    public function limit($limit,$offset=null) {
        $this->limit = $limit;
        if(!empty($offset)) {
            $this->offset($offset);
        }
        return $this;
    }
    public function offset($offset) {
        $this->offset = $offset;
        return $this;
    }
    public function orderBy($orderBy) {
        $this->orderBy = $orderBy;
        return $this;
    }
    public function having() {
        return $this->having;
    }

    public function query($count=false,$rowCallback=null) {
        $sql = $this->getQuery($count);
        $values = $this->getValues();
        $db = DHFW::db();
        $sth = $db->run($sql,$values);
        if($sth->isError()) {
            throw new \Exception("Error running query: ".$sth->getError()->getMessage());
        }
        if($count) {
            $row = $db->next($sth);
            if(!empty($rowCallback)) {
                $row = $rowCallback($row);
            }
            return $row["count"];
        }
        $rows = [];
        while($row = $db->next($sth)) {
            if(!empty($rowCallback)) {
                $row = $rowCallback($row);
            }
            $rows[] = $row;
        }
        return $rows;
    }

    public function getQuery($count=false) {
        $sql = "SELECT ";
        if($count) {
            $sql.="COUNT(*) as `count`";
        } else {   
            if(empty($this->select)) {
                $sql.="*";
            } else {
                $sql.=implode(",",$this->select);
            }
        }
        $sql.=" FROM ".$this->joins;
        $whereSql = $this->toSql();
        if(!empty($whereSql)) {
            $sql.=" WHERE ".$whereSql;
        }
        if(!empty($this->groupBy)) {
            $sql.=" GROUP BY ".$this->groupBy;
        }
        if(!empty($this->having->toSql())) {
            $sql.=" HAVING ".$this->having->toSql();
        }
        if(!empty($this->orderBy)) {
            $sql.=" ORDER BY ".$this->orderBy;
        }
        if(!empty($this->limit)) {
            if(!empty($this->offset)) {
                $sql.=" LIMIT ?,?";
            } else {
                $sql.=" LIMIT ?";
            }
        }
        return $sql;
    }

    public function getValues() {
        $values = parent::getValues();
        $havingValues = $this->having->getValues();
        if(!empty($havingValues)) {
            $values = array_merge($values,$havingValues);
        }
        if(!empty($this->limit)) {
            if(!empty($this->offset)) {
                $values[] = $this->offset;
            }
            $values[] = $this->limit;
        }
        return $values;
    }
    

    public static function sqlComparatorToComparator($sqlComparator) {
        switch(strtolower($sqlComparator)) {
            case "eq":
            case "=":
                return "eq";
            case "ne":
            case "!=":
            case "<>":
                return "ne";
            case "lt":
            case "<":
                return "lt";
            case "gt":
            case ">":
                return "gt";
            case "lte":
            case "<=":
                return "lte";
            case "gte":
            case ">=":
                return "gte";
            case "like":
                return "like";
            case "nlike":
            case "not like":
                return "nlike";
            case "bw":
            case "sw":
            case "begins with":
                return "bw";
            case "bn":
            case "sn":
            case "not begins with":
                return "bn";
            case "ew":
            case "ends with":
                return "ew";
            case "en":
            case "not ends with":
                return "en";
            case "in":
                return "in";
            case "nin":
            case "not in":
                return "nin";
        }
    }
    public static function comparatorToSqlComparator($comparator) {
        switch(strtolower($comparator)) {
            case "eq":
            case "=":
                return "=";
            case "ne":
            case "!=":
            case "<>":
                return "<>";
            case "lt":
            case "<":
                return "<";
            case "gt":
            case ">":
                return ">";
            case "lte":
            case "<=":
                return "<=";
            case "gte":
            case ">=":
                return ">=";
            case "like":
                return "LIKE";
            case "nlike":
            case "not like":
                return "NOT LIKE";
            case "bw":
            case "sw":
            case "begins with":
                return "LIKE";
            case "bn":
            case "sn":
            case "not begins with":
                return "NOT LIKE";
            case "ew":
            case "ends with":
                return "LIKE";
            case "en":
            case "not ends with":
                return "NOT LIKE";
            case "in":
                return "IN";
            case "nin":
            case "not in":
                return "NOT IN";
        }
        return false;
    }

    public static function makeForSql($fieldType,$columnName,$comparator,$value) {
        $sql = "";
        $sqlComparator = static::comparatorToSqlComparator($comparator);
        $comparator = static::sqlComparatorToComparator($comparator);
        switch(strtolower($fieldType)) {
            case "bool":
            case "boolean":
                switch($comparator) {
                    case "eq":
                        $sql = "`".$columnName."` $sqlComparator ?";
                        $value = $value ? "1" : "0";
                        break;
                    case "ne":
                        $sql = "`".$columnName."` $sqlComparator ?";
                        $value = $value ? "1" : "0";
                        break;
                }
                break;
            case "date":
            case "datetime":
                switch($comparator) {
                    case "eq":
                    case "ne":
                    case "lt":
                    case "gt":
                    case "lte":
                    case "gte":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        $value = date("Y-m-d H:i:s",strtotime($value));
                        break;
                }
                break;
            case "select":
            case "owner":
            case "selectuser":
                switch($comparator) {
                    case "eq":
                    case "ne":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        break;
                    case "in":
                    case "nin":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        break;
                }
                break;
            case "text":
            case "textarea":
            case "int":
            case "double":
            case "number":
            case "currency":
                switch($comparator) {
                    case "eq":
                    case "ne":
                    case "lt":
                    case "gt":
                    case "lte":
                    case "gte":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        break;
                    case "like":
                    case "nlike":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        $value = "%".$value."%";
                        break;
                    case "bw":
                    case "sw":
                    case "sn":
                    case "bn":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        $value = $value."%";
                        break;
                    case "ew":
                    case "en":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        $value = "%".$value;
                        break;
                    case "in":
                    case "nin":
                        $sql = "`".$columnName."` ".$sqlComparator." ?";
                        break;
                }
                break;
        }
        return ["sql"=>$sql,"value"=>$value];
    }
}