<?php
namespace boru\dhfw\util\query;

class ConditionGroup implements ConditionInterface {
    /** @var ConditionInterface[] */
    private $conditions = [];
    private $glue = "AND";// "AND" or "OR"

    public function __construct($glue="AND") {
        $this->glue = $glue;
    }

    public function glue($glue) {
        $this->glue = $glue;
        return $this;
    }

    /**
     * Add a new ConditionGroup
     * @param string $glue 
     * @return ConditionGroup 
     */
    public function group($glue="AND") {
        $group = new ConditionGroup($glue);
        $this->conditions[] = $group;
        return $group;
    }

    /**
     * Query on a field, optional field type
     * @param string $field
     * @param string $value
     * @param string $comparator
     * @param string $fieldType
     * @return $this
     */
    public function where($fieldName,$comparator,$value,$fieldType="text") {
        $this->conditions[] = new Condition($fieldType,$fieldName,$comparator,$value);
        return $this;
    }

    /**
     * Query on a text field
     * @param string $field
     * @param string $value
     * @param string $comparator
     * @return $this
     */
    public function whereText($fieldName,$comparator,$value) {
        $this->conditions[] = new Condition("text",$fieldName,$comparator,$value);
        return $this;
    }

    /**
     * Query on a number field
     * @param string $field
     * @param int|float $value
     * @param string $comparator
     * @return $this
     */
    public function whereNumber($fieldName,$comparator,$value) {
        $this->conditions[] = new Condition("number",$fieldName,$comparator,$value);
        return $this;
    }

    /**
     * Query on a date field
     * @param string $field
     * @param string $value
     * @param string $comparator
     * @return $this
     */
    public function whereDate($fieldName,$comparator,$value) {
        $this->conditions[] = new Condition("date",$fieldName,$comparator,$value);
        return $this;
    }

    /**
     * Query on a datetime field
     * @param string $field
     * @param string $value
     * @param string $comparator
     * @return $this
     */
    public function whereDateTime($fieldName,$comparator,$value) {
        $this->conditions[] = new Condition("datetime",$fieldName,$comparator,$value);
        return $this;
    }

    /**
     * Query on a boolean field
     * @param string $field
     * @param bool $value
     * @param string $comparator
     * @return $this
     */
    public function whereBool($fieldName,$comparator,$value) {
        $this->conditions[] = new Condition("bool",$fieldName,$comparator,$value);
        return $this;
    }
    /**
     * 
     * @param ConditionInterface $condition A Condition or ConditionGroup object
     * @return $this 
     */
    public function whereCondition($condition) {
        $this->conditions[] = $condition;
        return $this;
    }

    public function toSql() {
        if(empty($this->conditions)) {
            return "";
        }
        $parts = [];
        foreach($this->conditions as $condition) {
            $parts[] = $condition->toSql();
        }

        $sql = "";
        $sql.="( ";
        $sql.=implode(" ".$this->glue." ",$parts);
        $sql.=" )";
        return $sql;
    }
    public function getValues() {
        $values = [];
        foreach($this->conditions as $condition) {
            $conditionValue = $condition->getValues();
            if(!is_array($conditionValue))
                $conditionValue = [$conditionValue];
            $values = array_merge($values,$conditionValue);
        }
        return $values;
    }
}