<?php
namespace boru\dhprocess;

use boru\dhprocess\config\Config;
use boru\dhprocess\config\WorkConfig;
use boru\dhprocess\config\WorkerConfig;
use boru\dhprocess\config\QueueConfig;
use boru\dhprocess\queue\Queue;
use boru\dhutils\dhGlobal;

class TaskQueue {
    private static $verbosity = [
        "start"=>false,
        "queue"=>false,
        "done"=>false,
        "detail"=>false,
    ];
    /** @var Config */
    private static $config;

    private static $expected;
    
    /** @var Queue */
    private static $queue;

    private static $collectionMap = [];

    /**
     * @param int|null $expected
     * @param Queue $queue (null)
     */
    public static function expected($expected=null) {
        if(is_null($expected)) return static::$expected;
        static::$expected = $expected;
        if(!is_null(static::$queue)) {
            static::$queue->setExpected($expected);
        }
    }

    /**
     * 
     * @param Task $task 
     * @param mixed $taskGroup 
     * @param mixed $queue 
     * @return Task 
     */
    public static function queue(Task $task,$skipCollect=false,$isGenerator=false) {
        if(is_null(static::$queue)) {
            static::$queue = static::init();
        }
        if(!$skipCollect) {
            $task->lastly(function($result) use ($task) {
                static::$collectionMap[$task->id()] = $result;
                $task->Work()->destroy();
            });
        }
        static::$queue->queue($task->Work(),$isGenerator);
        return $task;
    }
    /**
     * 
     * @param Task $task 
     * @param mixed $taskGroup 
     * @param mixed $queue 
     * @return Task 
     */
    public static function add(Task $task) {
        return static::queue($task);
    }

    /**
     * 
     * @param null|Queue $queue 
     * @return true|null 
     */
    public static function wait() {
        if(!is_null(static::$queue)) {
            static::$queue->wait();
            return true;
        }
        return null;
    }
    /**
     * Collects data from the Queue. Data is only collectable if TaskQueue::discardWork is set intentionally to false, work results are otherwise discarded.
     * 
     * You can collect results either by using an onDone handler and collecting it that way, or using the &$collectableArray applied to ::init() and/or ::create()
     * @param null|Queue $queue 
     * @return null|mixed
     */
    public static function collect() {
        if(!is_null(static::$queue)) {
            return static::$queue->collect();
        }
        return null;
    }

    /**
     * Init the task queue.
     * @param array $configOptions See \boru\dhprocess\config\Config::$data for available options
     * @param mixed $collectionArray If you want to collect results from the queue, you can pass an array by reference here. The array will be populated with the results of the work.
     * @return Queue 
     */
    public static function init($configOptions=[],&$collectionArray=null) {
        static::initConfig($configOptions,false);
        static::$config->applyOptions($configOptions);
        static::$queue = static::create(static::$config,$collectionArray);
        if(!empty(static::$rateLimit)) {
            foreach(static::$rateLimit as $key=>$rl) {
                static::$queue->setRateLimit($key,$rl["limit"],$rl["time"]);
            }
        }
        return static::$queue;
    }

    public static function create($config=null,&$collectionArray=null) {
        if(is_array($config)) {
            $config = new Config($config);
        } if($config instanceof Config) {

        } else {
            $config = new Config();
        }
        $queue = new Queue($config);


        if(!is_null(static::$expected)) {
            static::expected(static::$expected,$queue);
        }

        if(!is_null($collectionArray)) {
            static::$collectionMap = &$collectionArray;

            $queue->set("discardOnDone",true);
        }

        if(!empty(static::$rateLimit)) {
            foreach(static::$rateLimit as $key=>$rl) {
                $queue->setRateLimit($key,$rl["limit"],$rl["time"]);
            }
        }
        
        
        //$queue->setWorkerLogDir("_logs/");
        //$queue->workerManager()->setLog("test.log");
        return $queue;
    }

    public static function getVerbosity() {
        return static::config()->verbosity();
    }
    public static function setVerbosity($start=false,$queue=false,$done=false,$detail=false) {
        static::config()->set("start",$start);
        static::config()->set("queue",$queue);
        static::config()->set("done",$done);
        static::config()->set("detail",$detail);
    }

    /**
     * 
     * @param mixed $name 
     * @param array $args 
     * @param mixed $generator
     * @return Task 
     */
    public static function Task($name,$args=[],$generator=false) {
        $task = new Task($name,$args);
        $verbosity = static::getVerbosity();
        $task->setVerbosity($verbosity["start"],$verbosity["queue"],$verbosity["done"],$verbosity["detail"]);
        return static::queue($task,true,$generator);
    }

    public static function Group($groupName,$collectResults=false) {
        return new TaskGroup($groupName,$collectResults);
    }



    //rate limiting implementation

    private static $rateLimit = [];

    /**
     * Sets a rate limit for a key. If the key is called more than the limit within the time period, rateLimit will return false.
     * @param string $key The key/name to rate limit
     * @param int $limit The number of times the key can be called within the time period
     * @param int $timeInSeconds The time period in seconds
     * @return void 
     */
    public static function setRateLimit($key,$limit,$timeInSeconds) {
        static::$rateLimit[$key] = [
            "limit"=>$limit,
            "time"=>$timeInSeconds,
        ];
        if(!is_null(static::$queue)) {
            static::$queue->setRateLimit($key,$limit,$timeInSeconds);
        }
    }

    //Config stuff, including backward compatibility



    public static function config($options=null) {
        static::initConfig([],false);
        if(is_null($options)) return static::$config;
        static::$config->applyOptions($options);
    }

    public static function get($key=null,$value=null) {
        static::initConfig([],false);
        return static::$config->get($key,$value);
    }
    public static function set($key,$value=null) {
        static::initConfig([],false);
        return static::$config->set($key,$value);
    }

    public static function initConfig($configOptions=null,$force=true) {
        if(is_null(static::$config) || $force) {
            static::$config = new Config();
        }
        if(is_array($configOptions)) {
            static::$config->applyOptions($configOptions);
        }
    }
    /**
     * @param null|array $queueConfig
     * @return QueueConfig
     */
    public static function queueConfig($queueConfig=null) {
        if(is_null($queueConfig)) {
            if(!is_null(static::$queue)) {
                return static::$queue->config();
            } else {
                return static::config();
            }
        } else {
            static::config()->applyOptions($queueConfig);
            if(!is_null(static::$queue)) {
                static::$queue->config()->applyOptions($queueConfig);
            }
            return static::config();
        }
    }

    /**
     * Quick configuration that enables visualization
     * @param mixed $numWorkers 
     * @param mixed $maxQueued 
     * @return Config 
     */
    public static function queueQuickConfigVisual($numWorkers=null,$maxQueued=null) {
        static::config();
        if(is_null($numWorkers)) {
            $numWorkers = intval(exec("nproc"));
        }
        if($numWorkers<=0) {
            $numWorkers=3;
        }
        static::config()->set("numWorkers",$numWorkers);
        static::config()->set("visualize",true);
        static::config()->set("extendedBar",true);
        static::config()->set("maxQueued",is_null($maxQueued) ? ($numWorkers+5) : $maxQueued);
        return static::config();
    }
    /**
     * Quick configuration that disables visualization
     * @param mixed $numWorkers 
     * @param mixed $maxQueued 
     * @return QueueConfig 
     */
    public static function queueQuickConfigNoVisual($numWorkers=null,$maxQueued=null) {
        if(is_null($numWorkers)) {
            $numWorkers = intval(exec("nproc"));
        } if($numWorkers<=0) {
            $numWorkers=3;
        }
        static::config();
        static::config()->set("numWorkers",$numWorkers);
        static::config()->set("visualize",false);
        static::config()->set("extendedBar",false);
        static::config()->set("maxQueued",is_null($maxQueued) ? ($numWorkers+5) : $maxQueued);
        return static::config();
    }
    
    /**
     * @param null|array $workerConfig
     * @return WorkerConfig
     */
    public static function workerConfig($workerConfig=null) {
        if(is_null($workerConfig)) {
            if(!is_null(static::$queue)) {
                return static::$queue->config();
            } else {
                return static::$config;
            }
        } else {
            static::$config->applyOptions($workerConfig);
            if(!is_null(static::$queue)) {
                static::$queue->config()->applyOptions($workerConfig);
            }
            return static::$config;
        }
    }
    /**
     * Quick Worker config
     * @param mixed $bootstrapFile 
     * @param mixed $logDir 
     * @param mixed $logPerWorker 
     * @return WorkerConfig 
     */
    public static function workerQuickConfig($bootstrapFile=null,$logDir="_logs",$logPerWorker=null) {
        static::config();
        if(!is_null($bootstrapFile)) {
            static::config()->set("bootstrapFile",$bootstrapFile);
        }
        if(!is_null($logDir)) {
            static::config()->set("logDir",$logDir);
        }
        if(!is_null($logPerWorker)) {
            static::config()->set("logPerWorker",$logPerWorker);
        }
        return static::config();
    }

    /**
     * @param null|array $workerConfig
     * @return WorkConfig
     */
    public static function workConfig($workConfig=null) {
        if(is_null($workConfig)) {
            if(!is_null(static::$queue)) {
                return static::$queue->config();
            } else {
                return static::$config;
            }
        } else {
            static::$config->applyOptions($workConfig);
            if(!is_null(static::$queue)) {
                static::$queue->config()->applyOptions($workConfig);
            }
            return static::$config;
        }
    }
}