<?php
namespace boru\dhprocess\process;

use boru\dhprocess\process\traits\GettersSetters;
use boru\dhprocess\message\Message;
use boru\dhprocess\queue\Queue;
use boru\dhprocess\queue\QueueLogger;
use boru\dhprocess\queue\QueueWorker;
use boru\dhutils\dhGlobal;
use boru\dhutils\dhOut;
use boru\dhutils\tools\ProcessQueue;
use Exception;
use React\Promise\Deferred;
use React\Promise\Promise;
use UnexpectedValueException;
use RuntimeException;

class WorkerProcess {

    use GettersSetters;

    private $command,$args,$meta;
    private $metaData;

    private $data;

    private $logging = false;
    /** @var dhOut */
    private $logger;

    /** @var bool */
    private $running = false;
    /** @var bool */
    private $done = false;
    /** @var bool */
    private $ready = false;

    private $messageBuffer;

    private $pid,$exitCode,$termSignal;

    /** @var callable */
    private $onStart,$onData,$onError,$onDone,$onStreamClose,$onStreamError;

    /** @var \React\Promise\Deferred */
    private $deferred;

    /** @var \React\ChildProcess\Process */
    private $process;

    /** @var QueueWorker */
    private $queueWorker;

    /** @var Queue */
    private $queue;

    /**
     * 
     * @param array $meta 
     * @param int $workerTimeout 
     * @param mixed $bootstrap 
     * @param mixed $logDir 
     * @param QueueWorker $queueWorker 
     * @param Queue $queue
     * @return void 
     * @throws Exception 
     */
    public function __construct($meta=[],$queueWorker=null,$queue=null) {
        $this->setPid(uniqid());
        $this->queueWorker = $queueWorker;
        $this->queue = $queue;
        $workerOptions = [];
        //$queue->get("timeout"),$queue->get("bootstrapFile"),$queue->get("logDir")
        $workerOptions[] = "id:".$this->getPid();
        if(!is_null($queue)) {
            $workerOptions[] = "qid:".$queue->id();
        }
        if(!is_null($queueWorker)) {
            $workerOptions[] = "wid:".$queueWorker->getId();
        }
        if(($bootstrap = $queue->get("bootstrapFile",false)) !== false) {
            if(is_array($bootstrap)) {
                $bsString = implode(" ",$bootstrap);
            } else {
                $bsString = $bootstrap;
            }
            $workerOptions[]="inc:".$bsString;
        }
        if($queue->get("logDir")) {
            $workerOptions[] = "log:".QueueLogger::getLogFile("default");
        }
        $logSettingString = $queue->config()->logLevelsToTransport();
        if(!empty($logSettingString)) {
            $workerOptions[] = "ll:".$logSettingString;
        }
        $workerString = empty($workerOptions) ? "" : implode(" ",$workerOptions);
        if(($workerFile = dhGlobal::fileIfExists(__DIR__."/../worker/WorkerClient.php")) === false) {
            throw new \Exception("Cannot find worker script.");
        };
        $this->setCommand("php -f ".$workerFile->path()." ".$workerString);
        //echo $this->getCommand()."\n";
        //exit();
        $this->setArgs([]);
        $this->setMeta($meta); //sets all the handlers/etc
        $this->setMetaData("lastTime",microtime(true));
        $this->setMetaData("workerTimeout",$queue->get("timeout"));
    }

    /**
     * Get the queueId
     */
    public function getQueueId() {
        return $this->queue->id();
    }
    public function queueId() {
        return $this->getQueueId();
    }
    public function getQueueConfig() {
        return $this->queue->getQueueConfig();
    }

    /**
     * Get the queueWorkerId
     */
    public function getQueueWorkerId() {
        return $this->queueWorker->getId();
    }
    public function queueWorkerId() {
        return $this->getQueueWorkerId();
    }

    /**
     * Starts the Child Process, returns the Promise for easy chaining
     * @return Promise|false 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     */
    public function start() {
        $args = '';
        if(!empty($this->args)) {
            $args = ProcessQueue::pack($this->args);
        }
        $this->process = new \React\ChildProcess\Process('exec '.$this->command.' '.$args);
        $this->process->start();
        
        $this->setupProcessChannels();

        $this->onStart();
        return $this->Promise();
    }

    public function stop() {
        $this->running=false;
        $this->process->terminate();
        $this->process=null;
    }

    
    public function terminate() {
        $this->running=false;
        if(!is_null($this->process)) {
            $this->process->terminate();
        }
        if(!is_null($this->deferred)) {
            $this->deferred->reject(new \Exception("terminated"));
        }
        $this->process=null;
    }

    public function get() {
        return [
            "pid"=>$this->pid,
            "running"=>$this->running,
            "done"=>$this->done,
            "meta"=>$this->meta,
            "args"=>$this->args,
            "command"=>$this->command,
            "exitCode"=>$this->exitCode,
            "termSignal"=>$this->termSignal,
            "data"=>$this->data,
        ];
    }

    public function write($data) {
        if(!is_null($this->process) && !is_null($this->process->stdin)) {
            return $this->process->stdin->write($data);
        }
    }

    /**
     * Wrapper for \React\Promise\Promise::then()
     * @param callable|null $onFulfilled 
     * @param callable|null $onRejected 
     * @return PromiseInterface 
     */
    public function then(callable $onFulfilled=null,callable $onRejected=null) {
        return $this->Promise()->then($onFulfilled,$onRejected);
    }



    /** Handler Methods */
    public function onDone($exitCode=null,$termSignal=null) {
        $this->done = true;
        $this->running = false;
        $this->ready = false;
        $this->exitCode = $exitCode;
        $this->termSignal = $termSignal;
        $this->queueWorkerOnDone($exitCode,$termSignal);
        $this->deferred->resolve(true);
        if(!is_null($this->onDone)) {
            $handler = $this->onDone;
            $handler($this,$exitCode,$termSignal);
        }
    }
    public function onError($chunk=null) {
        $this->queueWorkerOnError($chunk);
        if(!is_null($this->onError)) {
            $handler = $this->onError;
            $handler($this,$chunk);
        }
    }
    public function onData($chunk=null) {

        $data = !empty($this->messageBuffer) ? $this->messageBuffer : "";
        $data.=$chunk;
        if(($message = Message::fromPacket($data)) !== false) {
            $chunk = $data;
            $this->messageBuffer = null;
        } else {
            $this->messageBuffer = $data;
            return;
        }
        $this->queueWorkerOnData($chunk);
        if(!is_null($this->onData)) {
            $handler = $this->onData;
            $handler($this,$chunk);
        }
    }
    public function onStart() {
        $this->running = true;
        $this->ready = false;
        $this->queueWorkerOnStart();
        if(!is_null($this->onStart)) {
            $handler = $this->onStart;
            $handler($this);
        }
    }
    public function onStreamClose($type) {
        $this->queueWorkerOnStreanClose($type);
        if(!is_null($this->onStreamClose)) {
            $handler = $this->onStreamClose;
            $handler($this,$type);
        }
    }
    public function onStreamError($type,$chunk=null) {
        $this->queueWorkerOnStreamError($type,$chunk);
        if(!is_null($this->onStreamError)) {
            $handler = $this->onStreamError;
            $handler($this,$type,$chunk);
        }
    }

    /**
     * Set the stdout/stderr channel listening
     * @return void 
     */
    private function setupProcessChannels() {
        $this->process->stdout->on('data', function($chunk) {
            $this->onData(trim($chunk));
        });
        $this->process->stdout->on('error', function($chunk) {
            $this->onStreamError("stdout",$chunk);
        });
        $this->process->stdout->on('close', function() {
            $this->onStreamClose("stdout");
        });
        $this->process->stderr->on('data', function($chunk) {
            $this->onError(trim($chunk));
        });
        $this->process->stderr->on('error', function($chunk) {
            $this->onStreamError("stderr",$chunk);
        });
        $this->process->stderr->on('close', function() {
            $this->onStreamClose("stderr");
        });
        $this->process->on('exit', function($exitCode,$termSignal) {
            $this->onDone();
        });
    }


    /**
     * Set the value of meta
     * @param   mixed  $meta  
     * @return  self
     */
    public function setMeta($meta) {
        $this->meta = $meta;
        $this->onData   = dhGlobal::getVal($meta, "onData",  null);
        $this->onError  = dhGlobal::getVal($meta, "onError", null);
        $this->onStart  = dhGlobal::getVal($meta, "onStart", null);
        $this->onDone   = dhGlobal::getVal($meta, "onDone",  null);
        $this->deferred = dhGlobal::getVal($meta, "deferred",null);
        $this->onStreamClose = dhGlobal::getVal($meta, "onStreamClose",null);
        $this->onStreamError = dhGlobal::getVal($meta, "onStreamError",null);
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
        }
        return $this;
    }

    public function jsonSerialize($array=null) {
        if(is_null($array)) {
            $array = $this->get();
        }
        return $array;
    }
    public function __toString() {
        return json_encode($this);
    }

    private function queueWorkerOnStart() {
        if(is_null($this->queueWorker)) return false;
        dhGlobal::trace("[worker]",$this->getId(),"worker process started");
        $this->queueWorker->onStart();
    }
    private function queueWorkerOnDone($exitCode=null,$termSignal=null) {
        if(is_null($this->queueWorker)) return false;
        $this->queueWorker->onDone($exitCode,$termSignal);

    }
    private function queueWorkerOnError($chunk=null) {
        if(is_null($this->queueWorker)) return false;
        $lines = explode("\n",$chunk);
        if(!empty($lines)) {
            foreach($lines as $line) {
                dhGlobal::trace("[worker-error]",$this->getId(),":error:",$line);
                $this->queueWorker->onError($line);
            }
        } else {
            dhGlobal::trace("[worker-error]",$this->getId(),":error:");
            $this->queueWorker->onError();
        }
    }
    private function queueWorkerOnData($chunk=null) {
        if(is_null($this->queueWorker)) return false;
        $lines = explode("\n",$chunk);
        if(!empty($lines)) {
            foreach($lines as $line) {
                dhGlobal::trace("[worker]",$this->getId(),":data:",$line);
                $this->queueWorker->onData($line);
            }
        }

    }
    private function queueWorkerOnStreanClose($type) {
        if(is_null($this->queueWorker)) return false;
        $this->queueWorker->disable();

    }
    private function queueWorkerOnStreamError($type,$chunk=null) {
        if(is_null($this->queueWorker)) return false;
        $this->queueWorker->disable();
    }

    /**
     * 
     * @param QueueWorker $queueWorker 
     * @return WorkerProcess 
     */
    public static function forQueue(QueueWorker $queueWorker,Queue $queue) {
        $instance = new self([],$queueWorker,$queue);
        return $instance;
    }
}