<?php
namespace boru\dhprocess\queue;

use boru\dhprocess\process\WorkerProcess;
use boru\dhprocess\config\WorkerConfig;
use boru\dhprocess\work\Work;
use boru\dhprocess\queue\Queue;
use boru\dhprocess\message\Message;
use boru\dhutils\dhGlobal;
use boru\dhutils\dhOut;
use React\EventLoop\Loop;
use React\EventLoop\TimerInterface;

class QueueWorker {
    private $id;
    /** @var Queue */
    private $queue;
    /** @var WorkerProcess */
    private $workerProcess;
    /** @var TimerInterface */
    private $timer;
    /** @var Work */
    private $work;
    private $workId;

     /** @var Work */
    private $stoppedWork;

    private $lastPing = 0;

    /** @var WorkerConfig */
    private $workerConfig;

    private $logging = false;
    private $logger;

    private $data = [
        "id"=>null,
        "bootstrapFile"=>null,
        "bootstrapCallable"=>null,
        "timeout"=>300,
        "maxWork"=>0,
        "logDir"=>null,
        "lastTime"=>null,
        "workDone"=>0,
        "pingTime"=>null,
        "status"=>"<init>",
        "memUsed"=>0,
        "enabled"=>true,
        "disabled"=>false,
        "active"=>false,
        "failures"=>0,
        "reassigns"=>0,
        "description"=>""
    ];

    public function __destruct() {
        if(!is_null($this->timer)) {
            Loop::cancelTimer($this->timer);
        }
    }

    public function __construct(Queue $queue) {
        $this->id = uniqid();
        $this->set("id",$this->id);
        $this->queue = $queue;
        $this->set("bootstrapFile"      , $this->getConfig("bootstrapFile"));
        $this->set("bootstrapCallable"  , $this->getConfig("bootstrapCallable"));
        $this->set("timeout"            , $this->getConfig("timeout"));
        $this->set("maxWork"            , $this->getConfig("maxWork"));
        $this->set("logDir"             , $this->getConfig("logDir"));
        
        $this->setLogging($this->get("logDir",false),$this->getConfig("logPerWorker"));
    }
    
    public function enable() {
        $this->set("enabled",true);
        $this->set("disabled",false);
        $this->queue->queueWorkerOnEnabled($this);
        $this->launch();
        Loop::addPeriodicTimer(0.1,function($timer) {
            $this->timer = $timer;
            $this->check();
            if(!$this->get("enabled",false) && !$this->get("active",false)) {
                Loop::cancelTimer($this->timer);
                $this->timer = null;
            }
        });
        return $this;
    }
    public function disable() {
        $this->set("enabled","false");
        $this->set("disabled",true);
        $this->stopWorkerProcess();
        $this->queue->queueWorkerOnDisabled($this);
        $this->queue->queueWorkerOnDeactivated($this);
        return $this;
    }

    private function check() {
        //dhGlobal::outLine("[QW] check..");
        if(!is_null($this->workerProcess)) {
            if(!$this->workerProcess->isRunning() || $this->workerProcess->isDone()) {
                $this->workerProcess = null;
            } else {
                $timeout = $this->get("timeout",0);
                if(!is_null($this->work) && ($workTimeout = $this->work->getMetaData("timeout",null)) !== null) {
                    $timeout = $workTimeout;
                }
                if($timeout>0 && $timeout< (microtime(true) - $this->get("lastTime"))) {
                    $this->logProcessError("timeout reached.. closing worker after timeout value of ",$timeout,"seconds elapsed");
                    if(!is_null($this->work)) {
                        $this->queue->workOnError($this->work,null,false);
                    }
                    $this->disable();
                } else {
                    /*$pingDiff = microtime(true) - max($this->get("pingTime",0),$this->lastPing);
                    if($pingDiff>5) {
                        $this->sendPing();
                    }*/
                }
            }
        }
        if(is_null($this->workerProcess) && $this->get("enabled",false)) {
            if($this->queue->hasQueued()) {
                $this->launch();
            }
        }
    }

    private function launch($force=false) {
        if(!is_null($this->workerProcess) && $force) {
            $this->stopWorkerProcess();
        } elseif(!is_null($this->workerProcess)) {
            return false;
        }
        $this->workerProcess = WorkerProcess::forQueue($this,$this->queue);
        QueueLogger::setObject($this->workerProcess);
        $this->workerProcess->start();
        $this->set("lastTime",microtime(true));
        $this->set("workDone",0);
    }
    private function stopWorkerProcess() {
        if(!is_null($this->workerProcess)) {
            if($this->workerProcess instanceof WorkerProcess) {
                $this->workerProcess->then(function($workerProcess) {
                    $this->logProcessInfo("stopped");
                },function($workerProcess) {
                    $this->logProcessInfo("stopped");
                });
            }
            $this->workerProcess->terminate();
        }
        if(!is_null($this->work)) {
            $this->stoppedWork = $this->work;
        }
        $this->workId = null;
        $this->work = null;
        $this->set("active",false);
        $this->queue->queueWorkerOnDeactivated($this);
        $this->workerProcess = null;
    }

    public function handleInit(Message $message) {
        $this->logProcessDebug("initialized");
        $this->set("lastTime",microtime(true));
        $this->setDisplay("<init>");
        $this->set("memUsed",$message->mem());
    }
    public function handleAck(Message $message) {
        $this->logProcessDebug("acknowledged");
        $this->set("lastTime",microtime(true));
        $this->set("memUsed",$message->mem());
    }
    public function handlePong(Message $message) {
        $this->logProcessDebug("pong");
        $this->set("pingTime",microtime(true));
        $this->set("memUsed",$message->mem());
        $this->queue->touchLastActivityTime();
    }
    public function handleLog(Message $message) {
        $this->set("lastTime",microtime(true));
        $this->set("memUsed",$message->mem());
        $this->callOnLog($this->workerProcess,$message);
    }
    public function handleReady(Message $message) {
        $this->set("lastTime",microtime(true));
        $this->setDisplay("<ready>");
        $this->set("memUsed",$message->mem());
        if($this->get("maxWork",0)>0 && $this->get("workDone",0) >= $this->get("maxWork",0)) {
            $this->logInfo("maxWork exceeded, relaunching");
            $this->launch(true);
        } else {
            if(!is_null($this->workId)) {
                $this->set("reassigns",$this->get("reassigns",0)+1);
                if($this->get("reassigns",0)>3) {
                    $this->queue->workOnError($this->work,null,false);
                    $this->logError("got the same work 3 times","workId:".$this->workId);
                    $this->work=null;
                    $this->workId=null;
                    $this->sendNoWork();
                } else {
                    $this->logDebug("re-assigedWork",$this->work->getId());
                    $this->sendWork($this->work);
                }
            } elseif(($work = $this->queue->next()) !== false) {
                $this->logDebug("assinged work",$work->getId());
                $this->work = $work;
                $this->workId = $work->getId();
                $this->setDisplay();//$work->getMetaData("name",$work->getId()));
                $this->sendWork($this->work);
                $this->set("reassigns",0);
            } else {
                $this->logInfo("noWork, disabling");
                $this->sendNoWork();
                $this->disable();
            }
        }
    }
    public function handleWork(Message $message) {
        $this->set("lastTime",microtime(true));
        $this->set("memUsed",$message->mem());
        if($message->isError()) {
            $this->queue->workOnError($this->work,$message,false);
            $this->work = null;
            $this->workId = null;
            //if killOnError...
        } else {
            $this->logProcessInfo("done with",$this->work->getId());
            $this->queue->workOnDone($this->work,$message);
            $this->set("workDone",$this->get("workDone")+1);
            $this->work = null;
            $this->workId = null;
        }
    }


    //response handling from the WorkerProcess:
    public function onStart() {
        $this->logProcessInfo("started");
        if(!is_null($this->get("bootstrapCallable",null))) {
            $this->sendBootstrap();
        }
        $this->set("active",true);
        $this->queue->queueWorkerOnActivated($this);
    }
    public function onError($chunk=null) {
        if(!is_null($chunk)) {
            $this->logProcessError($this->workerProcess,"error:",$chunk);
        } else {
            $this->logProcessError($this->workerProcess,"error");
        }
    }
    public function onDone($exitCode=null,$termSignal=null) {
        if(is_null($this->workerProcess)) {
            $workerId = "unknown";
        } else {
            $workerId = $this->workerProcess->getId();
        }
        dhGlobal::trace("[worker]",$workerId,"worker process exited");
        $this->logProcessInfo("exited");
        $this->setDisplay("<done>");
        $this->set("active",false);
        $this->queue->queueWorkerOnDeactivated($this);
    }
    public function onData($chunk=null) {
        if(($message = Message::fromPacket($chunk)) !== false) {
            if($message->getWorkId() != "log") {
                $this->logMsgRecv($message->toPacket());
            }
            if($message->getWorkId() == "init") {
                $this->logProcessInfo("received",$message->getWorkId());
                $this->logProcessDebug("messagePacket:",$message->toPacket());
                $this->handleInit($message);
            } elseif($message->getWorkId() == "ack") {
                $this->logProcessInfo("received",$message->getWorkId());
                $this->logProcessDebug("messagePacket:",$message->toPacket());
                $this->handleAck($message);
            } elseif($message->getWorkId() == "pong") {
                $this->logProcessInfo("received",$message->getWorkId());
                $this->logProcessDebug("messagePacket:",$message->toPacket());
                $this->handlePong($message);
            } elseif($message->getWorkId() == "ready") {
                $this->logProcessInfo("received",$message->getWorkId());
                $this->logProcessDebug("messagePacket:",$message->toPacket());
                $this->handleReady($message);
            } elseif($message->getWorkId() == "log") {
                $this->handleLog($message);
            } elseif($message->getWorkId() == "genwork") {
                $this->logProcessInfo("received",$message->getWorkId());
                $this->logProcessDebug("messagePacket:",$message->toPacket());
                $this->queue->workOnGenerator($this->work,$message);
            } elseif($message->getWorkId() == $this->workId) {
                $this->logProcessInfo("received",$message->getWorkId());
                $this->logProcessDebug("messagePacket:",$message->toPacket());
                $this->handleWork($message);
            }
        }
    }

    public function get($key=null,$default=null) {
        if(is_null($key)) {
            return $this->data;
        }
        return dhGlobal::getVal($this->data,$key,$default);
    }
    public function set($key,$value=null) {
        dhGlobal::dotAssign($this->data,$key,$value);
        return $this;
    }

    public function sendMessage($workId=null,$data=null,$work=null) {
        if(!is_null($this->workerProcess)) {
            $message = new Message();
            $message->setWorkId($workId);
            $message->setData($data);
            if($work && $work->isGenerator()) {
                $message->isGenerator(true);
            }
            dhGlobal::trace("sent","work",$message->toPacket());
            $this->logDebug("sendMessage",$this->workerProcess->getId()," :: ",$message->getWorkId());
            $this->logDebug("messagePacket:",$message->toPacket());
            $this->logMsgSend($message->toPacket());
            $this->workerProcess->write($message->toPacket()."\n");
        } else {
            $this->logError("sendMessage","no workerProcess to send to for workId",$workId);
        }
    }

    public function sendWork(Work $work) {
        $this->sendMessage($work->getId(),$work->getWork(),$work);
    }
    public function sendNoWork() {
        $this->sendMessage("nowork",["nowork"=>true,"time"=>microtime(true)]);
    }
    public function sendPing() {
        $this->lastPing = microtime(true);
        $this->sendMessage("ping",["ping"=>true,"time"=>microtime(true)]);
    }
    public function sendBootstrap() {
        $this->sendMessage("init",["callable"=>["static","bootstrap"],"args"=>[$this->getConfig("bootstrapCallable")]]);
    }

    private function logProcessInfo(...$args) {
        QueueLogger::info($this,...$args);
    }
    private function logProcessError(...$args) {
        $message = new Message();
        $message->setWorkId("log");
        $message->setData(["level"=>"error","message"=>implode(" ",$args)]);
        $this->callOnLog($this->workerProcess,$message);
        QueueLogger::error($this,...$args);
    }
    private function logProcessDebug(...$args) {
        QueueLogger::debug($this,...$args);
    }
    private function logInfo(...$args) {
        QueueLogger::info($this,...$args);
    }
    private function logError(...$args) {
        QueueLogger::error($this,...$args);
    }
    private function logDebug(...$args) {
        QueueLogger::debug($this,...$args);
    }
    private function logMsgSend(...$args) {
        QueueLogger::msgSend($this,...$args);
    }
    private function logMsgRecv(...$args) {
        QueueLogger::msgRecv($this,...$args);
    }

    private function setLogging($logDir=null,$logPerWorker=false) {
        QueueLogger::setObject($this,$logDir,$logPerWorker);
    }

    /**
     * Get the queueId
     */
    public function getQueueId() {
        return $this->queue->id();
    }
    public function queueId() {
        return $this->getQueueId();
    }
    public function queueWorkerId() {
        return $this->id();
    }

    public function getStoppedWork() {
        return $this->stoppedWork;
    }

    /**
     * Get the value of work
     */ 
    public function getWork()
    {
        return $this->work;
    }

    /**
     * Set the value of work
     *
     * @return  self
     */ 
    public function setWork($work)
    {
        $this->work = $work;

        return $this;
    }

    /**
     * Get the value of workId
     */ 
    public function getWorkId()
    {
        return $this->workId;
    }

    /**
     * Set the value of workId
     *
     * @return  self
     */ 
    public function setWorkId($workId)
    {
        $this->workId = $workId;

        return $this;
    }

    /**
     * Get the value of id
     */ 
    public function getId()
    {
        return $this->id;
    }
    public function id() {
        return $this->getId();
    }

    /**
     * Set the value of id
     *
     * @return  self
     */ 
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    public function setDisplay($value=null) {
        $staticValues = ["<init>","<ready>","<done>"];
        if(is_null($value) && !is_null($this->work)) {
            $description = $this->work->getMetaData("description","");
            $value = $this->work->getMetaData("name",$this->workId);
            $this->set("status",$value);
            $this->set("description",$description);
        } else {
            $this->set("status",$value);
            $this->set("description",$value);
        }
    }

    public function getQueueConfig() {
        return $this->queue->getQueueConfig();
    }
    public function getConfig($key=null,$default=null) {
        return $this->queue->get($key,$default);
    }
    public function setConfig($key,$value=null) {
        $this->queue->set($key,$value);
        return $this;
    }
    public function callOnLog($process,$message) {
        $callable = $this->getConfig("onLog");
        if(is_callable($callable)) {
            $callable($process,$message);
        } elseif(is_null($callable)) {
            if($message->get("level") == "error") {
                dhGlobal::outLine("****",strtoupper($message->get("level")),"|","WC-".$process->getId(),"|",$message->get("message"));
            }
        }
    }
}