<?php
namespace boru\dhprocess\queue;

use boru\dhprocess\config\Config;
use boru\dhprocess\config\QueueConfig;
use boru\dhprocess\config\WorkConfig;
use boru\dhprocess\config\WorkerConfig;
use boru\dhprocess\message\Result;
use boru\dhprocess\work\Work;
use boru\dhutils\dhGlobal;
use boru\dhutils\dhOut;
use boru\dhprocess\queue\QueueStatus;
use boru\dhutils\tools\StdOut;
use boru\dhutils\tools\Template;
use UnexpectedValueException;
use RuntimeException;
use Exception;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;

class Queue {
    private $id;
    /** @var Work[] */
    private $work=[];

    /** @var Work[] */
    private $queued=[],$active=[],$done=[];
    private $expected;

    private $retryCounts=[];

    /** @var QueueWorker[] */
    private $queueWorkers = [];
    private $queueWorkersEnabled = [];
    private $queueWorkersActive = [];

    /** @var Config */
    private $config;

    /** @var \React\Promise\Deferred */
    private $deferred;
    private $isDone = false;
    private $isWaiting = false;
    private $hasStarted = false;
    private $maxIdleTime = 300;
    private $lastActivity=null;
    private $lastIdleUpdate=0;
    private $maxQueued=false;
    private $workerCountEmptySince;

    /** @var QueueStatus */
    private $progress;

    /** @var dhOut */
    private $logger;

    /** @var Template */
    private $template;

    private $timerProgress;
    private $timerThreadUpdate;
    

    public function __construct($config=null) {
        $this->id = uniqid();
        if(is_null($config)) {
            $config = new Config();
        }
        $this->config = $config;

        $this->deferred = new Deferred();
        if($this->get("visualize",false)) {
            $this->template = new Template();
            $this->progress = new QueueStatus(" >> ");
            $this->promise()->then(function() {
                $this->updateProgress();
                $this->progress->stop();
            });
            if($this->get("extendedBar",false)) {
                $this->progress->showExec(true);
                $this->progress->setThreads($this->get("numWorkers",1),$this->get("threadTemplate"));
            }
            StdOut::progressBar($this->progress);
        }
    }
    public function progress() {
        return $this->progress();
    }
    public function __destruct() {
        if(!is_null($this->progress)) {
            $this->progress->stop();
        }
    }
    public function updateInterval($interval=0.5) {
        if($this->get("visualize",false) && !is_null($this->progress)) {
            $this->progress->updateInterval($interval);
        }
        return $this;
    }

    /**
     * Add work to the Queue
     * @param Work $work 
     * @return Work 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     * @throws Exception 
     */
    public function queue(Work $work,$skipMaxQueuedCheck=false) {
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
        }
        $this->work[$work->getId()] = $work;
        $this->moveToQueued($work);
        $maxQueued = intval($this->get("maxQueued",0));
        $queueSpace = 1;
        if(empty($maxQueued) || $skipMaxQueuedCheck) {
            $maxQueued = false;
        } else {
            $queueSpace = $maxQueued - count($this->queued);
        }
        if($maxQueued === false || $queueSpace <= 1) {
            $this->checkWorkers();
            $this->checkMaxQueued();
        }
        if(!$this->hasStarted && $this->get("visualize",false) && count($this->queued)%250 == 0) {
            $this->updateProgress(true);
        }
        //$this->total++;
        return $work;
    }

    public function reset() {
        $this->deferred=null;
        $this->hasStarted = false;
        $this->queued=[];
        $this->active=[];
        $this->done=[];
        $this->work=[];
        return $this;
    }
    
    /**
     * Get the next queued item, or false if nothing is queued 
     * @return Work|false */
    public function next() {
        $this->hasStarted = true;
        if(!empty($this->queued)) {
            $work = $this->nextWork();
            if($work) {
                $this->moveToActive($work);
                return $work;
            }
        }
        return false;
    }

    /**
     * This will find the next non-rate limited work item, or false if nothing is queued
     * @return Work|false 
     */
    private function nextWork() {
        $reAdd = [];
        $work = false;
        foreach($this->queued as $k=>$w) {
            if($w->rateLimitKey() && !$this->isWithinRateLimit($w->rateLimitKey())) {
                
            } else {
                $work = $w;
                unset($this->queued[$k]);
                break;
            }
        }
        return $work;
    }

    public function promise() {
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
            $this->deferred->resolve([]);
        }
        return $this->deferred->promise();
    }
    public function wait() {
        $this->log("wait called");
        $this->isWaiting=true;
        $this->isDone=false;
        $this->promise()->then(function() use (&$isDone) {
            $this->isDone=true;
        });
        $this->setupTimers();
        Loop::addPeriodicTimer(0.1,function($timer)  {
            if($this->isDone) {
                $this->stopTimers();
                Loop::cancelTimer($timer);
                Loop::stop();
            } else {
                $this->checkWorkers();
                if($this->workerCount() <= 0) {
                    if(is_null($this->workerCountEmptySince)) {
                        $this->workerCountEmptySince = microtime(true);
                    }
                    if((microtime(true)-$this->workerCountEmptySince) > 5) {
                        $this->isDone=true;
                    }
                } else {
                    $this->workerCountEmptySince = null;
                }
            }
        });
        while(!$this->isDone) {
            Loop::run();
        }
        //\React\Async\await($this->promise());
        return $this;
    }
    public function collect() {
        $this->wait();
        return $this->work;
    }
    private function checkMaxQueued() {
        $this->setupTimers();
        $maxQueued = intval($this->get("maxQueued",0));
        if(empty($maxQueued)) {
            $maxQueued = 0;
        }
        if($maxQueued>0 && count($this->queued) >= $maxQueued) {
            Loop::addPeriodicTimer(0.1,function($timer) use ($maxQueued) {
                if($maxQueued>0 && count($this->queued) >= $maxQueued) {
                } else {
                    Loop::stop();
                }
            });
            Loop::run();
        }
    }

    public function workerCount() {
        return count($this->queueWorkers);
    }
    public function workerActiveCount() {
        return count($this->queueWorkersActive);
    }
    public function workerEnabledCount() {
        return count($this->queueWorkersEnabled);
    }
    public function checkWorkers() {
        $queueCount = count($this->getQueued());
        $tempMax = min($this->get("numWorkers",1),$queueCount);
        //dhGlobal::outLine("checkWorkers: ",$queueCount,$tempMax,"w:",$this->workerCount(),"e",$this->workerEnabledCount());
        foreach($this->queueWorkers as $k=>$qw) {
            if($qw->get("disabled",false)) {
                if($qw->getStoppedWork()) {
                    $this->moveToQueued($qw->getStoppedWork());
                    dhGlobal::trace("re-queing",$qw->getStoppedWork()->getId());
                }
                dhGlobal::trace("Removing disabled queueWorker..");
                unset($this->queueWorkers[$k]);
            }
        }
        while($this->workerCount() < $tempMax) {
            $queueWorker = new QueueWorker($this);
            $queueWorker->enable();
            $this->queueWorkers[] = $queueWorker;
        }
    }

    public function id() {
        return $this->id;
    }

    /** @return bool true if queued array is not empty, false if empty */
    public function hasQueued(){
        return !empty($this->queued);
    }
    /** @return bool true if the active array is not empty, false if empty */
    public function hasActive(){
        return !empty($this->active);
    }
    /** @return bool true if the done array is not empty, false if empty */
    public function hasDone(){
        return !empty($this->done);
    }

    /**
     * Retrieve the work record by ID
     * @param string $workId 
     * @return Work|false 
     */
    public function getWork($workId) {
        return isset($this->work[$workId]) ? $this->work[$workId] : false;
    }
    /**
     * Retrieve full work array
     * @return Work[]|false 
     */
    public function getAllWork() {
        return !empty($this->work) ? $this->work : false;
    }

    /** @return bool true if $work is in the active state, false otherwise */
    public function workIsActive(Work $work) {
        return $this->getActive($work->getId()) !== false ? true : false;
    }
    /** @return bool true if $work is in the queued state, false otherwise */
    public function workIsQueued(Work $work) {
        return $this->getActive($work->getId()) !== false ? true : false;
    }
    /** @return bool true if $work is in the done state, false otherwise */
    public function workIsDone(Work $work) {
        return $this->getDone($work->getId()) !== false ? true : false;
    }

    /**
     * Get the ueue, or get a Work from the queue by ID
     * @param string|null $workId
     * @return Work[]|false|Work
     */
    public function getQueued($workId=null) {
        if(!is_null($workId)) {
            return dhGlobal::getVal($this->queued,$workId,false);
        }
        return $this->queued;
    }

    /**
     * Get the active queue, or get a Work from the queue by ID
     * @param string|null $workId
     * @return Work[]|false|Work
     */
    public function getActive($workId=null) {
        if(!is_null($workId)) {
            return dhGlobal::getVal($this->active,$workId,false);
        }
        return $this->active;
    }

    /**
     * Get the done queue, or get a Work from the queue by ID
     * @param string|null $workId
     * @return Work[]|false|Work
     */
    public function getDone($workId=null) {
        if(!is_null($workId)) {
            return dhGlobal::getVal($this->done,$workId,false);
        }
        return $this->done;
    }

    public function status() {
        $workers = count($this->queueWorkers);
        $activeWorkers = 0;
        foreach($this->queueWorkers as $k => $queueWorker) {
            if($queueWorker->get("active",false)) {
                $activeWorkers++;
            }
        }
        return [
            "done"=>count($this->getDone()),
            "expected"=>$this->expected,
            "active"=>count($this->getActive()),
            "queued"=>count($this->getQueued()),
            "workers"=>$workers,
            "activeWorkers"=>$activeWorkers,
            "threadData"=>$this->queueWorkers
        ];
    }

    public function updateProgress($force=false) {
        if(!$force) {
            if(!$this->get("visualize",false) || $this->workerCount()<=0 || is_null($this->progress)) {
                return;
            }
        }
        $this->progress->update(count($this->done),count($this->active),count($this->queued),$this->expected);
        if(!$force) {
            $this->progress->setThreadData($this->queueWorkers);
            if(!is_null($this->lastActivity)) {
                $lastActivityDiff = microtime(true)-$this->lastActivity;
                $lastIdleDisplay = microtime(true)-$this->lastIdleUpdate;
                $maxIdleTime = $this->get("maxIdleTime",300);
                $idleUpdateTime = $this->get("idleUpdateInterval",60);
                if($maxIdleTime >0 && $lastActivityDiff >= $maxIdleTime) {
                    $time = round(microtime(true)-$this->lastActivity);
                    $dtF = new \DateTime('@0');
                    $dtT = new \DateTime("@$time");
                    
                    $str = dhGlobal::dateIntervalToElapsed($dtF->diff($dtT),true,false,2,"");
                    dhGlobal::outLine("it has been",$str,"since anything happened.. closing");
                    if($this->isWaiting) {
                        $this->isDone=true;
                    } else {
                        exit();
                    }
                } elseif($idleUpdateTime >0 && min($lastActivityDiff,$lastIdleDisplay) >= $idleUpdateTime) {
                    $this->lastIdleUpdate = microtime(true);
                    $this->idleUpdate();
                }
                
            }
        } else {
            $this->progress->draw();
        }
    }

    public function touchLastActivityTime() {
        $this->lastActivity = microtime(true);
    }
    
    public function moveToQueued(Work $work) {
        $work->onWorkQueued();
        return $this->moveTo($work,$this->queued);
    }
    public function moveToActive(Work $work) {
        $work->onWorkStart();
        return $this->moveTo($work,$this->active);
    }
    public function moveToDone(Work $work) {
        $w = $this->moveTo($work,$this->done);
        if($this->get("discardOnDone",false)) {
            $this->done[$work->getId()] = null;
            unset($this->work[$work->getId()]);// = null;
        }
        return $w;
    }

    public function workOnGenerator(Work $work,$message=null) {
        $this->log("workOnGenerator",$message);
        if(!is_null($message)) {
            $work->onWorkGenerator($message->getDataObject());
        } else {
            $work->onWorkGenerator(new Result([],$work->getId(),false));
        }
    }
    public function workOnDone(Work $work,$message=null) {
        $this->logWork($work,"workOnDone");
        $this->moveToDone($work);
        if(!is_null($message)) {
            $work->done($message->getDataObject());
        } else {
            $work->done(new Result([],$work->getId(),false));
        }
        
    }
    public function workOnError(Work $work,$message=null,$workerClosed=false) {
        if($workerClosed) {
            $this->logWork($work,"workOnError","worker closed...");
            if($this->get("retriesOnError",0)) {
                dhGlobal::debug("Work stopped due to worker close, handling with retry process");
                $this->handleWorkOnError($work,$message);
            } else {
                $this->moveToQueued($work);
                dhGlobal::debug("Work re-queued due to worker close");
            }
        } else {
            $e = new \Exception();
            //dhGlobal::debug("workOnError called",$work->getId(),$e->getTraceAsString());
            //$this->logWork($work,"workOnError called",$e->getTraceAsString());
            $this->handleWorkOnError($work,$message);
        }
    }
    private function handleWorkOnError(Work $work,$message=null) {
        $this->logWork($work,"handleWorkOnError",!is_null($message) ? $message : "");
        $retryCount = isset($this->retryCounts[$work->getId()]) ? $this->retryCounts[$work->getId()] : 0;
        $retryCount++;
        $retriesOnError = $this->get("retriesOnError",0);
        if($retriesOnError>0 && $retryCount<=$retriesOnError) {
            dhGlobal::trace("[retry]","retry count for",$work->getId(),"is",$retryCount);
            $this->logWork($work,"RETRY #".$retryCount);
            $this->retryCounts[$work->getId()] = $retryCount;
            $this->moveToQueued($work);
        } else {
            //dhGlobal::debug("moveToDone",$work->getId());
            $this->moveToDone($work);
            if(!is_null($message) && is_object($message)) {
                $work->error($message->getErrorObject());
            } else {
                $work->error(new Result([],$work->getId(),true));
            }
        }
    }

    private function moveTo(Work $work,&$destinationArray) {
        $this->lastActivity = microtime(true);
        if(!isset($this->work[$work->getId()])) {
            //throw new \Exception("Work not found in work array");
            dhGlobal::error("[work]",$work->getId(),"Not found in work array, unable to move to other array");
            return;
        }
        unset($this->queued[$work->getId()]);
        unset($this->active[$work->getId()]);
        unset($this->done[$work->getId()]);
        $destinationArray[$work->getId()] = &$this->work[$work->getId()];
        if(!$this->hasActive() && !$this->hasQueued()) {
            if(!is_null($this->deferred)) {
                //we need to resolve the QueuePromise after the final 'work' promise, otherwise we will resolve the QueuePromise before the WorkPromise is processed.
                $work->always(function() {
                    if(!$this->hasQueued()) {
                        $this->deferred->resolve($this->done);
                    }
                });
            }
        }
        $this->updateProgress();
        return $work;
    }

    private function log(...$args) {
        array_unshift($args,"[QUEUE ]");
        $this->_log(...$args);
    }
    private function logWork(Work $work,...$args) {
        array_unshift($args,"[ WORK ]",":".$work->getId().":");
        $this->_log(...$args);
    }
    private function _log(...$args) {
        if($this->get("log",false) !== false) {
            if(is_null($this->logger)) {
                $this->logger = new dhOut(false,$this->get("log"));
            }
            $this->logger->line(...$args);
        }
    }

    public static function callableWork($callable,...$args) {
        $work = new Work($callable);
        $work->args($args);
        $work->asJson(false);
        return $work;
    }
    
    private function threadStatusDisplay() {
        $this->touchLastActivityTime();
        $templateFormat = "{id}   |  {lastTime|since|timeFormat|pad:8}   |  Mem:{memUsed|formatBytes} / done:{workDone|pad:3}  |  {description|max:100}";
        $templateFormat = $this->get("summaryTemplate",$templateFormat);
        dhGlobal::outLine("[THREAD STATUS UPDATE]");
        foreach($this->queueWorkers as $queueWorker) {
            $line = $this->template->parse($templateFormat,$queueWorker->get());
            dhGlobal::outLine($line);
            //dhGlobal::outLine($queueWorker->get());
        }
        dhGlobal::outLine("[/THREAD STATUS UPDATE]");
    }
    private function idleUpdate() {
        $status = $this->status();
        $templateFormat = "Queued: {queued}, Active: {active}   |   Workers: {workers} total, {activeWorkers} active";
        $templateFormat = $this->get("idleUpdateTemplate",$templateFormat);
        dhGlobal::outLine($this->template->parse($templateFormat,$status));
    }

    private function setupTimers() {
        if(is_null($this->timerProgress)) {
            $this->timerProgress = Loop::addPeriodicTimer(0.5,function($t) {
                $this->updateProgress();
            });
        }
        if(is_null($this->timerThreadUpdate)) {
            $this->timerThreadUpdate = Loop::addPeriodicTimer($this->get("threadStatusInterval",60),function($t) {
                if(!$this->isDone && $this->get("visualize",false)) {
                    $this->threadStatusDisplay();
                }
            });
        }
    }
    private function stopTimers() {
        if(!is_null($this->timerProgress)) {
            Loop::cancelTimer($this->timerProgress);
        }
        if(!is_null($this->timerThreadUpdate)) {
            Loop::cancelTimer($this->timerThreadUpdate);
        }
    }

    private $rateLimit = [];

    /**
     * Sets a rate limit for a key. If the key is called more than the limit within the time period, rateLimit will return false.
     * @param string $key The key/name to rate limit
     * @param int $limit The number of times the key can be called within the time period
     * @param int $timeInSeconds The time period in seconds
     * @return void 
     */
    public function setRateLimit($key,$limit,$timeInSeconds) {
        $this->rateLimit[$key] = [
            "limit"=>$limit,
            "time"=>$timeInSeconds,
            "last"=>0,
            "count"=>0,
        ];
    }

    /**
     * Returns true if the key is not rate limited, false if it is.
     * @param string $key 
     * @return bool 
     */
    public function isWithinRateLimit($key) {
        if(!isset($this->rateLimit[$key])) {
            //no rate limit set,
            return true;
        }
        $rl = &$this->rateLimit[$key];
        $now = time();
        if($rl["last"]+$rl["time"] < $now) {
            $rl["last"] = $now;
            $rl["count"] = 0;
        }
        if($rl["count"] < $rl["limit"]) {
            $rl["count"]++;
            //rate limit not reached, return true
            return true;
        }
        //rate limit reached, return false
        return false;
    }


    public function queueWorkerOnEnabled(QueueWorker &$queueWorker) {
        $this->queueWorkersEnabled[$queueWorker->getId()] = &$queueWorker;
    }
    public function queueWorkerOnActivated(QueueWorker &$queueWorker) {
        $this->queueWorkersActive[$queueWorker->getId()] = &$queueWorker;
    }
    public function queueWorkerOnDisabled(QueueWorker &$queueWorker) {
        unset($this->queueWorkersEnabled[$queueWorker->getId()]);
    }
    public function queueWorkerOnDeactivated(QueueWorker &$queueWorker) {
        unset($this->queueWorkersActive[$queueWorker->getId()]);
    }

    /**
	 * Get the value of expected
     * @return  int
     */
    public function getExpected() {
        return $this->expected;
    }

    /**
     * Set the value of expected
     * @param   int  $expected  
     * @return  self
	 */
    public function setExpected($expected) {
        $this->expected = $expected;
        return $this;
    }

    /**
     * Get the overall config
     * @param mixed $config 
     * @return Config|$this 
     */
    public function config($config=null) {
        if(is_null($config)) {
            return $this->config;
        } else {
            $this->config = $config;
            return $this;
        }
    }
    public function getQueueConfig() {
        return $this->config;
    }
    public function get($key,$default=null) {
        return $this->config->get($key,$default);
    }
    public function set($key,$value) {
        $this->config->set($key,$value);
        return $this;
    }
}