<?php
namespace boru\dhprocess\work;

use boru\dhprocess\message\ExceptionResult;
use boru\dhprocess\message\Result;
use boru\dhutils\dhGlobal;
use boru\dhutils\queue\QueueableInterface;
use React\Promise\Deferred;

class Work implements QueueableInterface {

    /** @var bool */
    private $running = false;
    /** @var bool */
    private $done = false;
    /** @var bool */
    private $ready = false;
    /** @var bool */
    private $success = false;

    private $id,$priority;
    
    /** @var \boru\dhutils\base\dhObject */
    private $result,$error;

    private $callable,$args,$asJson;
    private $isGenerator = false;

    /** @var callable */
    private $onStart,$onData,$onError,$onDone,$onQueued,$onGenerator;

    /** @var \React\Promise\Deferred */
    private $deferred;

    /** @var \boru\dhprocess\queue\Queue */
    private $queue;
    private $bootstrap;

    private $metaData;

    private $rateLimitKey;

    /**
     * 
     * @param callable $serializeableCallable either a string or an array, not a lambda function
     * @param mixed $args 
     * @param array $options 
     * @return void 
     */
    public function __construct($serializeableCallable,$options=[]) {
        if(is_object($serializeableCallable)) {
            throw new \Exception("serializeableCallable must be either a string or an array");
        }
        $this->id = uniqid();
        $this->callable  = $serializeableCallable;
        $this->args      = dhGlobal::getVal($options, "args",     null);
        $this->asJson    = dhGlobal::getVal($options, "asJson",   false);
        $this->onData    = dhGlobal::getVal($options, "onData",   null);
        $this->onError   = dhGlobal::getVal($options, "onError",  null);
        $this->onStart   = dhGlobal::getVal($options, "onStart",  null);
        $this->onDone    = dhGlobal::getVal($options, "onDone",   null);
        $this->bootstrap = dhGlobal::getVal($options, "bootstrap",null);
        $this->queue     = dhGlobal::getVal($options, "queue",    null);
        $this->deferred  = dhGlobal::getVal($options, "deferred", null);
        $this->metaData  = dhGlobal::getVal($options, "metaData", null);
        $this->isGenerator = dhGlobal::getVal($options, "isGenerator", false);
        $this->onGenerator = dhGlobal::getVal($options, "onGenerator", null);
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
        }
    }

    public function start() { 
        return $this->run();
    }
    public function destroy() {
        $this->id = null;
        $this->callable = null;
        $this->args = null;
        $this->asJson = null;
        $this->onData = null;
        $this->onError = null;
        $this->onStart = null;
        $this->onDone = null;
        $this->bootstrap = null;
        $this->queue = null;
        $this->deferred = null;
        $this->metaData = null;
        $this->result = null;
        $this->error = null;
        $this->onGenerator = null;
    }

    public function getId() {
        return $this->id;
    }
    public function getPriority() {
        return $this->priority;
    }
    public function getResult() {
        return $this->result;
    }
    public function getError() {
        return $this->error;
    }

    public function isDone() {
        return $this->done;
    }
    public function isRunning() {
        return $this->running;
    }
    public function isReady() {
        return $this->ready;
    }
    public function hasSuccess() {
        return $this->success;
    }
    public function isSuccess() {
        return $this->success;
    }
    public function isGenerator($isGenerator=null) {
        if(!is_null($isGenerator)) {
            $this->isGenerator = $isGenerator;
        }
        return $this->isGenerator;
    }
    /**
     * @param mixed|null $args 
     * @return mixed 
     */
    public function args($args=null) {
        if(is_null($args)) {
            return $this->args;
        }
        $this->args = $args;
        return $this;
    }
    /**
     * @param bool|null $asJson 
     * @return bool 
     */
    public function asJson($asJson=null) {
        if(is_null($asJson)) {
            return $this->asJson;
        }
        if($asJson) {
            $asJson = true;
        } else {
            $asJson = false;
        }
        $this->asJson = $asJson;
        return $this;
    }

    public function done($result) {
        $this->success=true;
        $this->onWorkDone($result);
    }
    public function error($data=null) {
        $this->onWorkError($data);
    }

    public function rateLimitKey($key=null) {
        if(is_null($key)) return $this->rateLimitKey;
        $this->rateLimitKey = $key;
        return $this;
    }

    public function terminate() { }

    public function getWork() {
        $packet = [];
        $packet["id"]       = $this->id;
        $packet["callable"] = $this->callable;
        $packet["args"]     = $this->args;
        $packet["asJson"]   = $this->asJson;
        return $packet;
    }

    /**
     * Wrapper for \React\Promise\Promise::then()
     * @param callable|null $onFulfilled 
     * @param callable|null $onRejected 
     * @return \React\Promise\PromiseInterface
     */
    public function then(callable $onFulfilled=null,callable $onRejected=null) {
        return $this->Promise()->then($onFulfilled,$onRejected);
    }
    public function always(callable $onFullfilledOrRejected=null) {
        return $this->Promise()->always($onFullfilledOrRejected);
    }

    /**
     * Making it easy to run work without having to queue.. creates a new queue for each run() call.
     * 
     * Options
     * * bootstrap - the bootstrap definition to use with the newly created queue.. ommitted if using existing queue
     * * queue - optional instance of \boru\dhutils\async\Queue, instead of creating a new queue, re-uses an existing one.
     * @param array $options 
     * @return $this 
     */
    public function run($options=[]) {
        $this->bootstrap = dhGlobal::getVal($options,"bootstrap",$this->bootstrap);
        $queue = dhGlobal::getVal($options,"queue",$this->queue);
        if(is_null($queue)) {
            $queue = dhGlobal::getAsyncQueue($this->bootstrap);
        }
        $queue->queue($this);
        return $this;
    }

    /** Handler Methods */
    public function onWorkDone($result=null) {
        $this->result = $result;
        $this->done = true;
        $this->running = false;
        $this->ready = false;
        $this->deferred->resolve($this->result);
        if(!is_null($this->onDone)) {
            $handler = $this->onDone;
            $handler($this,$this->result);
        }
    }
    public function onWorkError($chunk=null) {
        if(!is_null($this->onError)) {
            $handler = $this->onError;
            $handler($this,$chunk);
        }
        $this->error = $chunk;
        if(!is_null($this->deferred)) {
            if($chunk instanceof Result) {
                if($chunk->isError()) {
                    $e = new ExceptionResult($chunk);
                } else {
                    $e = new ExceptionResult($chunk);
                    $e->setMessage("Received workError with chunk: ".$chunk);
                }
            } else {
                $e = new \Exception("Received workError with chunk: ".$chunk);
            }
            $this->deferred->reject($e);
        }
    }
    public function onWorkData($chunk=null) {
        if(!is_null($this->onData)) {
            $handler = $this->onData;
            $handler($this,$chunk);
        }
    }
    public function onWorkStart() {
        $this->running = true;
        $this->ready = false;
        if(!is_null($this->onStart)) {
            $handler = $this->onStart;
            $handler($this);
        }
    }
    public function onWorkQueued() {
        $this->running = true;
        $this->ready = false;
        if(!is_null($this->onQueued)) {
            $handler = $this->onQueued;
            $handler($this);
        }
    }
    public function onWorkGenerator($result=null) {
        if(!is_null($this->onGenerator)) {
            $handler = $this->onGenerator;
            $handler($result);
        }
    }

    /**
     * Get the value of running
     * @return  bool
     */
    public function getRunning() {
        return $this->running;
    }

    /**
     * Set the value of running
     * @param   bool  $running  
     * @return  self
     */
    public function setRunning($running) {
        $this->running = $running;
        return $this;
    }

    /**
     * Get the value of done
     * @return  bool
     */
    public function getDone() {
        return $this->done;
    }

    /** Set the value of done
     * @param   bool  $done  
     * @return  self */
    public function setDone($done) {
        $this->done = $done;
        return $this;
    }

    /**
     * Set the Deferred that will be resolved on completion
     * @param Deferred $deferred 
     * @return $this 
     */
    public function setDeferred(Deferred $deferred) {
        $this->deferred = $deferred;
        return $this;
    }

    /**
     * Get the value of deferred
     *
     * @return  mixed
     */
    public function getDeferred() {
        return $this->deferred;
    }

    /** @return \React\Promise\Promise */
    public function Promise() {
        return $this->deferred->promise();
    }

    /**
	 * Get the value of queue
     * @return  mixed
     */
    public function getQueue() {
        return $this->queue;
    }

    /**
     * Set the value of queue
     * @param   mixed  $queue  
     * @return  self
	 */
    public function setQueue($queue) {
        $this->queue = $queue;
        return $this;
    }

    /**
	 * Get the value of bootstrap
     * @return  mixed
     */
    public function getBootstrap() {
        return $this->bootstrap;
    }

    /**
     * Set the value of bootstrap
     * @param   mixed  $bootstrap  
     * @return  self
	 */
    public function setBootstrap($bootstrap) {
        $this->bootstrap = $bootstrap;
        return $this;
    }

    public function onStart(callable $callback) {
        $this->onStart = $callback;
        return $this;
    }
    public function onDone(callable $callback) {
        $this->onDone = $callback;
        return $this;
    }
    public function onData(callable $callback) {
        $this->onData = $callback;
        return $this;
    }
    public function onError(callable $callback) {
        $this->onError = $callback;
        return $this;
    }
    public function onQueued(callable $callback) {
        $this->onQueued = $callback;
        return $this;
    }
    public function onGenerator(callable $callback) {
        $this->onGenerator = $callback;
        return $this;
    }

    public function setMetaData($key,$val="",$append=false) {
        if(strpos($key,".") !== false) {
            if($append) {
                $check = dhGlobal::getDot($this->metaData,$key);
                if(!is_null($check)) {
                    if(is_array($check)) {
                        $check[] = $val;
                        $val = $check;
                    } else {
                        $narr = [];
                        $narr[] = $check;
                        $narr[] = $val;
                        $val = $narr;
                    }
                }
            }
            dhGlobal::dotAssign($this->metaData,$key,$val);
        }
        else {
            if(isset($this->metaData[$key]) && $append) {
                if(is_array($this->metaData[$key])) {
                    $this->metaData[$key][] = $val;
                } else {
                    $temp = $this->metaData[$key];
                    $array[$key] = [];
                    $array[$key][] = $temp;
                    $array[$key][] = $val;
                }
            } else {
                $this->metaData[$key] = $val;
            }
        }
        return $this;
    }
    
    public function getMetaData($key=null,$default=null,$exists=false) {
        if(is_null($key)) {
            if($exists) {
                return !empty($this->metaData) ? true : false;
            } else {
                return !empty($this->metaData) ? $this->metaData : $default;
            }
        }
        if(strpos($key,".") !== false) {
            $uniqueid = uniqid("getArray",true);
            if(($check = dhGlobal::getDot($this->metaData,$key,$uniqueid)) !== $uniqueid) {
                return $exists ? true : $check;
            };
        }
        if($exists) {
            return isset($this->metaData[$key]);
        } else {
            return isset($this->metaData[$key]) ? $this->metaData[$key] : $default;
        }
        
    }

    public function metaDataExists($key) {
        return $this->getMetaData($key,null,true);
    }

    public static function fromCallable($callable,...$args) {
        $work = new self($callable);
        $work->args($args);
        return $work;
    }
    public static function fromCallableGenerator($callable,...$args) {
        $work = new self($callable);
        $work->args($args);
        $work->isGenerator =true;
        return $work;
    }
}