<?php
namespace boru\dhprocess\process;

use boru\dhprocess\process\traits\GettersSetters;
use boru\dhutils\async\Message;
use boru\dhutils\dhGlobal;
use boru\dhutils\multithread\parts\Buffer;
use boru\dhutils\queue\QueueableInterface;
use boru\dhutils\tools\ProcessQueue;
use React\Promise\Deferred;
use React\Promise\Promise;
use UnexpectedValueException;
use RuntimeException;

class Process {

    use GettersSetters;

    private $command,$args,$meta;
    private $metaData;

    private $data;

    /** @var bool */
    private $running = false;
    /** @var bool */
    private $done = false;
    /** @var bool */
    private $ready = false;
    
    /** @var Buffer */
    private $buffer;
    private $useBuffer = true;
    private $isMessageFrames = false;
    private $messageBuffer;

    private $pid,$exitCode,$termSignal;

    /** @var callable */
    private $onStart,$onData,$onError,$onDone,$onStreamClose,$onStreamError;

    /** @var \React\Promise\Deferred */
    private $deferred;

    /** @var \React\ChildProcess\Process */
    private $process;

    public function __construct($command,$args=[],$meta=[]) {
        $this->setCommand($command);
        $this->setArgs($args);
        $this->setMeta($meta); //sets all the handlers/etc
        $this->setPid(uniqid());

        $this->buffer = new Buffer();
    }

    /**
     * Starts the Child Process, returns the Promise for easy chaining
     * @return Promise|false 
     * @throws UnexpectedValueException 
     * @throws RuntimeException 
     */
    public function start() {
        $args = '';
        if(!empty($this->args)) {
            $args = ProcessQueue::pack($this->args);
        }
        $this->process = new \React\ChildProcess\Process('exec '.$this->command.' '.$args);
        $this->process->start();
        
        $this->setupProcessChannels();

        $this->onStart();
        return $this->Promise();
    }

    public function stop() {
        $this->running=false;
        $this->process->terminate();
    }

    
    public function terminate() {
        $this->running=false;
        $this->process->terminate();
        if(!is_null($this->deferred)) {
            $this->deferred->reject("terminated");
        }
    }

    public function get() {
        return [
            "pid"=>$this->pid,
            "running"=>$this->running,
            "done"=>$this->done,
            "meta"=>$this->meta,
            "args"=>$this->args,
            "command"=>$this->command,
            "buffer"=>$this->buffer,
            "exitCode"=>$this->exitCode,
            "termSignal"=>$this->termSignal,
            "data"=>$this->data,
        ];
    }

    public function write($data) {
        return $this->process->stdin->write($data);
    }

    /**
     * Get the buffer
     * @return Buffer 
     */
    public function Buffer() {
        return $this->buffer;
    }
    
    /**
     * Wrapper for \React\Promise\Promise::then()
     * @param callable|null $onFulfilled 
     * @param callable|null $onRejected 
     * @return PromiseInterface 
     */
    public function then(callable $onFulfilled=null,callable $onRejected=null) {
        return $this->Promise()->then($onFulfilled,$onRejected);
    }



    /** Handler Methods */
    public function onDone($exitCode=null,$termSignal=null) {
        $this->done = true;
        $this->running = false;
        $this->ready = false;
        $this->exitCode = $exitCode;
        $this->termSignal = $termSignal;
        $this->deferred->resolve($this->buffer);
        if(!is_null($this->onDone)) {
            $handler = $this->onDone;
            $handler($this,$exitCode,$termSignal);
        }
    }
    public function onError($chunk=null) {
        if(!is_null($this->onError)) {
            $handler = $this->onError;
            $handler($this,$chunk);
        }
        if($this->useBuffer) {
            $this->buffer->err($chunk);
        }
    }
    public function onData($chunk=null) {
        if($this->isMessageFrames) {
            $data = !empty($this->messageBuffer) ? $this->messageBuffer : "";
            $data.=$chunk;
            if(($message = Message::fromPacket($data)) !== false) {
                $chunk = $data;
                $this->messageBuffer = null;
            } else {
                $this->messageBuffer = $data;
                return;
            }
        }
        if(!is_null($this->onData)) {
            $handler = $this->onData;
            $handler($this,$chunk);
        }
        if($this->useBuffer) {
            $this->buffer->out($chunk);
        }
    }
    public function onStart() {
        $this->running = true;
        $this->ready = false;
        if(!is_null($this->onStart)) {
            $handler = $this->onStart;
            $handler($this);
        }
    }
    public function onStreamClose($type) {
        if(!is_null($this->onStreamClose)) {
            $handler = $this->onStreamClose;
            $handler($this,$type);
        }
    }
    public function onStreamError($type,$chunk=null) {
        if(!is_null($this->onStreamError)) {
            $handler = $this->onStreamError;
            $handler($this,$type,$chunk);
        }
    }

    /**
     * Set the stdout/stderr channel listening
     * @return void 
     */
    private function setupProcessChannels() {
        $this->process->stdout->on('data', function($chunk) {
            $this->onData(trim($chunk));
        });
        $this->process->stdout->on('error', function($chunk) {
            $this->onStreamError("stdout",$chunk);
        });
        $this->process->stdout->on('close', function() {
            $this->onStreamClose("stderr");
        });
        $this->process->stderr->on('data', function($chunk) {
            $this->onError(trim($chunk));
        });
        $this->process->stderr->on('error', function($chunk) {
            $this->onStreamError("stderr",$chunk);
        });
        $this->process->stderr->on('close', function() {
            $this->onStreamClose("stderr");
        });
        $this->process->on('exit', function($exitCode,$termSignal) {
            $this->onDone();
        });
    }

    
    /**
     * Set the value of meta
     * @param   mixed  $meta  
     * @return  self
     */
    public function setMeta($meta) {
        $this->meta = $meta;
        $this->onData   = dhGlobal::getVal($meta, "onData",  null);
        $this->onError  = dhGlobal::getVal($meta, "onError", null);
        $this->onStart  = dhGlobal::getVal($meta, "onStart", null);
        $this->onDone   = dhGlobal::getVal($meta, "onDone",  null);
        $this->deferred = dhGlobal::getVal($meta, "deferred",null);
        $this->onStreamClose = dhGlobal::getVal($meta, "onStreamClose",null);
        $this->onStreamError = dhGlobal::getVal($meta, "onStreamError",null);
        $this->useBuffer = dhGlobal::getVal($meta, "useBuffer", true);
        $this->isMessageFrames = dhGlobal::getVal($meta, "isMessageFrames", false);
        if(is_null($this->deferred)) {
            $this->deferred = new Deferred();
        }
        return $this;
    }

    public function jsonSerialize($array=null) {
        if(is_null($array)) {
            $array = $this->get();
        }
        return $array;
    }
    public function __toString() {
        return json_encode($this);
    }
}