<?php

use boru\dhprocess\message\ExceptionMessage;
use boru\dhutils\dhGlobal;
use boru\dhutils\dhOut;
use boru\dhprocess\message\Message;
use boru\dhprocess\queue\QueueLogger;
use boru\dhprocess\worker\WorkerException;
use boru\dhprocess\worker\WorkerUtils;
use React\Stream\ReadableResourceStream;
use React\Stream\WritableResourceStream;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;
use Opis\Closure\SerializableClosure;

$preloadPackets = $includes = [];
$maxWaitBeforeExit=5;
$logDir = null;
$logPerWorker = false;
$parentId = null;
$queueId = null;
$workerId = null;
$logFilePrefix = null;
$logSettings = null;
if(isset($argv) && is_array($argv) && !empty($argv)) {
    array_shift($argv);
    foreach($argv as $i=>$v) {
        cwSetIfArg("log",$v,$logDir);
        cwSetIfArg("id",$v,$parentId);
        cwSetIfArg("qid",$v,$queueId);
        cwSetIfArg("wid",$v,$workerId);
        cwSetIfArg("ll",$v,$logSettings);
        cwPacketOrInclude($v,$preloadPackets,$includes);
        if($v == "debug") {
            $maxWaitBeforeExit=0;
        }
    }
}
if(!empty($includes)) {
    foreach($includes as $include) {
        if(file_exists($include)) {
            require_once $include;
        }
    }
}

$pparts = explode(DIRECTORY_SEPARATOR,__DIR__);
$pparts = array_reverse($pparts);

if($pparts[2] == "dhprocess" && $pparts[4] == "vendor") {
    require_once __DIR__."/../../../../autoload.php";
} else {
    require_once __DIR__."/../../vendor/autoload.php";
}

/**
 * I understand completely that we shouldn't throw random ini_set definitions within libraries, but centos does weird things.
 */
if(php_sapi_name() == "cli") {
    ini_set("memory_limit",-1);
}

if(is_null($parentId)) $parentId = uniqid();

//if(!is_null($logDir)) {
    QueueLogger::initClient($parentId,$logDir);
//}
QueueLogger::getConfig()->logLevelsFromTransport(empty($logSettings) ? "" : $logSettings);

$utilWorker = new Worker($preloadPackets,$maxWaitBeforeExit,$parentId,$workerId,$queueId);

/**
 * @name WorkerClient
 */
class Worker {
    private $id;
    private $delay = 0.1;
    private $maxWaitBeforeExit = 5;
    public static $bootstrap;
    private $running = false;
    private $checking = false;
    private $terminated = false;
    private $queueId = null;
    private $queueWorkerId = null;

    private $readyLastSent;

    private static $enclosurePrefix = 'C:32:"Opis\Closure\SerializableClosure"';
    

    /** @var \React\Stream\ReadableResourceStream */
    private $stdin;
    /** @var \React\Stream\WritableResourceStream */
    private $stdout,$stderr;

    private $partialLine = "";

    private $buffersInit = false;

    public function __construct($preloadPackets=null,$maxWaitBeforeExit=5,$parentId=null,$queueWorkerId=null,$queueId=null) {
        $this->id = !is_null($parentId) ? $parentId : uniqid();
        QueueLogger::setObject($this);
        $this->maxWaitBeforeExit = $maxWaitBeforeExit;
        $this->queueWorkerId = $queueWorkerId;
        $this->queueId = $queueId;
        stream_set_blocking(STDIN, 0);
        if(!empty($preloadPackets) && is_array($preloadPackets)) {
            foreach($preloadPackets as $packet) {
                $this->processPacket($packet);
            }
        }
        if(($ml = dhGlobal::get("asyncMaxLogResponseLength",false)) !== false) {
            $this->maxLogResponseLength = $ml;
        }

        $this->stdin = new ReadableResourceStream(STDIN);
        $this->stdout = new WritableResourceStream(STDOUT);
        $this->stderr = new WritableResourceStream(STDERR);

        $this->stdin    ->on("data", function($chunk) { $this->onInput("stdin",$chunk); });
        $this->stdout   ->on("data", function($chunk) { $this->onInput("stdout",$chunk); });
        $this->stderr   ->on("data", function($chunk) { $this->onInput("stderr",$chunk); });
        $this->stdin    ->on("close",function() { $this->onClose("stdin"); });
        $this->stdout   ->on("close",function() { $this->onClose("stdout"); });
        $this->stderr   ->on("close",function() { $this->onClose("stderr"); });
        $this->stdin    ->on("error",function(\Exception $e) { $this->onError("stdin",$e); });
        $this->stdout   ->on("error",function(\Exception $e) { $this->onError("stdout",$e); });
        $this->stderr   ->on("error",function(\Exception $e) { $this->onError("stderr",$e); });

        Loop::addPeriodicTimer($this->delay,function($timer) {
            if($this->generatorDeferred !== null) {
                $this->checkGeneratorTasks();
            }
            if(!$this->running && !$this->checking) {
                $this->sendMessage($this->makeReadyMessage(),false);
                $this->checking=true;
                $this->readyLastSent = microtime(true);
            }
            if($this->maxWaitBeforeExit>0 && $this->checking && !is_null($this->readyLastSent) && microtime(true)-$this->readyLastSent>$this->maxWaitBeforeExit) {
                $elapsed = microtime(true)-$this->readyLastSent;
                $this->sendMessage($this->responseMessage("info",["what"=>"terminating","why"=>round($elapsed,2)." seconds elapsed since 'ready' was sent, exceeds limit of ".$this->maxWaitBeforeExit." seconds"]),false);
                $this->terminate(false);
            }
        });
        $this->buffersInit = true;
        $this->logInfo("init complete, started loop");
    }
    public function __destruct() {
        if(!$this->terminated) {
            $this->logInfo("process exited abruptly");
        }
        $this->logInfo("process exited");
    }
    
    private function onInput($streamType,$chunk) {
        
        if($streamType == "stdin") {
            if(substr($chunk,-1) != "\n") {
                $this->partialLine.=$chunk;
            } else {
                $this->logMsgRecv(trim($this->partialLine.$chunk));
                $data = explode("\n",$this->partialLine.$chunk);
                $this->partialLine="";
                foreach($data as $line) { 
                    $this->processPacket($line);
                }
            }
        } else {
            $this->logInfo("[".$streamType."]",trim($chunk));
        }
    }
    private function onClose($streamType) {
        $this->logInfo("[".$streamType."]","---CLOSED---");
    }
    private function onError($streamType,\Exception $e) {
        $this->logInfo("[".$streamType."]","---ERROR---",$e->getMessage());
    }

    public function processPacket($framedPacket) {
        if(($message = Message::fromPacket($framedPacket)) !== false) {
            $this->logDebug("processPacket: $framedPacket");
            $this->handlePacketMessage($message)->then(function($message) {
                $this->sendMessage($message);
            },function($message) {
                $this->sendMessage($message);
            });
        } else {
            if(!empty(trim($framedPacket))) {
                $this->logError("unable to parse packet: $framedPacket");
            }
        }
    }
    private function handlePacketMessage($message) {
        $packetDeferred = new Deferred();
        $this->running=true;
        if(!$message->isValid()) {
            $packetDeferred->reject($this->makeErrorMessage(0,Message::E_FRAME_ERROR,"unable to parse message frame"));
            
        } else {
            $workId = $message->getWorkId();
            $this->sendMessage($this->makeAckMessage($workId),false);

            if($workId == "nowork"){
                $this->processNoWork($workId,$message,$packetDeferred);
            } elseif($workId == "ping") {
                $this->processPing($workId,$message,$packetDeferred);
            } elseif($message->isGenerator()) {
                $this->processGenerator($workId,$message,$packetDeferred);
            } elseif($message->get("callable",false) !== false) {
                $this->processCallable($workId,$message,$packetDeferred);

            } else {
                $this->checking=false;
                $packetDeferred->reject($this->makeErrorMessage($workId,Message::E_UNKNOWN_FRAME,$message->get()));

            }
        }
        return $packetDeferred->promise();
    }

    private function processPing($workId,$message,&$packetDeferred) {
        $packetDeferred->resolve($this->makePongMessage());
    }
    private function processNoWork($workId,$message,&$packetDeferred) {
        $this->checking=false;
        $packetDeferred->resolve(true);
        Loop::stop();
        $this->terminate();
        return $packetDeferred;
    }
    
    private function processCallable($workId,$message,&$packetDeferred) {
        if(($callable = $message->get("callable",false)) !== false) {
            $this->checking=false;
            $args = $message->get("args",[]);
            $asJson = $message->get("asJson",false);
            $stdOutFromExec = "";
            $executeResult = null;
            try {
                ob_start();
                $executeResult = $this->execute($workId,$callable,$args,$asJson);
                $stdOutFromExec = ob_get_contents();
                if(!empty($stdOutFromExec)) {
                    dhGlobal::info("Exec Output Buffer:",$stdOutFromExec);
                }
                ob_end_clean();
            } catch(\Exception $e) {
                $packetDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$e->getMessage(),$e->getTrace()));
                return $packetDeferred;
            }
            if($executeResult instanceof Message) {
                $packetDeferred->resolve($executeResult);
            } elseif($executeResult instanceof ExceptionMessage) {
                $packetDeferred->reject($executeResult);
            } elseif($executeResult instanceof PromiseInterface) {
                $executeResult->then(function($executeMessage) use ($packetDeferred) {
                    $packetDeferred->resolve($executeMessage);
                },function($e) use($workId,$packetDeferred) {
                    $packetDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$e->getMessage(),$e->getTrace()));
                });
            }
        } else {
            $packetDeferred->resolve(false);
        }
        return $packetDeferred;
    }
    private function callableToString($callable) {
        $callable = $this->parseCallable($callable);
        if(is_array($callable)) {
            $callable = implode("::",$callable);
        } elseif(is_object($callable)) {
            $callable = get_class($callable);
        }
    }

    private function parseCallable($callable) {
        if(is_string($callable)) {
            $wrapper = \Opis\Closure\unserialize($callable);
            if($wrapper && is_callable($wrapper)) {
                return $wrapper;
            }
            if(0==1) {// &&$wrapper && is_object($wrapper) && $wrapper instanceof SerializableClosure) {
                $callable = $wrapper->getClosure();
                if(is_callable($callable)) {
                    return $callable;
                }
            } else {
                if(strpos($callable,"::") !== false) {
                    $callable = explode("::",$callable);
                } else {
                    $callable = false;
                }
            }
        }
        if(is_array($callable) && count($callable) == 2) {
            $class = $callable[0];
            $method = $callable[1];
            if(method_exists($class,$method)) {
                return $callable;
            }
        }
        return false;
    }


    private function execute($workId,$callable,$args,$asJson=false) {
        $this->logExecuteStart($callable,$workId);
        if(!is_array($args)) { $args = [$args]; }
        $result = $output = "";
        $trace = null;
        $success = false;
        $callable = $this->parseCallable($callable);
        if(!is_callable($callable)) {
            $this->logExecuteError($callable,Message::E_NOT_CALLABLE,"",$workId);
            $stringCallable = is_array($callable) ? implode("::",$callable) : $callable;
            return $this->makeErrorMessage($workId,Message::E_NOT_CALLABLE,"$stringCallable is not callable",null);
        }
        $result = call_user_func($callable,...$args);
        $success = true;
        /*try {
            $result = call_user_func($callable,...$args);
            $success = true;
        } catch (WorkerException $e) {
            $this->logExecuteError($callable,"worker_exception",$e->getMessage(),$workId);
            return $this->makeErrorMessage($workId,$e->getType(),$e->getMessage());
        } catch (\Exception $e) {
            $success=false;
            $result = $e->getMessage();
            $trace = $e->getTrace();
        }*/
        if($result instanceof \React\Promise\PromiseInterface) {
            return $this->handleExecDeferredResult($result,$workId,$success,$output,$callable,$asJson,$trace);
        } else {
            return $this->handleExecDirectResult($result,$workId,$success,$output,$callable,$asJson,$trace);
        }
    }

    private function processGenerator($workId,$message,&$packetDeferred) {
        if(($callable = $message->get("callable",false)) !== false) {
            $this->checking=false;
            $args = $message->get("args",[]);
            $asJson = $message->get("asJson",false);
            $stdOutFromExec = [];
            $executeResult = null;
            $message = $this->makeSuccessMessage($workId,["callable"=>$callable,"result"=>"gen-done","stdout"=>""]);
            try {
                ob_start();
                $executeResult = $this->executeGenerator($workId,$callable,$args,$asJson);
                $executeResult->then(function($executeMessage) use ($message,&$packetDeferred) {
                    $this->logInfo("executeGenerator: promise resolved");
                    $packetDeferred->resolve($message);
                },function($e) use($workId,&$packetDeferred) {
                    $this->logInfo("executeGenerator: promise rejected");
                    $packetDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$e->getMessage(),$e->getTrace()));
                });
                $stdOutFromExec = ob_get_contents();
                if(!empty($stdOutFromExec)) {
                    dhGlobal::info("Exec Output Buffer:",$stdOutFromExec);
                }
                ob_end_clean();
            } catch(\Exception $e) {
                $packetDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$e->getMessage(),$e->getTrace()));
            }
        } else {
            $packetDeferred->resolve(false);
        }
    }
    private $generatorWorkId;
    private $generatorDeferred;
    private $generatorTasks = [];
    private function checkGeneratorTasks() {
        if(empty($this->generatorTasks)) {
            $this->generatorDeferred->resolve($this->makeSuccessMessage($this->generatorWorkId,["callable"=>"generator","result"=>"gen-done","stdout"=>""]));
            $this->generatorDeferred = null;
        }
    }   
    private function executeGenerator($workId,$callable,$args,$asJson=false) {
        $this->generatorWorkId = $workId;
        $this->generatorDeferred = new Deferred();
        $this->logGeneratorStart($callable,$workId);
        if(!is_array($args)) { $args = [$args]; }
        $result = $output = "";
        $trace = null;
        $success = false;
        $callable = $this->parseCallable($callable);
        if(!is_callable($callable)) {
            $this->logExecuteError($callable,Message::E_NOT_CALLABLE,"",$workId);
            $stringCallable = is_array($callable) ? implode("::",$callable) : $callable;
            return $this->makeErrorMessage($workId,Message::E_NOT_CALLABLE,"$stringCallable is not callable",null);
        }
        try {
            $success = true;
            $genI=0;
            foreach(call_user_func($callable,...$args) as $result) {
                $genI++;
                if($result instanceof \React\Promise\PromiseInterface) {
                    $generatorResult = $this->handleExecDeferredResultGenerator($result,"genwork",$success,$output,$callable,$asJson,$trace,$genI);
                    
                } else {
                    $generatorResult = $this->handleExecDirectResultGenerator($result,"genwork",$success,$output,$callable,$asJson,$trace);
                    $this->sendMessage($generatorResult,false);
                }
            }
            $success = true;
        } catch (WorkerException $e) {
            $this->logExecuteError($callable,"worker_exception",$e->getMessage(),$workId);
            return $this->makeErrorMessage($workId,$e->getType(),$e->getMessage());
        } catch (\Exception $e) {
            $success=false;
            $result = $e->getMessage();
            $trace = $e->getTrace();
        }
        return $this->generatorDeferred->promise();
        
    }
    private function handleExecDirectResultGenerator($result,$workId,$success,$output,$callable,$asJson,$trace) {
        $rawResult = null;
        if($success && $asJson && !is_array($result)) {
            $rawResult = $result;
            $result = json_decode($result,true);
        }
        if(!$success) {
            $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION,$result,$workId);
            return $this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$result,$trace);
        }
        $returnData = ["callable"=>$callable,"result"=>$result,"stdout"=>$output];
        if(!is_null($rawResult)) {
            $returnData["raw"]=$rawResult;
        }
        $this->logGeneratorResult($callable,$workId);
        return $this->makeSuccessMessage($workId,$returnData);
    }
    private function handleExecDeferredResultGenerator($result,$workId,$success,$output,$callable,$asJson,$trace,$genI=0) {
        $execDeferred = new Deferred();
        $result->then(function($actualResult) use ($workId,$success,$output,$callable,$asJson,$trace,&$execDeferred,$genI) {
            
            $rawResult = null;
            if($success && $asJson && !is_array($actualResult)) {
                $rawResult = $actualResult;
                $actualResult = json_decode($actualResult,true);
            }
            if(!$success) {
                $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION,$rawResult,$workId);
                $execDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$actualResult,$trace));
                return false;
            } else {
                $returnData = ["callable"=>$callable,"result"=>$actualResult,"stdout"=>$output];
                if(!is_null($rawResult)) {
                    $returnData["raw"]=$rawResult;
                }
                $this->logGeneratorResult($callable,$workId);
                $message = $this->makeSuccessMessage($workId,$returnData);
                $this->sendMessage($message,false);
                $execDeferred->resolve($message);
                
            }
            unset($this->generatorTasks[$genI]);
        },function($e) use ($workId,$callable,&$execDeferred,$genI) {
            if(is_object($e) && method_exists($e,"getMessage")) {
                $result = $e->getMessage();
            } else {
                $result = $e;
            }
            if(is_object($e) && method_exists($e,"getTrace")) {
                $trace = $e->getTrace();
            } else {
                $trace = null;
            }
            $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION,$result,$workId);
            $execDeferred->resolve($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$result,$trace));
            unset($this->generatorTasks[$genI]);
            return false;
        });
        $this->generatorTasks[$genI] = $execDeferred;
        return $execDeferred->promise();
    }
    private function handleExecDirectResult($result,$workId,$success,$output,$callable,$asJson,$trace) {
        $rawResult = null;
        if($success && $asJson && !is_array($result)) {
            $rawResult = $result;
            $result = json_decode($result,true);
        }
        if(!$success) {
            $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION,$result,$workId);
            return $this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$result,$trace);
        }
        $returnData = ["callable"=>$callable,"result"=>$result,"stdout"=>$output];
        if(!is_null($rawResult)) {
            $returnData["raw"]=$rawResult;
        }
        $this->logExecuteSuccess($callable,$workId);
        return $this->makeSuccessMessage($workId,$returnData);
    }
    private function handleExecDeferredResult($result,$workId,$success,$output,$callable,$asJson,$trace) {
        $execDeferred = new Deferred();
        $result->then(function($actualResult) use ($workId,$success,$output,$callable,$asJson,$trace,&$execDeferred) {
            $rawResult = null;
            if($success && $asJson && !is_array($actualResult)) {
                $rawResult = $actualResult;
                $actualResult = json_decode($actualResult,true);
            }
            if(!$success) {
                $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION,$rawResult,$workId);
                $execDeferred->reject($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$actualResult,$trace));
                return false;
            } else {
                $returnData = ["callable"=>$callable,"result"=>$actualResult,"stdout"=>$output];
                if(!is_null($rawResult)) {
                    $returnData["raw"]=$rawResult;
                }
                $this->logExecuteSuccess($callable,$workId);
                $execDeferred->resolve($this->makeSuccessMessage($workId,$returnData));
                
            }
        },function($e) use ($workId,$callable,&$execDeferred) {
            if(is_object($e) && method_exists($e,"getMessage")) {
                $result = $e->getMessage();
            } else {
                $result = $e;
            }
            if(is_object($e) && method_exists($e,"getTrace")) {
                $trace = $e->getTrace();
            } else {
                $trace = null;
            }
            $this->logExecuteError($callable,Message::E_EXECUTE_EXCEPTION,$result,$workId);
            $execDeferred->resolve($this->makeErrorMessage($workId,Message::E_EXECUTE_EXCEPTION,$result,$trace));
            return false;
        });
        return $execDeferred->promise();
    }

    private function logExecuteStart($callable,$msg="",$workId=null) {
        $stringCallable = $this->callableToString($callable);
        $this->logInfo("execute",$stringCallable,$workId,"START  ",$msg);
    }
    private function logGeneratorStart($callable,$msg="",$workId=null) {
        $stringCallable = $this->callableToString($callable);
        $this->logInfo("generator",$stringCallable,$workId,"START  ",$msg);
    }
    private function logGeneratorResult($callable,$msg="",$workId=null) {
        $stringCallable = $this->callableToString($callable);
        $this->logInfo("generator",$stringCallable,$workId,"RESULT  ",$msg);
    }
    private function logExecuteSuccess($callable,$msg="",$workId=null) {
        $stringCallable = $this->callableToString($callable);
        $this->logInfo("execute",$stringCallable,$workId,"SUCCESS ".$msg);
    }
    private function logExecuteError($callable,$type="",$msg="",$workId=null) {
        $stringCallable = $this->callableToString($callable);
        $this->logError("execute",$stringCallable,$workId,strtoupper($type)." - ".$msg);
    }

    public function makeSuccessMessage($workId,$data) {
        return $this->responseMessage($workId,$data);
    }
    public function makeErrorMessage($workId,$code,$message="",$trace=null) {
        return $this->responseMessage($workId,null,["code"=>$code,"message"=>$message,"trace"=>null],true);
    }
    public function makeReadyMessage() {
        return $this->responseMessage("ready",["time"=>microtime(true),"mem"=>memory_get_usage()]);
    }
    public function makeAckMessage($workId=null) {
        return $this->responseMessage("ack",["workId"=>$workId,"time"=>microtime(true),"mem"=>memory_get_usage()]);
    }
    public function makePongMessage() {
        return $this->responseMessage("pong",["time"=>microtime(true),"mem"=>memory_get_usage()]);
    }
    public function makeLogMessage($message,$logLevel="info") {
        if(is_array($message)) {
            $message = implode(" ",$message);
        }
        return $this->responseMessage("log",["message"=>$message,"level"=>$logLevel,"time"=>microtime(true),"mem"=>memory_get_usage()]);
    }
    /**
     * @param string $workId
     * @param mixed $data
     * @param mixed $error
     * @param bool $asException
     * @return Message|ExceptionMessage
     */
    public function responseMessage($workId,$data=null,$error=null,$asException=false) {
        $message = new Message();
        $message->setWorkId($workId);
        $message->setData($data);
        $message->setError($error);
        $message->time(microtime(true));
        $message->mem(memory_get_usage());
        if($asException) {
            return new ExceptionMessage($message,1);
        }
        return $message;
    }
    /**
     * 
     * @param Message|ExceptionMessage $message 
     * @param bool $updateRunning 
     * @param bool $skipLog 
     * @return void 
     */
    public function sendMessage($message,$updateRunning=true,$skipLog=false) {
        if($message instanceof ExceptionMessage) {
            $message = $message->messageObject();
        }
        if(!$skipLog) {
            $this->logMsgSend($message->toPacket());
        }
        $this->stdout->write($message->toPacket()."\n");
        if($updateRunning) {
            $this->running=false;
        }
    }
    private function terminate($success=true) {
        $this->terminated=true;
        if(!$success) {
            exit(1);
        }
        exit();
    }

    private function log($logLevel,...$args) {
        if($logLevel == "info") {
            $this->logInfo(...$args);
        } else if($logLevel == "error") {
            $this->logError(...$args);
        } else if($logLevel == "debug") {
            $this->logDebug(...$args);
        } else {
            QueueLogger::log($logLevel,...$args);
        }
    }
    private function logMsgSend(...$args) {
        QueueLogger::msgSend($this,...$args);
    }
    private function logMsgRecv(...$args) {
        QueueLogger::msgRecv($this,...$args);
    }
    private function logInfo(...$args) {
        QueueLogger::info($this,...$args);
    }
    private function logError(...$args) {
        QueueLogger::error($this,...$args);
    }
    private function logDebug(...$args) {
        QueueLogger::debug($this,...$args);
    }

    public function getId() {
        return $this->id;
    }
    public function id() {
        return $this->id;
    }
    public function queueWorkerId() {
        return $this->queueWorkerId;
    }
    public function queueId() {
        return $this->queueId;
    }
    /**
     * Send a log message to the parent process
     * @param mixed $message 
     * @param string $logLevel info|error|debug
     * @return void 
     */
    public function sendLogMessage($message,$logLevel="info") {
        $this->sendMessage($this->makeLogMessage($message,$logLevel),false,true);
    }

    public static function bootstrap($bootstrap) {
        return WorkerUtils::bootstrap($bootstrap);
    }
    public static function testError(...$args) {
        return WorkerUtils::testError(...$args);
    }
    public static function exec(...$args) {
        return WorkerUtils::exec(...$args);
    }
    public static function http(...$args) {
        return WorkerUtils::http(...$args);
    }
}

//helper function for initial load
function cwSetIfArg($key,$arg,&$var) {
    if(substr($arg,0,strlen($key)+1) == $key.":") {
        $var = substr($arg,strlen($key)+1);
        return $var;
    }
    return false;
}
function cwPacketOrInclude($arg,&$preloadPackets,&$includes) {
    if(substr($arg,0,4) == "inc:") {
        $includes[] = substr($arg,4);
        return true;
    }
    if(substr($arg,0,4) == "pkt:") {
        $preloadPackets[] = substr($arg,4);
        return true;
    }
    return false;
}