<?php
namespace boru\dhsession;

use boru\dhsession\handlers\HandlerInterface;

class Session {
    /**
     * @var HandlerInterface $handler
     */
    private static $handler;

    /**
     * @var bool $hasInit
     */
    private static $hasInit = false;

    public static function setHandler($handler,$args=[]) {
        if($handler instanceof HandlerInterface) {
            self::$handler = $handler;
        } else {
            try {
                self::$handler = self::createHandler($handler,$args);
            } catch(\Exception $e) {
                self::$handler = self::createHandler("default",$args);
            }
        }
    }
    public static function getHandler() {
        return self::$handler;
    }
    public static function createHandler($type="default",$options=[]) {
        $class = "boru\\dhsession\\handlers\\".ucfirst($type)."Handler";
        if(!class_exists($class)) {
            throw new \Exception("Handler $type not found");
        }
        return new $class($options);
    }
    public static function start($handler=null) {
        if($handler) {
            self::setHandler($handler);
        }
        self::assertInit();
    }
    public static function get($key) {
        self::assertInit();
        return isset($_SESSION[$key]) ? $_SESSION[$key] : null;
    }
    public static function set($key, $value) {
        self::assertInit();
        $_SESSION[$key] = $value;
    }
    public static function delete($key) {
        self::assertInit();
        unset($_SESSION[$key]);
    }
    public static function destroy() {
        self::assertInit();
        session_destroy();
    }
    public static function refresh() {
        self::assertInit();
        self::$handler->refresh();
    }
    public static function getId() {
        self::assertInit();
        return self::$handler->getSessionId();
    }
    public static function getName() {
        self::assertInit();
        return self::$handler->getSessionName();
    }
    public static function setMetaId($metId) {
        self::assertInit();
        self::$handler->setMetaId($metId);
    }
    public static function getMetaId() {
        self::assertInit();
        return self::$handler->getMetaId();
    }


    //Management functions
    public static function getSession($sessionId) {
        self::assertInit();
        $data = self::$handler->read($sessionId);
        return $data ? unserialize($data) : null;
    }
    public static function setSession($sessionId, $data) {
        self::assertInit();
        self::$handler->write($sessionId, serialize($data));
        return true;
    }
    public static function deleteSession($sessionId,$key=null) {
        self::assertInit();
        if($key) {
            $data = self::getSession($sessionId);
            unset($data[$key]);
            self::setSession($sessionId,$data);
            return true;
        } else {
            self::$handler->destroy($sessionId);
            return true;
        }
    }
    public static function destroySession($sessionId) {
        self::assertInit();
        self::$handler->destroy($sessionId);
        return true;
    }
    public static function gc() {
        if(!self::$handler) {
            throw new \Exception("Session Handler not initialized");
        }
        self::$handler->gc(ini_get('session.gc_maxlifetime'));
        return true;
    }
    public static function listSessions($key=null,$value=null) {
        self::assertInit();
        if($key == "ip") {
            return self::$handler->listByIp($value);
        } elseif($key == "metaId") {
            return self::$handler->listByMetaId($value);
        } elseif($key == "userAgent") {
            return self::$handler->listByUserAgent($value);
        } else {
            return self::$handler->listSessions();
        }
    }

    //Private functions
    private static function assertInit() {
        if(!self::$handler) {
            throw new \Exception("Session Handler not initialized");
        }
        if(!self::$hasInit) {
            self::$handler->init();
            self::$hasInit = true;
        }
    }
}