<?php
namespace boru\dhsession\handlers;

class AdodbHandler extends HandlerInterface {

    private $db;
    
    public function __construct($options=[]) {
        if(isset($options["db"])) {
            $this->db = $options["db"];
            unset($options["db"]);
        } else {
            throw new \Exception("db not set");
        }
        parent::__construct($options);
    }

    public function init() {
        $sql = "CREATE TABLE IF NOT EXISTS ".$this->getTableName()." (
            `sessionid` varchar(255) NOT NULL,
            `ip` varchar(100) NOT NULL,
            `useragent` varchar(255) NOT NULL,
            `metaid` varchar(100) NOT NULL,
            `data` longtext NOT NULL,
            `modified` int NOT NULL,
            PRIMARY KEY (`sessionid`),
            KEY `idx_ip` (`ip`),
            KEY `idx_metaid` (`metaid`),
            KEY `idx_agent` (`useragent`),
            KEY `idx_modified` (`modified`)
        )";
        $this->db->query($sql);
        return parent::init();
    }
    /**
     * @return bool
     */
    public function refresh() {
        $currentId = session_id();
        $newId = session_regenerate_id();
        $sql = "update ".$this->getTableName()." set sessionid = ?, modified = ? where sessionid = ?";
        $this->db->pquery($sql, [$newId, time(), $currentId]);
        $this->setSessionId($newId);
        return true;
    }
    /**
     * @return bool
     */
    public function close() { 
        return $this->gc();
    }

    /**
     * @return bool
     */
    public function destroy($id) {
        $sql = "delete from ".$this->getTableName()." where sessionid = ?";
        $this->db->pquery($sql, [$id]);
        return true;
    }

    /**
     * @return bool
     */
    public function gc($max_lifetime = null) {
        $sql = "delete from ".$this->getTableName()." where modified < ?";
        $this->db->pquery($sql, [$this->getExpireTime($max_lifetime)]);
        return true;
    }

    /**
     * @return bool
     */
    public function open($path, $name) {
        $sql = "delete from ".$this->getTableName()." where sessionid = ? AND modified < ?";
        $this->db->pquery($sql, [$this->getSessionId(), $this->getExpireTime()]);
        $this->setSessionName($name);
        return true;
    }

    /**
     * @return string|false
     */
    public function read($id) {
        $sql = "select data from ".$this->getTableName()." where sessionid = ?";
        $result = $this->db->pquery($sql, [$id]);
        if ($result && $row = $this->db->fetchAssoc($result)) {
            return $row["data"];
        }
        if(empty($this->getSessionId())) {
            $this->refresh();
        }
        return "";
    }

    /**
     * @return bool
     */
    public function write($id, $data) {
        $sql = "insert into ".$this->getTableName()." (sessionid, ip, `useragent`, `metaid`, data, modified) values (?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE data = ?, modified = ?, metaid=?";
        $this->db->pquery($sql, [$id, $_SERVER["REMOTE_ADDR"], $_SERVER['HTTP_USER_AGENT'], $this->getMetaId(), $data, time(), $data, time(), $this->getMetaId()]);
        return true;
    }
    /**
     * @return false|array
     */
    public function readRaw($id) {
        $sql = "select data from ".$this->getTableName()." where sessionid = ?";
        $result = $this->db->pquery($sql, [$id]);
        if ($result && $row = $this->db->fetchAssoc($result)) {
            return $row;
        }
        return false;
    }

    public function setMetaId($metaId) {
        $sql = "update ".$this->getTableName()." set metaid = ? where sessionid = ?";
        $this->db->pquery($sql, [$metaId, $this->getSessionId()]);
        parent::setMetaId($metaId);
    }

    public function listSessions() {
        $sql = "select * from ".$this->getTableName();
        $result = $this->db->query($sql);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }
    public function listByIp($ip) {
        $sql = "select * from ".$this->getTableName()." where ip = ?";
        $result = $this->db->pquery($sql, [$ip]);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }
    public function listByUserAgent($userAgent) {
        $sql = "select * from ".$this->getTableName()." where useragent = ?";
        $result = $this->db->pquery($sql, [$userAgent]);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }
    public function listByMetaId($metId) {
        $sql = "select * from ".$this->getTableName()." where metaid = ?";
        $result = $this->db->pquery($sql, [$metId]);
        $sessions = [];
        while($row = $this->db->fetchAssoc($result)) {
            $sessions[] = $row;
        }
        return $sessions;
    }
}