<?php
namespace boru\dhsession\handlers;

class DhdbHandler extends HandlerInterface {

    /**
     * @var \boru\dhdb\dhDB $db
     */
    private $db;
    
    public function __construct($options=[]) {
        if(isset($options["db"])) {
            $this->db = $options["db"];
            unset($options["db"]);
        } else {
            throw new \Exception("db not set");
        }
        parent::__construct($options);
    }

    public function init() {
        $sql = "CREATE TABLE IF NOT EXISTS ".$this->getTableName()." (
            `sessionid` varchar(255) NOT NULL,
            `ip` varchar(100) NOT NULL,
            `useragent` varchar(255) NOT NULL,
            `metaid` varchar(100) NOT NULL,
            `data` longtext NOT NULL,
            `modified` int NOT NULL,
            PRIMARY KEY (`sessionid`),
            KEY `idx_ip` (`ip`),
            KEY `idx_metaid` (`metaid`),
            KEY `idx_agent` (`useragent`),
            KEY `idx_modified` (`modified`)
        )";
        $this->db->query($sql);
        return parent::init();
    }
    /**
     * @return bool
     */
    public function refresh() {
        $currentId = session_id();
        session_regenerate_id();
        $newId = session_id();
        $sql = "update ".$this->getTableName()." set sessionid = ?, modified = ? where sessionid = ?";
        $this->db->query($sql, [$newId, time(), $currentId]);
        $this->setSessionId($newId);
        return true;
    }
    /**
     * @return bool
     */
    public function close() { 
        return $this->gc();
    }

    /**
     * @return bool
     */
    public function destroy($id) {
        $sql = "delete from ".$this->getTableName()." where sessionid = ?";
        $this->db->query($sql, [$id]);
        return true;
    }

    /**
     * @return bool
     */
    public function gc($max_lifetime = null) {
        $sql = "delete from ".$this->getTableName()." where modified < ?";
        $this->db->query($sql, [$this->getExpireTime($max_lifetime)]);
        return true;
    }

    /**
     * @return bool
     */
    public function open($path, $name) {
        $this->setSessionName($name);
        return true;
    }

    /**
     * @return string|false
     */
    public function read($id) {
        $sql = "select data from ".$this->getTableName()." where sessionid = ? AND modified > ?";
        $result = $this->db->query($sql, [$id, $this->getExpireTime()]);
        if ($row = $this->db->next($result)) {
            return $row->get("data");
        }
        if(empty($this->getSessionId())) {
            //$this->refresh();
            session_regenerate_id();
            $this->setSessionId(session_id());
        }
        return "";
    }

    /**
     * @return bool
     */
    public function write($id, $data) {
        $sql = "insert into ".$this->getTableName()." (sessionid, ip, `useragent`, `metaid`, data, modified) values (?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE data = ?, modified = ?, `metaid`=?";
        $this->db->query($sql, [$id, $_SERVER["REMOTE_ADDR"], $_SERVER["HTTP_USER_AGENT"], $this->getMetaId(), $data, time(), $data, time(), $this->getMetaId()]);
        return true;
    }

    /**
     * @return false|array
     */
    public function readRaw($id) {
        $sql = "select data from ".$this->getTableName()." where sessionid = ?";
        $result = $this->db->query($sql, [$id]);
        if ($row = $this->db->next($result)) {
            return $row->asArray();
        }
        return false;
    }

    public function setMetaId($metaId) {
        $sql = "update ".$this->getTableName()." set metaid = ? where sessionid = ?";
        $this->db->query($sql, [$metaId, $this->getSessionId()]);
        parent::setMetaId($metaId);
    }

    public function listSessions() {
        $sql = "select * from ".$this->getTableName();
        $result = $this->db->query($sql);
        $sessions = [];
        while($row = $this->db->next($result)) {
            $sessions[] = $row->asArray();
        }
        return $sessions;
    }
    public function listByIp($ip) {
        $sql = "select * from ".$this->getTableName()." where ip = ?";
        $result = $this->db->query($sql, [$ip]);
        $sessions = [];
        while($row = $this->db->next($result)) {
            $sessions[] = $row->asArray();
        }
        return $sessions;
    }
    public function listByUserAgent($userAgent) {
        $sql = "select * from ".$this->getTableName()." where useragent = ?";
        $result = $this->db->query($sql, [$userAgent]);
        $sessions = [];
        while($row = $this->db->next($result)) {
            $sessions[] = $row->asArray();
        }
        return $sessions;
    }
    public function listByMetaId($metId) {
        $sql = "select * from ".$this->getTableName()." where metaid = ?";
        $result = $this->db->query($sql, [$metId]);
        $sessions = [];
        while($row = $this->db->next($result)) {
            $sessions[] = $row->asArray();
        }
        return $sessions;
    }
}