<?php
namespace boru\dhsession\handlers;

use boru\dhsession\lib\SessionHandler;

abstract class HandlerInterface extends SessionHandler {

    /**
     * Session Lifetime, default 2 hrs
     *
     * @var int $expiry The expiry time in seconds
     */
    private $expiry = 7200;
    private $sessionId;
    private $sessionName = "PHPSESSID";
    private $tableName = "dh_sessions";
    private $callbackOnStart = null;
    private $callbackOnClose = null;
    private $callbackOnRead = null;

    private $metaId = "";
    private $isStarted = false;

    public function __construct($options=[]) {
        $this->setOptions($options);
        session_set_save_handler($this, true);
    }

    public function setOptions($options=[]) {
        if (isset($options["expiry"])) {
            $this->setExpiry($options["expiry"]);
        }
        if (isset($options["sessionName"])) {
            $this->setSessionName($options["sessionName"]);
        }
        if (isset($options["tableName"])) {
            $this->setTableName($options["tableName"]);
        }
        if (isset($options["startCallback"])) {
            $this->setStartCallback($options["startCallback"]);
        }
        if (isset($options["closeCallback"])) {
            $this->setCloseCallback($options["closeCallback"]);
        }
        if (isset($options["readCallback"])) {
            $this->setReadCallback($options["readCallback"]);
        }
    }

    public function init() {
        ini_set("session.gc_maxlifetime", $this->expiry);
        ini_set("session.gc_probability", "0");
    }
    public function start() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
            $this->setSessionId(session_id());
            $this->setSessionName(session_name());
            $this->isStarted = true;
            return true;
        }
        if(!$this->isStarted) {
            $this->startCallback();
        }
        $this->setSessionId(session_id());
        $this->setSessionName(session_name());
        $this->isStarted = true;
        return false;
    }

    public function startCallback() {
        if ($this->callbackOnStart) {
            $callback = $this->callbackOnStart;
            $callback($this);
        }
    }
    public function closeCallback() {
        if ($this->callbackOnClose) {
            $callback = $this->callbackOnClose;
            $callback($this);
        }
    }
    public function readCallback($data) {
        if ($this->callbackOnRead) {
            $callback = $this->callbackOnRead;
            $callback($data);
        }
    }

    /**
     * @return bool
     */
    //abstract public function close();

    /**
     * @return bool
     */
    //abstract public function destroy($id);

    /**
     * @return bool
     */
    //abstract public function gc($max_lifetime);

    /**
     * @return bool
     */
    //abstract public function open($path, $name);

    /**
     * @return string|false
     */
    //abstract public function read($id);

    /**
     * @return bool
     */
    //abstract public function write($id, $data);

    /**
     * @return bool
     */
    //abstract public function refresh();

    /** 
     * @return false|array
     */
    abstract public function readRaw($id);


    abstract public function listSessions();
    abstract public function listByIp($ip);
    abstract public function listByUserAgent($userAgent);
    abstract public function listByMetaId($metaId);

    public function setExpiry($expiry) {
        $this->expiry = $expiry;
    }
    public function getExpiry() {
        return $this->expiry;
    }
    public function setSessionId($sessionId) {
        $this->sessionId = $sessionId;
        session_id($sessionId);
    }
    public function getSessionId() {
        return $this->sessionId;
    }
    public function setSessionName($sessionName) {
        $this->sessionName = $sessionName;
    }
    public function getSessionName() {
        return $this->sessionName;
    }
    public function setTableName($tableName) {
        $this->tableName = $tableName;
    }
    public function getTableName() {
        return $this->tableName;
    }
    public function setMetaId($metaId) {
        $this->metaId = $metaId;
    }
    public function getMetaId() {
        return $this->metaId;
    }
    public function getExpireTime($max_lifetime = null) {
        return time() - ($max_lifetime ? $max_lifetime : $this->getExpiry());
    }
    public function setStartCallback($callback) {
        $this->callbackOnStart = $callback;
    }
    public function setCloseCallback($callback) {
        $this->callbackOnClose = $callback;
    }
    public function setReadCallback($callback) {
        $this->callbackOnRead = $callback;
    }
}